package org.lsst.ccs.subsystem.common.devices.dataforth;

import java.util.logging.Logger;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.commons.annotations.LookupPath;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.dataforth.Maq20;
import org.lsst.ccs.drivers.dataforth.Maq20DiscretePulse;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.subsystem.common.ErrorUtils;

/**
 *  Handles a Dataforth Maq20 system pulse control channel.
 *
 *  @author Owen Saxton
 */
public class Maq20PulseControl implements HasLifecycle, Maq20Device.Maq20Control {

    /**
     *  Data fields.
     */
    @LookupName
    protected String name;
    @LookupPath
    protected String path;
    @LookupField(strategy=LookupField.Strategy.ANCESTORS)
    protected Maq20Device maqDevc;

    protected Integer modIndex;   // The module index
    protected Integer funcIndex;  // The function index
    protected int pulsesPerRevn = 1;

    private static final Logger LOG = Logger.getLogger(Maq20PulseControl.class.getName());
    private Maq20Device.ModuleData modData;
    private Maq20DiscretePulse pulse;


    /**
     *  Life-cycle initialization.
     * 
     *  Checks the configuration
     */
    @Override
    public void init()
    {
        if (modIndex == null) {
            ErrorUtils.reportConfigError(LOG, path, "modIndex", "not specified");
        }
        modData = maqDevc.getModuleData(modIndex);
        if (modData == null) {
            ErrorUtils.reportConfigError(LOG, path, "modIndex (" + modIndex + ")", "specifies non-existent module");
        }
        if (modData.modDef.opType != Maq20.OPER_DISCRETE) {
            ErrorUtils.reportConfigError(LOG, path, "modIndex (" + modIndex + ")", "specifies non-discrete module");
        }
        if (funcIndex == null) {
            ErrorUtils.reportConfigError(LOG, path, "funcIndex", "not specified");
        }
        if (funcIndex != 0 && funcIndex != 1) {
            ErrorUtils.reportConfigError(LOG, path, "funcIndex (" + funcIndex + ")", "is not 0 or 1");
        }
        if (modData.discFunc[funcIndex] != Maq20Device.DISC_FUNC_PULSE) {
            ErrorUtils.reportConfigError(LOG, path, "funcIndex (" + funcIndex + ")", "specifies non-pulse function");
        }
    }


    /**
     *  Post-connection initialization.
     *
     *  @throws DriverException
     */
    @Override
    public void initialize() throws DriverException
    {
        pulse = modData.maqDiscPulse;
        pulse.enable(funcIndex);
        setPulsesPerRevn(pulsesPerRevn);
    }


    /**
     *  Pre-closure cleanup.
     */
    @Override
    public void close()
    {
        pulse = null;
    }


    /**
     *  Sets the pulses/revolution.
     *
     *  @param  pulses  The number of pulses per revolution
     *  @throws DriverException
     */
    public void setPulsesPerRevn(int pulses) throws DriverException
    {
        pulsesPerRevn = pulses;
        if (pulse != null) {
            pulse.setPulsesPerRevn(funcIndex, pulsesPerRevn);
            pulse.armFunction(funcIndex);
        }
    }


    /**
     *  Gets the pulses/revolution.
     *
     *  @return  The number of pulses/revn
     */
    public int getPulsesPerRevn()
    {
        return pulsesPerRevn;
    }


    /**
     *  Gets the name.
     * 
     *  @return  The name
     */
    public String getName()
    {
        return name;
    }


    /**
     *  Gets the path.
     * 
     *  @return  The path
     */
    public String getPath()
    {
        return path;
    }

}
