package org.lsst.ccs.subsystem.common;

import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.monitor.DerivedChannel;

/**
 *  A DerivedChannel to compute a heat transfer rate from three
 *  configurable inpu Channels:  cooling fluid flow rate and inlet 
 *  outlet temperatures.
 *
 *  The formula is constant * flow * Delts-T, where the constant
 *  factor is settable, and should be a product of density, specific
 *  heat, and any units-adjusting factors.
 *
 *  @author eisner
 */
public class HeatTransferChannel extends DerivedChannel {

    @LookupField(strategy = LookupField.Strategy.TREE)
    Map<String, Channel> channelMap = new HashMap<>();

    @LookupName
    private String name;

    private static final Logger LOG = Logger.getLogger(HeatTransferChannel.class.getName());

    private double factor = 1.0;
    private Channel flowChannel;
    private Channel inletTChannel;
    private Channel outletTChannel;

    @ConfigurationParameter(name = "flowChanPath", category = "General",
                            isFinal = true, units = "unitless",
                            description = "path to flow-rate Channel")
    protected volatile String flowChanPath;

    @ConfigurationParameter(name = "inletTChanPath", category = "General",
                            isFinal = true, units = "unitless",
                            description = "Path to inlet temperature Channel")
    protected volatile String inletTChanPath;

    @ConfigurationParameter(name = "outletTChanPath", category = "General",
                            isFinal = true, units = "unitless",
                            description = "Path to outlet temperature Channel")
    protected volatile String outletTChanPath;

   /**
    *  Checks validity of a proposed set of ConfigurationParameters
    *
    *  Check Reqested input Channels must exist,
    *
    *  @param Map of ConfigurationParameters from name to value
    */
    @Override
    public void validateBulkChange(Map<String,Object> params) {
        super.validateBulkChange(params);
        String chpath = (String) params.get("flowChanPath");
        if (channelMap.get(chpath) == null) {
            throw new RuntimeException(name + ": flowChanPath " + chpath + " not found");
        }       
        chpath = (String) params.get("inletTChanPath");
        if (channelMap.get(chpath) == null) {
            throw new RuntimeException(name + ": inletTChanPath " + chpath + " not found");
        }       
        chpath = (String) params.get("outletTChanPath");
        if (channelMap.get(chpath) == null) {
            throw new RuntimeException(name + ": outletTChanPath " + chpath + " not found");
        }       
    }

   /**
    *  Set configuration parameters to new values.
    *  
    *  Updates Channels to be used in evaluation.
    *
    *  @param  Map of parameters for which a change is requested,
    *          keyed by names
    */
    @Override
    public void setParameterBulk(Map<String,Object> params)  {
        super.setParameterBulk(params);
        Set<String> keys = params.keySet();
        if (keys.contains("flowChanPath")) {
            flowChanPath = (String) params.get("flowChanPath");
            flowChannel = (Channel) channelMap.get(flowChanPath);
        }
        if (keys.contains("inletTChanPath")) {
            inletTChanPath = (String) params.get("inletTChanPath");
            inletTChannel = (Channel) channelMap.get(inletTChanPath);
        }
        if (keys.contains("outletTChanPath")) {
            outletTChanPath = (String) params.get("outletTChanPath");
            outletTChannel = (Channel) channelMap.get(outletTChanPath);
        }
    }

   /**
    *  Set constant factor in evaluation formula
    *
    *  @param  double factor
    */
    public void setFactor(double fac) {
        factor = fac;
        LOG.info(name + ": Constant factor set to " + factor);
    }

   /**
    *  Evaluate heat-transfer rate.
    *
    *  @return  double 
    */
    @Override
    public double evaluateDerivedValue() {
        double trans = (outletTChannel.getValue() - inletTChannel.getValue())
	    * flowChannel.getValue() * factor;
        return trans;
    }
}

