package org.lsst.ccs.subsystem.common.devices.turbopump;

import java.util.HashMap;
import java.util.Map;
import java.util.logging.Logger;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupPath;
import org.lsst.ccs.config.ConfigurationParameterDescription;
import org.lsst.ccs.config.HasConfigurationParameterDescription;
import org.lsst.ccs.drivers.twistorr.TwisTorr84;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.commons.DriverTimeoutException;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.services.AgentStateService;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystem.common.ErrorUtils;
import org.lsst.ccs.subsystem.common.alerts.TurboAlerts;
import org.lsst.ccs.subsystem.common.devices.vacuum.states.TurboState;

/* Pre-defined ommand levels */
import static org.lsst.ccs.command.annotations.Command.NORMAL;
import static org.lsst.ccs.command.annotations.Command.ENGINEERING_ROUTINE;
import static org.lsst.ccs.command.annotations.Command.ENGINEERING_ADVANCED;
import static org.lsst.ccs.command.annotations.Command.ENGINEERING_EXPERT;

/**
 *  Interfaces with driver for Agilent TwisTorr84 turbo pump.
 *
 *  The Channels read are TurboPump driver/hardware quantities.
 *
 *  The groovy Channel type  identifies whether a quantity to be read 
 *  is of type logical ("Bool"), integer ("Numeric") or alphanumeric 
 *  ("Alpha").  The subtype identifies the specific quantity tto be read.
 *
 *  @author Al Eisner
 */
public class TwisTorr84Device extends Device implements HasConfigurationParameterDescription {

   /**
    *  Data fields.
    */

    @LookupPath
    protected String path;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentStateService stateService;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AlertService alertService;

    private final TwisTorr84 turbo = new TwisTorr84();  // Associated driver
    private Integer chanKey;
    private volatile TurboState lastState;

    /* Enumerate supported TurboPump models (all with OnBoard controller) */

    public enum Model {
        model74,
        model84,
        model304,
        model305;
    }

    /* Configuration parameters related to hardware settings */

    @ConfigurationParameter(isFinal = true, units = "unitless",
                            description = "serial port")
    protected volatile String devcId;       // Device connection identification

    // The existence of several hardware settings depends on pumo model.
    // For those associated with ConfigurationParameters, the latter are
    // disabled when not available in the hardware.
    @ConfigurationParameter(isBuild = true, category = "build", 
                            units = "unitless",
                            description = "enumerated pump model")
    protected volatile Model model;     

    @ConfigurationParameter(isFinal = true, units = "unitless",
                            description = "true/false for low/normal speed")
    protected volatile Boolean lowSpeedMode;

    @ConfigurationParameter(isFinal = true, description = "Soft-Start mode <true|false>, settable only if pump is stopped", units = "unitless")
    protected volatile Boolean softStartMode; 

    @ConfigurationParameter(isFinal = true, description = "Active-Stop mode <true|false>, settable only if pump is stopped", units = "unitless")
    protected volatile Boolean activeStopMode;

    @ConfigurationParameter(isFinal = true, description = "Setpoint value for digital outout", units = "Hz")
    protected volatile Integer setpointValue;

    private static final Logger LOG = Logger.getLogger(TwisTorr84Device.class.getName());
    private static final String autotuneMsg = "AStress due to leak, for-pressure too high, gas flow too high, or turbo damage. Mitigate within 15 min to avoid FAIL state.";
    private static final long READALL_PAUSE = 100;  // milliseconds

    /* Hard-code hardware-initialization parameters  */

    private final boolean waterCooling = false;
    private final boolean interlockType = true;      // Continuous
    private final boolean ventvalveByCmnd = true;    // By-command mode
    private final boolean speedReadActivate = true;  // Read speed after Stop
    private final int setpointType = 0;              // Output is frequency
    private final boolean setpointLevel = false; 
    private final int ventDelay = 30;                // 6 seconds
    // This ventvalve type empiricslly gives best behavior 
    private final boolean ventvalveType = true;      // Model 305 only

    /* Hard-coded hardware-initialization parameters (depend on pump model)*/

    private final int rotfreq_set_84  = 1350;  // normal rotation frequncy (Hz) 
    private final int rotfreq_set_74  = 1167;
    private final int rotfreq_set_304 = 1010;
    private final int rotfreq_set_305 = 1010;
    private final int rotfreq_low_84  = 1100;  // slow rotation frequncy (Hz) 
    private final int rotfreq_low_74  = 1000;
    private final int rotfreq_low_304 =  900;
    private final int rotfreq_low_305 =  900;
    private int rotfreq_set, rotfreq_low;

    private final boolean gasType_84 = false;  // true/false for Argon/Nitrogen
    private final int gasType_305 = 1;         // Nitrogen

    /* Omit serialType, since driver will not allow setting this */
    // private final Boolean serialType = false;  // true/false for RS-485/RS-232

   /**
    *  Disable ConfigurationParameters which are associated with hardware
    *  commands not implemented for configured pumo model.
    *
    *  @param  String parameter name
    *  @return ConfigurationParameterDecription
    */
    @Override
    public ConfigurationParameterDescription getConfigurationParameterDescription(String parName) {
        ConfigurationParameterDescription cpd = null;
        if (parName.equals("activeStopMode") || parName.equals("waterCooling")) {
            boolean enableParam = (model != Model.model305);
            cpd = new ConfigurationParameterDescription();
             cpd.withIsEnabled(enableParam);
        }
        return cpd;
    }

   /**
    *  Performs basic initialization.  
    */
    @Override
    protected void initDevice() {

        stateService.registerState(TurboState.class, "Turbo-pump State", this);
        alertService.registerAlert(TurboAlerts.IN_AUTOTUNING.newAlert(path));

        chanKey = 0;

        if (devcId.equals("")) {
            ErrorUtils.reportConfigError(LOG, name, "devcId", "is missing");
        }

        /* Prepare some quantities for hardware settings */
        switch (model) {
            case model74:
                rotfreq_set = rotfreq_set_74;
                rotfreq_low = rotfreq_low_74;
                break;
            case model84:
                rotfreq_set = rotfreq_set_84;
                rotfreq_low = rotfreq_low_84;
                break;
            case model304:
                rotfreq_set = rotfreq_set_304;
                rotfreq_low = rotfreq_low_304;
                break;
            case model305:
                rotfreq_set = rotfreq_set_305;
                rotfreq_low = rotfreq_low_305;
                break;
            default:
                ErrorUtils.reportConfigError(LOG, name, "model ", "invalid");
                break;
        }
    }


   /**
    *  Performs full initialization.
    */
    @Override
    protected void initialize() {

        try {
            turbo.open(devcId);
            int status = turbo.readNumeric(TwisTorr84.CmndNumeric.STATUS);
            TurboState state = TurboState.getState(status);
            LOG.info("\n Connected to " + fullName + ", Turbo state = "
                     + state.toString() + ", starting hardware settings");
            if (state != TurboState.STOP) {
                LOG.warning(name + " not in STOP state, so a few writes will be ignored, but writeAndVerity accepts value read if it matches requested value");
            } 

            /* Hardware initialization */

            turbo.writeAndVerifyBool(TwisTorr84.CmndBool.LOW_SPEED, lowSpeedMode);
            turbo.writeAndVerifyBool(TwisTorr84.CmndBool.SOFT_START, softStartMode);
            turbo.writeAndVerifyBool(TwisTorr84.CmndBool.INTERLOCK_TYPE, interlockType);
            turbo.writeAndVerifyBool(TwisTorr84.CmndBool.VENTVALVE_BY_CMND, ventvalveByCmnd);
            turbo.writeAndVerifyBool(TwisTorr84.CmndBool.SPEED_READ_ACTIVATE, speedReadActivate);
            turbo.writeAndVerifyNumeric(TwisTorr84.CmndNumeric.SETPOINT_TYPE, setpointType);
            turbo.writeAndVerifyBool(TwisTorr84.CmndBool.SETPOINT_LEVEL, setpointLevel);
            turbo.writeAndVerifyNumeric(TwisTorr84.CmndNumeric.SETPOINT_VALUE, setpointValue);
            turbo.writeAndVerifyNumeric(TwisTorr84.CmndNumeric.VENT_DELAY, ventDelay);;

            /* These settimgs depend on pump model a */

            turbo.writeAndVerifyNumeric(TwisTorr84.CmndNumeric.ROTFREQ_SET, rotfreq_set);
            turbo.writeAndVerifyNumeric(TwisTorr84.CmndNumeric.ROTFREQ_LOW, rotfreq_low);
  
            if (model == Model.model305) {
                turbo.writeAndVerifyBool(TwisTorr84.CmndBool.VENTVALVE_TYPE, ventvalveType);
            } else { 

                turbo.writeAndVerifyBool(TwisTorr84.CmndBool.ACTIVE_STOP,activeStopMode);
                turbo.writeAndVerifyBool(TwisTorr84.CmndBool.WATER_COOLING,waterCooling);
            }
            if (model == Model.model305 || model ==  Model.model304) {
                turbo.writeAndVerifyNumeric(TwisTorr84.CmndNumeric.GAS_TYPE, gasType_305);
            } else {
                turbo.writeAndVerifyBool(TwisTorr84.CmndBool.GAS_TYPE_ARGON, gasType_84);
            }

            initSensors();
            setOnline(true);
            LOG.info(name + " Hardware initialization succeeded");
        }

        catch (DriverException e) {
            if (!inited) {
                LOG.severe("Error connecting to or initializaing " 
                          + fullName + ": " + e);
            }
            close();
         }
        inited = true;
    }


   /**
    *  Closes the connection.
    */
    @Override
    protected void close() {
        try {
            turbo.close();
        }
        catch (DriverException e) {
        }
    }

    /**
     * Pump commands and settings
     */

   /**
    *  Start pump
    *
    *  @throws  DriverException
    */
    @Command(type=Command.CommandType.ACTION, name="startTurboPump", 
             description="Start turbo pump")
    public void startTurboPump() throws DriverException
    {
        turbo.writeBool(TwisTorr84.CmndBool.START_STOP, true);
    }

   /**
    *  Stop pump
    *
    *  @throws  DriverException
    */
    @Command(type=Command.CommandType.ACTION, name="stopTurboPump", 
             description="Stop turbo pump")
    public void stopTurboPump() throws DriverException 
    {
        turbo.writeBool(TwisTorr84.CmndBool.START_STOP, false);
    }

   /**
    *  Open vent valve
    *
    *  @throws  DriverException
    */
    @Command(type=Command.CommandType.ACTION, name="openVentValve",
             description="If model305, open vent valve")
    public void openVentValve() throws DriverException 
    {
        if (model == Model.model305) {
            boolean type = turbo.readBool(TwisTorr84.CmndBool.VENTVALVE_TYPE);
            turbo.writeBool(TwisTorr84.CmndBool.VENTVALVE_OPEN, type ? true : false);
        }
    }

   /**
    *  Close vent valve
    *
    *  @throws  DriverException
    */
    @Command(type=Command.CommandType.ACTION, name="closeVentValve",
             description="If model305, close vent valve")
    public void closeVentValve() throws DriverException 
    {
        if (model == Model.model305) {
            boolean type = turbo.readBool(TwisTorr84.CmndBool.VENTVALVE_TYPE);
            turbo.writeBool(TwisTorr84.CmndBool.VENTVALVE_OPEN, type ? false : true);
        }
    }

   /**
    *  Show vent valve settings and status
    *
    *  @return  String
    *  @throws  DriverException
    */
    @Command(type=Command.CommandType.QUERY, name="showVentValveStatus",
             description="If model305, show vent valve settings and status")
    public String showVentValveStatus() throws DriverException 
    {
        String text = "";
        if (model == Model.model305) {
            Boolean type = turbo.readBool(TwisTorr84.CmndBool.VENTVALVE_TYPE);
            Boolean open = turbo.readBool(TwisTorr84.CmndBool.VENTVALVE_OPEN);
            Boolean status = (type ? open : !open);
            String stext = (status ? "Open" : "Closed");
            text = "VENTVALVE_TYPE (140) = " + type + ",  VENTVALVE_OPEN (122) = " + open + ",  Valve is " + stext;
        } else {
            text = "Implemented only for model 305-FS";
        }
        return text;
    }

   /**
    *  Checks a channel's parameters for validity.
    *  The numeric type returned is just an ordinal Channel index;
    *  the numeric subtyoe is not used.
    *
    *  @param   Channel ch    The channel
    *  @return  A two-element array containing the encoded type [0] and
    *           subtype [1] values.
    *  @throws  Exception if any errors found in the parameters.
    */
    @Override
    protected int[] checkChannel(Channel ch) throws Exception
    {
        String type = ch.getTypeStr();
        String subtype = ch.getSubTypeStr();
        String name = ch.getName();
        try {
            switch (type) {
                case "Bool":
                    TwisTorr84.CmndBool.valueOf(subtype);
                    break;
                case "Numeric":
                    TwisTorr84.CmndNumeric.valueOf(subtype);
                    break;
                case "Alpha":
                    TwisTorr84.CmndAlpha.valueOf(subtype);
                    break;
                default:
                    ErrorUtils.reportChannelError(LOG, name, "type", type);
            }
        }
        catch (IllegalArgumentException e) {
            ErrorUtils.reportChannelError(LOG, name, type + " subtype ",
					subtype);
        }
        chanKey++;
        return new int[]{chanKey, 0};
    }


   /**
    *  Reads a channel.
    *
    *  @param  Channel ch
    *  @return  The read value or NaN
    */
    @Override
    protected double readChannel(Channel ch)
    {
        double value = super.readChannel(ch);     //NaN
        String itemType = ch.getTypeStr();
        String item = ch.getSubTypeStr();
	// Note:  the framework assures that null "get" returns do not occur

        try {
	    switch (itemType) {
                case "Bool":
                    if (turbo.readBool(TwisTorr84.CmndBool.valueOf(item))) {
                        value = 1.0;
			    } else {
                        value = 0.0;
                    }
                    break;
                case "Numeric":
                    value = turbo.readNumeric(TwisTorr84.CmndNumeric.valueOf(item));
                    if (item.equals("STATUS")) {
                        int status = (int) Math.round(value);
                        TurboState state = TurboState.getState(status);
                        if (state != lastState) {
                            stateService.updateAgentComponentState(this, state);
                            if (state == TurboState.AUTOTUNING) {
                                alertService.raiseAlert(TurboAlerts.IN_AUTOTUNING.newAlert(path), AlertState.WARNING, autotuneMsg);
                            } else if (lastState == TurboState.AUTOTUNING) {
                                alertService.raiseAlert(TurboAlerts.IN_AUTOTUNING.newAlert(path), AlertState.NOMINAL, "out of autotuning state");
                            }
                            if (state == TurboState.FAIL) {
                                try {
                                    String cause = getTurboFailCode();
                                    LOG.info(name + " TURBO FAILURE, cause(s) = " + cause);
				}
                                catch (Exception ex) {
                                    LOG.severe(name + " failed to read error code, " + ex);
                                }
                            }
                            lastState = state;
                        }
                    }
                    break;
                case "Alpha":
                    turbo.readAlpha(TwisTorr84.CmndAlpha.valueOf(item));
                    // Not implemented, relevant hardware not installed
                    break;
	    }
        }        
        catch (DriverTimeoutException et) {
            LOG.severe(name + ": TurboPump timeout reading data " + item + ": " + et);
            setOnline(false);
        }
	catch (DriverException e) {
            LOG.severe(name + ": TurboPump exception reading data " + item + ": " + e);
        }
        
        return value;
    }

   /**
    *  Get and decode error word (bits show reason for pump failure)
    *
    *  @return String
    *  @throws  DriverException
    */
    private String getTurboFailCode()  throws DriverException
    {
        int errcode = turbo.readNumeric(TwisTorr84.CmndNumeric.ERRCODE);
        return Integer.toString(errcode,16) + ", "+ TwisTorr84.decodeError(errcode);
    }

    /* Read-commands for command-line usage */

   /** 
    *  Read and decode pump status
    *
    *  @return  String indicating pump state or failure
    *  @throws  DriverException
    */
    @Command(type=Command.CommandType.QUERY, level=NORMAL, name="readTurboStatus", 
             description="Get pump status, including operating modes and failure")
    public TwisTorr84.PumpStatus readTurboStatus()  throws DriverException
    {
        int status = turbo.readNumeric(TwisTorr84.CmndNumeric.STATUS);
        return TwisTorr84.PumpStatus.decodeStatus(status);
    }

   /** 
    *  Read error code (bits show reason for pump failure)
    *
    *  @return  String giving error code in hexadecimal and decoding it
    *  @throws  DriverException
    */
    @Command(type=Command.CommandType.QUERY, name="readTurboFailCode", 
             description="Find reason(s) for failure", level=NORMAL)
    public String readTurboFailCode()  throws DriverException
    {
        return getTurboFailCode();
    }

   /** 
    *  Read TurboPump running-time in hours
    *
    *  @return  String giving active time
    *  @throws  DriverException 
    */
    @Command(type=Command.CommandType.QUERY,level=ENGINEERING_ROUTINE,
             name="readTurboLife",
             description="show TurboPump running-time in hours")
    public String readTurboLife() throws DriverException
    {
        int life = turbo.readNumeric(TwisTorr84.CmndNumeric.PUMP_HOURS);
        return Integer.toString(life) + " hours";
    }

   /**
    * Read all settings and data from pump controller.
    *
    * Loops over all read commands and returns them in a table format.
    * All DriverExceptions are caught; if one occurs, the data field
    * is replaced by the text (String) associated with the exception.
    *  
    * @throws InterruptedException
    * @return  String reporting all data read and exceptions.
    */
    @Command(type=Command.CommandType.QUERY,level=ENGINEERING_ROUTINE,
             name="readAll", 
             description="Read all TurboPump controller settings and data")
    public String readAll() throws InterruptedException
    {
        String table = "Read all turbo-pump settings and data\n" +"\n";

        /* Boolean commands */

        TwisTorr84.CmndBool cmndB[] = TwisTorr84.CmndBool.values();
        int nB = cmndB.length;
        for (int i = 0; i < nB; i++) {
            Thread.sleep(READALL_PAUSE);
            table += String.format("\n   %-22s", cmndB[i]);
            try {
                boolean respB = turbo.readBool(cmndB[i]);
                table += Boolean.toString(respB);
            } catch (DriverException ex) {
                table += ex.getMessage();
            }
        }
        table += "\n";

        /* Numeric commands */

        TwisTorr84.CmndNumeric cmndN[] = TwisTorr84.CmndNumeric.values();
        int nN = cmndN.length;
        for (int i = 0; i < nN; i++) {
            table += String.format("\n   %-22s", cmndN[i]);
            try {
                int respN = turbo.readNumeric(cmndN[i]);
                Thread.sleep(READALL_PAUSE);
                table += Integer.toString(respN);
            } catch (DriverException ex) {
                table += ex.getMessage();
            }
        }
        table += "\n";

        TwisTorr84.CmndAlpha cmndA[] = TwisTorr84.CmndAlpha.values();
        int nA = cmndA.length;
        for (int i = 0; i < nA; i++) {
            table += String.format("\n   %-22s", cmndA[i]);
            try {
                String respA = turbo.readAlpha(cmndA[i]);
                table += respA;
            } catch (DriverException ex) {
                table += ex.getMessage();
            }
        }

        return table;
    }

   /**
    *  Get the current TurboState
    *
    *  @return TurboState
    */
    public TurboState getTurboState() {
        return lastState;
    }
}
