package org.lsst.ccs.subsystem.common.service;

import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.logging.Level;
import org.lsst.ccs.PersistencyService;
import org.lsst.ccs.ServiceLifecycle;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.Persist;
import org.lsst.ccs.framework.ClearAlertHandler;
import org.lsst.ccs.localdb.statusdb.server.TrendingData;
import org.lsst.ccs.localdb.trendserver.TrendingClientService;
import org.lsst.ccs.localdb.utils.TrendingConnectionUtils;
import org.lsst.ccs.services.AgentService;
import org.lsst.ccs.services.alert.AlertService;

/**
 * An AgentService to be used by Agents that need to accumulate historical data
 * by retrieving the latest value from the local database.
 * <p>
 * This service is loaded and started only upon request by invoking the class
 * static method {@code DataAccumulationService.isNeeded()} in the constructor
 * of a class node of the Agent's component tree. This method must be invoked
 * before the HasLifecycle::build phase; the method can be invoked multiple times.
 * 
 * Once the service is successfully loaded it will be available via the Lookup annotations:
 * 
 * <pre>
 * 
 *      {@code @LookupField(strategy=LookupField.Strategy.TREE)}
 *      {@code private DataAccumulationService dataAccumulationService; }
 * </pre>
 * 
 * The next step requires that all the historical data channels are registered with
 * this service. Data registration must happen before the completion of the
 * HasLifecycle::init phase and is done by invoking method
 * <pre>
 * 
 *      {@code dataAccumulationService.registerAccumulatedDataPath(String path);}
 * </pre>
 * <p>
 * The path provided during data registration is the same path with which the data
 * is published. 
 * <p>
 * Incremental data accumulation for a given path is requested by invoking method:
 * 
 * <pre>
 * 
 *       double accumulatedValue = dataAccumulationService.accumulateData(path, increment);
 * </pre>
 * where variable <b>increment</b> is the value to be added to the historical data.
 * Only finite values will be used as valid increments. 
 * The method returns the overall accumulated value only if the provided increment is
 * finite and the service was able to retrieve the historical data from the database.
 * The method will return Double.NaN if it could not fetch the historical data value
 * from the database or the increment itself in the case in which the increment value
 * is not finite. 
 * <p>
 * This logic is meant to prevent the publication of valid values
 * when the service was unable to retrieve the historical data value from the database;
 * a valid publication would, in this case, wipe out the last historical data value
 * and reset the history.
 * <p>
 * The calling code is expected to publish the returned <b>accumulatedValue</b> with
 * the same registration path with which the increment was provided for accumulation.
 * 
 * <p>
 * Once the service is started, it will try to establish a connection with the
 * rest server on the buses to retrieve the historical baseline for all registered paths.
 * Once all baselines have been retrieved successfully for all registered paths
 * the service will no longer contact the rest server. On the other hand, if there is
 * a failure to connect to the rest service, or to retrieve the data for any of the
 * registered paths, the service will periodically attempt to fully initialize all
 * the needed historical baselines.
 * For any registered path without a valid database historical baseline, the incremental
 * accumulation will be done in memory and the cumulative value is stored to file
 * via the Persistence mechanism every time a valid incremental value is provided
 * to the service. Upon successfully retrieving a valid historical baseline from the
 * database, any locally persisted cumulative value will be added to the historical baseline
 * from the database.
 * <p>
 * The service will raise two alerts at WARNING level:
 * <ul>
 * <li><b>UnavailableRestServer</b> when the service fails to connect to the rest server on the buses</li>
 * <li><b>UnavailableDataFromRestServer</b> when the historical database baseline cannot be retrieved
 * for some of the registered paths. This alert will contain the list of channels that failed
 * initialization</li>
 * </ul>
 * <p>
 * These alerts will be reset to NOMINAL when the service is back to proper
 * operating conditions.
 * 
 * 
 * 
 * 
 * @author The LSST CCS Team
 * 
 */
public class DataAccumulationService implements AgentService, ServiceLifecycle {

    private static final java.util.logging.Logger LOG = java.util.logging.Logger.getLogger(DataAccumulationService.class.getCanonicalName());
    
    private static boolean isNeeded = false;
    
    @LookupField(strategy = LookupField.Strategy.TREE)
    AlertService alertService;
    
    @LookupField(strategy = LookupField.Strategy.TREE)
    PersistencyService persistencyService;
    
    @LookupField(strategy = LookupField.Strategy.TREE)
    private TrendingClientService trendingService;
    
    @LookupField(strategy = LookupField.Strategy.TOP)
    private Subsystem subsys;

    //We use the persistency service to store accumulated data if/when the database connection
    //cannot be established. The persisted values will be used once the database
    //connection is available.
    @Persist
    public volatile Map<String,Double> persistedBaselinesMap = new ConcurrentHashMap<>();

    //Frequency at which we check the database values (if needed)
    private final Integer CHECK_INTERVAL = 10000;
    private volatile long lastCheckTime = 0;

    
    //This map contains the values fetched from the database
    public volatile Map<String,Double> databaseBaselinesMap = new ConcurrentHashMap<>();
    
    private final Object accumulationDataLock = new Object();

    public final Set<String> accumulatedDataSet = new HashSet<>();
    public final Set<String> missingBaselinesSet = new HashSet<>();
    
    private final Alert cannotConnectToRestServerAlert = new Alert("UnavailableRestServer","Alert raised when the connection to the rest-server could not be achieved.");
    private final Alert problemFetchingBaselines = new Alert("UnavailableDataFromRestServer","Alert raised when data cannot be retrieved from the rest server.");
    
    private volatile boolean connectedToRestServer = false;
    private boolean hasBeenInitialized = false;
    
    @Override
    public String getAgentServiceName() {
        return "dataAccumulationService";
    }

    @Override
    public boolean startForAgent(AgentInfo agentInfo) {
        return isNeeded;
    }
    
    /**
     * Invoke this method from a class node constructor to request the use of this service.
     * Once this method is invoked, the service will be loaded and started during the 
     * HasLifecycle::build step.
     * 
     */
    public static void isNeeded() {
        System.setProperty("org.lsst.ccs.agent." + TrendingClientService.USES_TRENDING_SERVICE, "true");
        isNeeded = true;
    }
    
    @Override 
    public void preStart() {
        hasBeenInitialized = true;
        persistencyService.load();
        
        Map<String, Double> baselines = new ConcurrentHashMap<>();
        
        //We need to clean the persisted map in case paths have been
        //removed from the registration or add new values (0.) if new
        //paths are added.
        for (String path : accumulatedDataSet) {
            Double persistedValue = persistedBaselinesMap.remove(path);
            baselines.put(path, persistedValue == null ? 0. : persistedValue);
        }
        persistedBaselinesMap.clear();
        persistedBaselinesMap.putAll(baselines);                
    }

    @Override
    public void preInit() {
        alertService.registerAlert(cannotConnectToRestServerAlert, new ClearAlertHandler() {
            @Override
            public ClearAlertHandler.ClearAlertCode canClearAlert(Alert alert, AlertState alertState) {
                if ( connectedToRestServer ) {
                    return ClearAlertHandler.ClearAlertCode.CLEAR_ALERT;
                }
                return ClearAlertHandler.ClearAlertCode.DONT_CLEAR_ALERT;
            }            
        });
        alertService.registerAlert(problemFetchingBaselines, new ClearAlertHandler() {
            @Override
            public ClearAlertHandler.ClearAlertCode canClearAlert(Alert alert, AlertState alertState) {
                if ( missingBaselinesSet.isEmpty() ) {
                    return ClearAlertHandler.ClearAlertCode.CLEAR_ALERT;
                }
                return ClearAlertHandler.ClearAlertCode.DONT_CLEAR_ALERT;
            }            
        });
        
        persistencyService.setAutomatic(false, true);
    }
    
    /**
     * Register the path for historical data that will be accumulated with this service.
     * The path is equivalent to the path with which the data is published on the buses.
     * This method must be invoked before or during the HasLifecycle::init phase.
     * 
     * @param path The historical data path.
     */
    public void registerAccumulatedDataPath(String path) {
        if ( hasBeenInitialized ) {
            throw new RuntimeException("Too late to register accumulated data paths. It has to be done before the HasLifecycle::start phase.");
        }
        if ( !accumulatedDataSet.add(path) ) {
            throw new RuntimeException("Path "+path+" was already added to the data accumulation service.");
        }
    }

    private void initializeAccumulatedBaselinesFromDatabase() {

        //If we have collected all the baselines, then we are done.
        if ( databaseBaselinesMap.size() == accumulatedDataSet.size() ) {
            return;
        }

        TrendingConnectionUtils trendingUtils = trendingService.getBusTrendingConnection();
        connectedToRestServer = trendingUtils != null;
        if (trendingUtils == null) {
            alertService.raiseAlert(cannotConnectToRestServerAlert, AlertState.WARNING, "Failed to connect to the database.", AlertService.RaiseAlertStrategy.ON_SEVERITY_CHANGE);
            return;
        }
        alertService.raiseAlert(cannotConnectToRestServerAlert, AlertState.NOMINAL, "Connected to the database", AlertService.RaiseAlertStrategy.ON_SEVERITY_CHANGE);

        boolean updatePersistedData = false;
        synchronized(accumulationDataLock) {
            StringBuilder sb = new StringBuilder();
            for (String path : accumulatedDataSet) {
                if (databaseBaselinesMap.containsKey(path)) {
                    continue;
                }
                String fullName = subsys.getName() + "/" + path;
                TrendingData data = trendingUtils.getLatestData(fullName);
                double databaseBaseline = Double.NaN;
                if (data != null) {
                    databaseBaseline = data.getValue("value");
                    if ( Double.isNaN(databaseBaseline) ) {
                        databaseBaseline = 0;
                    }
                    LOG.log(Level.INFO, "Got baseline from database for {0}: {1}", new Object[]{path, databaseBaseline});
                    double persistencyBaseline = persistedBaselinesMap.get(path);
                    if (persistencyBaseline != 0) {
                        databaseBaseline += persistencyBaseline;
                        persistedBaselinesMap.put(path, 0.);
                        updatePersistedData = true;
                    }
                    databaseBaselinesMap.put(path, databaseBaseline);
                    missingBaselinesSet.remove(path);
                } else {
                    sb.append(path).append(" ");
                    if (missingBaselinesSet.add(path)) {
                        LOG.log(Level.WARNING, "Could not fetch baseline from database for {0}", new Object[]{path});
                    }
                }
            }
        }
        
        if ( updatePersistedData ) {
            persistencyService.persistNow();
        }
        
        if ( missingBaselinesSet.isEmpty() ) {
            alertService.raiseAlert(problemFetchingBaselines, AlertState.NOMINAL, "All baselines have been initialized.", AlertService.RaiseAlertStrategy.ON_SEVERITY_CHANGE);
        } else {
            alertService.raiseAlert(problemFetchingBaselines, AlertState.WARNING, "Baselines could not be fetched for data: "+missingBaselinesSet, AlertService.RaiseAlertStrategy.ON_SEVERITY_CHANGE);            
        }
    }
    
    /**
     * Add the provided value to the historical data value for the given path.
     * The value will be added internally and returned only if the historical value
     * is synchronized with the value in the database. If the historical data value
     * is not synchronized with the database we return NaN in order not to publish
     * a value that might otherwise corrupt the historical value in the database.
     * 
     * @param path The path of the historical data
     * @param increment The incremental value to add to the historical data
     * @return The cumulative value if the data is synchronized with the database. NaN otherwise.
     * 
     */
    public double accumulateData(String path, double increment) {
        long time = System.currentTimeMillis();
        synchronized (CHECK_INTERVAL) {
            if (time - lastCheckTime >= CHECK_INTERVAL) {
                lastCheckTime = time;
                initializeAccumulatedBaselinesFromDatabase();
            }
        }
        if ( Double.isFinite(increment) ) {
            synchronized (accumulationDataLock) {
                Map<String, Double> accumulationMap = getAccumulatedMapForPath(path);
                //If we don't have a baseline from the database we need to keep the current accumulated value
                //in memory to later use it to increment the database value once we
                //establish a connection. If this is happening we must publish NaN to prevent
                //corrupted data from being stored in the database.
                boolean returnNaN = accumulationMap != databaseBaselinesMap;

                double existingAccumulatedValue = accumulationMap.get(path);
                double accumulatedValue = existingAccumulatedValue + increment;
                accumulationMap.put(path, accumulatedValue);
                if ( returnNaN ) {
                    persistencyService.persistNow();
                }
                return returnNaN ? Double.NaN : accumulatedValue;
            }
        }
        return increment;
    }
    
    /**
     * Returns the accumulated data value as retrieved from the database
     * or NaN otherwise.
     * 
     * @param path The path of the accumulated data.
     * @return The accumulated value from the database or NaN otherwise.
     */
    public double getAccumulatedValueForPath(String path) {
        Map<String, Double> accumulationMap = getAccumulatedMapForPath(path);
        if ( !accumulationMap.containsKey(path) ) {
            throw new IllegalArgumentException("Path "+path+" has not been registered as an accumulated data channel "+getHistoricalDataNames());
        }
        return accumulationMap != databaseBaselinesMap ? Double.NaN : accumulationMap.get(path);
    }
    
    private Map<String,Double> getAccumulatedMapForPath(String path) {
        synchronized (accumulationDataLock) {
            boolean baselineIsFromDB = databaseBaselinesMap.containsKey(path);
            return baselineIsFromDB ? databaseBaselinesMap : persistedBaselinesMap;
        }
    }
    
    /**
     *  Command to get the set with the paths for the accumulated data channels.
     * 
     *  @return The set of names
     */
    @Command(type = Command.CommandType.QUERY, level = 0, description = "Get the set of accumulated data names")
    public Set<String> getHistoricalDataNames() {
        return new HashSet(accumulatedDataSet);
    }

    /**
     *  Command to set the historical value for a given path.
     * 
     *  @param  path  The data path
     *  @param  value The historical value. This value must be finite.
     */
    @Command(description = "Set the historical value for a given data path")
    public void setHistoricalValueForPath(@Argument(description = "Data path") String path,
                              @Argument(description = "Historical value") double value) {
        if (!accumulatedDataSet.contains(path)) {
            throw new IllegalArgumentException("Invalid historical data path: " + path+". Allowed values are: "+accumulatedDataSet);
        }
        if ( !Double.isFinite(value) ) {
            throw new IllegalAccessError("The provided value ("+value+") must be finite.");
        }
        synchronized (accumulationDataLock) {
            getAccumulatedMapForPath(path).put(path, value);            
        }
        LOG.log(Level.INFO, "Updateded historical data baseline for path {0} to value {1}.", new Object[]{path,value});
    }
    
    
    
}
