package org.lsst.ccs.subsystem.common;

import java.time.Duration;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.command.annotations.Command.CommandType;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.services.AgentPropertiesService;
import org.lsst.ccs.subsystem.common.data.BasicState;
import org.lsst.ccs.subsystem.common.data.MonitorControl;

/**
 *  Implements a basic monitoring subsystem.
 *
 *  @author Owen Saxton
 */
public class BasicMain extends Subsystem implements HasLifecycle {

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPropertiesService propertiesService;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPeriodicTaskService periodicTaskService;

    private final String agentType;
    private MonitorTaskControl monitorControl;


    /**
     *  Constructor.
     *
     *  @param  agentName  The agent name
     *  @param  agentType  The agent type property
     */
    public BasicMain(String agentName, String agentType)
    {
        super(agentName, AgentInfo.AgentType.WORKER);
        this.agentType = agentType;
        getAgentInfo().getAgentProperties().setProperty("org.lsst.ccs.use.full.paths", "true");
    }


    /**
     *  Build phase
     */
    @Override
    public void build()
    {
        // Create the monitor task control object and node
        monitorControl = MonitorTaskControl.createNode(this, MonitorControl.NODE_NAME);

        // Create and schedule an AgentPeriodicTask to update and publish the basic system state
        AgentPeriodicTask pt = new AgentPeriodicTask("basic-state", () -> updateState()).withPeriod(Duration.ofMillis(1000));
        periodicTaskService.scheduleAgentPeriodicTask(pt);
    }


    /**
     *  Subsystem post-initialization
     */
    @Override
    public void postInit()
    {
        // Set a property to define this Agent's type.
        propertiesService.setAgentProperty(agentType, getClass().getCanonicalName());
    }


    /**
     *  Gets the basic state.
     *
     *  @return  The basic system state
     */
    @Command(type=CommandType.QUERY, description="Get the full state", level=0)
    public BasicState getFullState()
    {
        return new BasicState(monitorControl.getPublishPeriod());
    }


    /**
     *  Updates the basic state.
     */
    private void updateState()
    {
        if (monitorControl.hasPeriodChanged()) {
            publishState();
        }
    }


    /**
     *  Publishes the basic system state.
     *
     *  This is intended to be called whenever any element of the state is changed.
     */
    private void publishState()
    {
        publishSubsystemDataOnStatusBus(new KeyValueData(BasicState.KEY, new BasicState(monitorControl.getPublishPeriod())));
    }    

}
