package org.lsst.ccs.monitor;

import java.util.ArrayList;
import java.util.List;
import org.lsst.ccs.bus.states.DataProviderState;
import org.lsst.ccs.commons.annotations.LookupField;

/**
 * An extension of DerivedChannel that supports basic operations among Channels.
 * 
 * @author The LSST CCS Team
 * 
 */
public class CalcChannel extends DerivedChannel {

    public static enum Operation {
        SUM,
        DIFF,
        PROD,
        RATIO;
    }
    
    @LookupField(strategy=LookupField.Strategy.TREE)
    protected Monitor    mon;        // Associated monitor object

    private Operation type = null;
    private List<String> channelPaths;
    
    private final List<Channel> channels = new ArrayList<>();

    
    @Override
    void initSensor() {
        if ( type == null ) {
            throw new IllegalArgumentException("The field \"type\" must be provided to specify the type of operation to perform.");
        }
        if ( channelPaths == null || channelPaths.isEmpty() ) {
            throw new IllegalArgumentException("The field \"channelNames\" must be provided with the list of the "
                    + "Channel paths to be used to evaluate the value for this DerivedChannel.");
        }
        if (channelPaths.size() == 1) {
            throw new IllegalArgumentException("More than one channel path must be provided.");
        } else if ( channelPaths.size() > 2 && (type == Operation.DIFF || type == Operation.RATIO) ) {
            throw new IllegalArgumentException("Operation of type "+type+" only supports two Channels, but "+channelPaths.size()+" were provided.");
        }
        
        if ( getDevice() == null ) {
            throw new RuntimeException("No Device was specified for channel "+getPath());
        }
        
        for (String path : channelPaths ) {
           
            Channel ch = mon.getChannel(path);
            if ( ch == null ) {
                throw new IllegalArgumentException("Invalid path: "+path+". Could not find a corresponding Channel.");
            }
            
            if ( ! getDevice().equals(ch.getDevice()) ) {
                throw new IllegalArgumentException("Input Channel "+ch.getPath()+" belongs to a Device (\""+ch.getDevice().getPath()+"\")"
                        + " which is different than the one for this Channel (\""+getDevice()+"\")");
            }
            channels.add(ch);
        }
        
    }
    
    

    
    @Override
    public double evaluateDerivedValue() {
        //If any of the provided channels are offline, return Double.NaN;
        for ( Channel ch : channels ) {
            if ( ch.getState() == DataProviderState.OFF_LINE ) {
                return Double.NaN;
            }
        }
        
        switch (type) {
            case DIFF:
                return channels.get(0).getValue() - channels.get(1).getValue();
            case RATIO:
                return channels.get(0).getValue() / channels.get(1).getValue();
            case SUM:
                double sum = 0;
                for ( Channel ch : channels ) {
                    sum += ch.getValue();
                }
                return sum;
            case PROD:
                double product = 0;
                for ( Channel ch : channels ) {
                    product *= ch.getValue();
                }
                return product;
        }
        return Double.NaN;
    }
    
    
    
    
}
