package org.lsst.ccs.subsystem.common.devices.dataforth;

import java.util.logging.Logger;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.commons.annotations.LookupPath;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.dataforth.Maq20;
import org.lsst.ccs.drivers.dataforth.Maq20DiscretePWM;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.subsystem.common.ErrorUtils;

/**
 *  Handles a Dataforth Maq20 system PWM control channel.
 *
 *  @author Owen Saxton
 */
public class Maq20PWMControl implements HasLifecycle, Maq20Device.Maq20Control {

    public static final double MAX_FREQUENCY = 1000.0;

    @LookupName
    protected String name;
    @LookupPath
    protected String path;
    @LookupField(strategy=LookupField.Strategy.ANCESTORS)
    protected Maq20Device maqDevc;

    protected Integer modIndex;   // Module index
    protected Integer funcIndex;  // Function index
    protected double frequency = 1000.0;  // Default value

    private static final Logger LOG = Logger.getLogger(Maq20PWMControl.class.getName());
    private Maq20Device.ModuleData modData;
    private Maq20DiscretePWM pwm;
    private double dutyCycle1 = 0.0, dutyCycle2 = 0.0;



    /**
     *  Life-cycle initialization.
     * 
     *  Checks the configuration
     */
    @Override
    public void init() {
        if (modIndex == null) {
            ErrorUtils.reportConfigError(LOG, path, "modIndex", "not specified");
        }
        modData = maqDevc.getModuleData(modIndex);
        if (modData == null) {
            ErrorUtils.reportConfigError(LOG, path, "modIndex (" + modIndex + ")", "specifies non-existent module");
        }
        if (modData.modDef.opType != Maq20.OPER_DISCRETE) {
            ErrorUtils.reportConfigError(LOG, path, "modIndex (" + modIndex + ")", "specifies non-discrete module");
        }
        if (funcIndex == null) {
            ErrorUtils.reportConfigError(LOG, path, "funcIndex", "not specified");
        }
        if (funcIndex != 0 && funcIndex != 1) {
            ErrorUtils.reportConfigError(LOG, path, "funcIndex (" + funcIndex + ")", "is not 0 or 1");
        }
        if (modData.discFunc[funcIndex] != Maq20Device.DISC_FUNC_PWM) {
            ErrorUtils.reportConfigError(LOG, path, "funcIndex (" + funcIndex + ")", "specifies non-PWM function");
        }
        if (frequency <= 0.0 || frequency > MAX_FREQUENCY) {
            ErrorUtils.reportConfigError(LOG, path, "frequency", "must be positive and no greater than " + MAX_FREQUENCY);
        }
    }


    /**
     *  Post-connection initialization.
     *
     *  @throws DriverException
     */
    @Override
    public void initialize() throws DriverException
    {
        pwm = modData.maqDiscPWM;
        if (pwm.isEnabled(funcIndex)) {
            int timeBase = pwm.getTimebase(funcIndex);
            int mult = timeBase == Maq20DiscretePWM.TIMEBASE_USECS ? 1
                         : timeBase == Maq20DiscretePWM.TIMEBASE_MSECS ? 1000 : 1_000_000;
            double period = mult * pwm.getPeriod(funcIndex);
            setDutyCycle1(1.0 - mult * pwm.getLowTime1(funcIndex) / period);
            setDutyCycle2(1.0 - mult * pwm.getLowTime2(funcIndex) / period);
        }
        else {
            pwm.enable(funcIndex);
        }
        pwm.setTimebase(funcIndex, Maq20DiscretePWM.TIMEBASE_USECS);
        double period = 1_000_000 / frequency;
        pwm.setPeriod(funcIndex, (int)period);
        pwm.setLowTime1(funcIndex, (int)(period * (1.0 - dutyCycle1)));
        pwm.setLowTime2(funcIndex, (int)(period * (1.0 - dutyCycle2)));
        pwm.enableOutput2(funcIndex, dutyCycle2 != 0.0);
        pwm.armFunction(funcIndex);
    }


    /**
     *  Pre-closure cleanup.
     */
    @Override
    public void close()
    {
        pwm = null;
    }


    /**
     *  Gets the frequency.
     *
     *  @return  The frequency
     */
    public double getFrequency()
    {
        return frequency;
    }


    /**
     *  Sets duty cycle 1.
     *
     *  @param  duty  The duty cycle (0 - 1)
     *  @throws DriverException
     */
    public void setDutyCycle1(double duty) throws DriverException
    {
        dutyCycle1 = Math.min(Math.max(duty, 0.0), 1.0);
        if (pwm != null) {
            int period = (int)(1_000_000 / frequency);
            pwm.setLowTime1(funcIndex, (int)(period * (1.0 - dutyCycle1)));
            pwm.armFunction(funcIndex);
        }
    }


    /**
     *  Gets duty cycle 1.
     *
     *  @return  The duty cycle (0 - 1)
     */
    public double getDutyCycle1()
    {
        return dutyCycle1;
    }


    /**
     *  Sets duty cycle 2.
     *
     *  @param  duty  The duty cycle (0 - 1)
     *  @throws DriverException
     */
    public void setDutyCycle2(double duty) throws DriverException
    {
        dutyCycle2 = Math.min(Math.max(duty, 0.0), 1.0);
        if (pwm != null) {
            int period = (int)(1_000_000 / frequency);
            pwm.setLowTime2(funcIndex, (int)(period * (1.0 - dutyCycle2)));
            pwm.enableOutput2(funcIndex, dutyCycle2 != 0.0);
            pwm.armFunction(funcIndex);
        }
    }


    /**
     *  Gets duty cycle 2.
     *
     *  @return  The duty cycle (0 - 1)
     */
    public double getDutyCycle2()
    {
        return dutyCycle2;
    }


    /**
     *  Gets the name.
     * 
     *  @return  The name
     */
    public String getName()
    {
        return name;
    }


    /**
     *  Gets the path.
     * 
     *  @return  The path
     */
    public String getPath()
    {
        return path;
    }

}
