package org.lsst.ccs.subsystem.common;

import java.time.Duration;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.command.annotations.Command.CommandType;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.services.AgentPropertiesService;
import org.lsst.ccs.subsystem.common.data.BasicState;

/**
 *  Implements a basic monitoring subsystem.
 *
 *  @author Owen Saxton
 */
public class BasicMain extends Subsystem implements HasLifecycle {

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPropertiesService propertiesService;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPeriodicTaskService periodicTaskService;

    private final String agentType;


    /**
     *  Constructor.
     *
     *  @param  agentName  The agent name
     *  @param  agentType  The agent type property
     */
    public BasicMain(String agentName, String agentType) {
        super(agentName, AgentInfo.AgentType.WORKER);
        this.agentType = agentType;
        getAgentInfo().getAgentProperties().setProperty("org.lsst.ccs.use.full.paths", "true");
    }


    /**
     *  Subsystem post-initialization
     */
    @Override
    public void postInit()
    {
        // Set a property to define that this Agent's type.
        propertiesService.setAgentProperty(agentType, getClass().getCanonicalName());
    }


    /**
     *  Sets the update (tick) period.
     *
     *  @param  period  The update period (milliseconds) to set.
     */
    @Command(type=CommandType.ACTION, description="Set the update period")
    public void setUpdatePeriod(@Argument(name="period", description="The tick period (msecs)")
                                int period)
    {
        setTickPeriod(period);
        publishState();    // Must do this for GUI
    }


    /**
     *  Gets the basic state.
     *
     *  @return  The full PDU system state
     */
    @Command(type=CommandType.QUERY, description="Get the full state")
    public BasicState getFullState()
    {
        return new BasicState(getTickPeriod());
    }


    /**
     *  Sets the tick period
     */
    private void setTickPeriod(long period)
    {
        periodicTaskService.setPeriodicTaskPeriod("monitor-publish", Duration.ofMillis(period));
    }
    

    /**
     *  Gets the tick period
     */
    private int getTickPeriod()
    {
        return (int)periodicTaskService.getPeriodicTaskPeriod("monitor-publish").toMillis();
    }
    

    /**
     *  Publishes the state of the Eaton PDU system.
     *
     *  This is intended to be called whenever any element of the state is changed.
     */
    private void publishState()
    {
        publishSubsystemDataOnStatusBus(new KeyValueData(BasicState.KEY, new BasicState(getTickPeriod())));
    }    

}
