package org.lsst.ccs.subsystem.common.devices.pluto;

import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.pluto.Pluto;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.subsystem.common.ErrorUtils;

/**
 *  Handles a Pluto PLC system.
 *
 *  @author Owen Saxton
 */
public class PlutoDevice extends Device {

    /**
     *  Constants.
     */
    private static final int
        TYPE_GLOBAL  = 0,
        TYPE_ADD_BIT = 1,
        TYPE_ADD_REG = 2,
        TYPE_ADD_INT = 3;

    /**
     *  Private lookup maps.
     */
    private final static Map<String, Integer> typeMap = new HashMap<>();
    static {
        typeMap.put("GLOBAL", TYPE_GLOBAL);
        typeMap.put("ADDBIT", TYPE_ADD_BIT);
        typeMap.put("ADDREG", TYPE_ADD_REG);
        typeMap.put("ADDINT", TYPE_ADD_INT);
    }

    /**
     *  Data fields.
     */
    @ConfigurationParameter(category="Device", isFinal=true)
    protected volatile String node;          // Network node name

    protected List<Integer> areas;  // Area configuration data

    private static final Logger LOG = Logger.getLogger(PlutoDevice.class.getName());
    protected Pluto plu;                // Associated Pluto object
    private final boolean configAreas;  // Whether to require area configuration
    private int nArea;                  // Area count
    private final int[] globData = new int[Pluto.NUM_MODULES];
    private final int[] addData = new int[Pluto.NUM_ADD_AREAS];
    private final Set<Integer> modsUsed = new HashSet<>();
    private final Set<Integer> areasUsed = new HashSet<>();


    /**
     *  Constructor.
     */
    public PlutoDevice()
    {
        configAreas = true;
    }


    /**
     *  Constructor.
     * 
     *  @param  nArea  The number of additional areas to use, or -1 if the areas need to be configured
     */
    public PlutoDevice(int nArea)
    {
        this.nArea = nArea;
        configAreas = nArea < 0;
    }


    /**
     *  Performs configuration.
     */
    @Override
    protected void initDevice()
    {
        super.configure(mon);
        if (node == null) {
            ErrorUtils.reportConfigError(LOG, getName(), "node", "is missing");
        }
        if (configAreas) {
            if (areas == null) {
                ErrorUtils.reportConfigError(LOG, getName(), "areas", "is missing");
            }
            if ((areas.size() & 1) != 0) {
                ErrorUtils.reportConfigError(LOG, getName(), "areas", "has odd number of elements");
            }
            nArea = areas.size() / 2;
        }
        fullName = "Pluto PLC system (" + node + ")";
    }


    /**
     *  Performs full initialization.
     */
    @Override
    protected void initialize()
    {
        try {
            if (!inited || plu == null) {
                plu = new Pluto();
            }
            plu.open(node);
            if (configAreas) {
                plu.configStart(-1, 15, 0, 100);
                for (int j = 0; j < areas.size(); j += 2) {
                    plu.configDataArea(j / 2, areas.get(j), areas.get(j + 1));
                }
                plu.configWrite();
            }
            setOnline(true);
            initSensors();
            setOutputLines();
            LOG.log(Level.INFO, "Connected to {0}", fullName);
        }
        catch (DriverException e) {
            if (!inited) {
                LOG.log(Level.SEVERE, "Error connecting to {0}: {1}", new Object[]{fullName, e});
            }
            if (plu != null) {
                close();
            }
        }
        finally {
            inited = true;
        }
    }


    /**
     *  Closes the connection.
     */
    @Override
    protected void close()
    {
        try {
            plu.close();
        }
        catch (DriverException e) {
        }
    }


    /**
     *  Checks a channel's parameters for validity.
     *
     *  @param  name     The channel name
     *  @param  hwChan   The hardware channel number
     *  @param  type     The channel type string
     *  @param  subtype  The channel subtype string
     *  @return  A two-element array containing the encoded type [0] and subtype [1] values.
     *  @throws  Exception if any errors found in the parameters.
     */
    @Override
    protected int[] checkChannel(String name, int hwChan, String type,
                                 String subtype)
        throws Exception
    {
        Integer iType = null;
        int id;
        String[] typeFields = type.split(":", -1);
        if (typeFields.length == 2) {
            iType = typeMap.get(typeFields[0].toUpperCase());
        }
        try {
            id = Integer.decode(typeFields[1]);
        }
        catch (NumberFormatException e) {
            id = -1;
        }
        int maxId = 0;
        if (iType != null) {
            maxId = iType == TYPE_GLOBAL ? Pluto.NUM_MODULES : nArea;
            int maxChan = iType == TYPE_GLOBAL ? 32 : iType == TYPE_ADD_BIT ? 16 : iType == TYPE_ADD_REG ? 2 : 1;
            if (hwChan < 0 || hwChan >= maxChan) {
                ErrorUtils.reportChannelError(LOG, name, "hwchan", hwChan);
            }
        }
        if (iType == null || id < 0 || id >= maxId) {
            ErrorUtils.reportChannelError(LOG, name, "type", type);
        }
        Set used = iType == TYPE_GLOBAL ? modsUsed : areasUsed;
        used.add(id);

        return new int[]{(iType << 8) | id, 0};
    }


    /**
     *  Reads all referenced channels.
     */
    @Override
    protected void readChannelGroup()
    {
        if (!online) return;
        try {
            for (int id : modsUsed) {
                globData[id] = plu.readGlobalData(id);
            }
            for (int id : areasUsed) {
                addData[id] = plu.readAdditionalData(id);
            }
        }
        catch (DriverException e) {
            LOG.log(Level.SEVERE, "Error reading {0}: {1}", new Object[]{fullName, e});
            setOnline(false);
        }
    }


    /**
     *  Reads a channel.
     *
     *  @param  hwChan   The hardware channel number.
     *  @param  type     The encoded channel type returned by checkChannel.
     *  @return  The read value
     */
    @Override
    protected double readChannel(int hwChan, int type)
    {
        double value = super.readChannel(hwChan, type);

        if (online) {
            int index = type & 0xff;
            switch (type >> 8) {

            case TYPE_GLOBAL:
                value = (globData[index] >> hwChan) & 1;
                break;

            case TYPE_ADD_BIT:
                value = (addData[index] >> hwChan) & 1;
                break;

            case TYPE_ADD_REG:
                value = (addData[index] << (16 * (1 - hwChan))) >> 16;  // Sign extend
                break;

            case TYPE_ADD_INT:
                value = addData[index];
                break;
            }
        }

        return value;
    }


    /**
     *  Checks an output line number.
     *
     *  @param  name  The name of the output line.
     *  @param  line  The hardware line number of the output line.
     *  @throws  Exception  If line number is invalid
     */
    @Override
    protected void checkHwLine(String name, int line) throws Exception
    {
        if (line < 0 || line >= 16 * Pluto.NUM_DTP_AREAS) {
            ErrorUtils.reportChannelError(LOG, name, "line", line);
        }
        addLine(line);
    }


    /**
     *  Sets an output line on or off.
     *
     *  @param  line  The output line number.
     *  @param  on    The on state to set: true or false
     */
    @Override
    protected void setHwLine(int line, boolean on)
    {
        writeBit(line / 16, line & 0x0f, on ? 1 : 0);
    }


    /**
     *  Gets the set state of an output line.
     *
     *  @param  line  The output line number.
     *  @return  Whether the line is on
     */
    @Override
    protected Boolean isHwLineSet(int line)
    {
        return readBit(line / 16, line & 0x0f) != 0;
    }


    /**
     *  Gets whether the PLC is active.
     *
     *  @return  Whether active, or null if offline (or other read error).
     */
    public Boolean isPLCActive()
    {
        try {
            return plu.readModuleStatus() != 0;
        }
        catch(DriverException e) {
            return null;
        }
    }


    /**
     *  Toggles a bit.
     *
     *  @param  area   The area number
     *  @param  bit    The bit number
     */
    public void toggleBit(int area, int bit)
    {
        writeBit(area, bit, 1);
        try {
            Thread.sleep(200);
        }
        catch (InterruptedException e) {}
        writeBit(area, bit, 0);
    }


    /**
     *  Writes a bit value.
     *
     *  @param  area   The area number
     *  @param  bit    The bit number
     *  @param  value  The bit value
     */
    public void writeBit(int area, int bit, int value)
    {
        try {
            plu.writeAreaBit(area, bit, value);
        }
        catch (DriverException e) {
            LOG.log(Level.SEVERE, "Error writing to {0}: {1}", new Object[]{fullName, e});
            setOnline(false);
        }
    }


    /**
     *  Reads a bit value.
     *
     *  @param  area   The area number
     *  @param  bit    The bit number
     *  @return  The bit value, or null if device is offline
     */
    public Integer readBit(int area, int bit)
    {
        try {
            return plu.readAreaBit(area, bit);
        }
        catch (DriverException e) {
            LOG.log(Level.SEVERE, "Error reading from {0}: {1}", new Object[]{fullName, e});
            setOnline(false);
            return null;
        }
    }


    /**
     *  Reads a bit value from an additional area.
     *
     *  @param  area   The area number
     *  @param  bit    The bit number (0 - 31)
     *  @return  The bit value, or null if device offline
     */
    public Integer readAddBit(int area, int bit)
    {
        if (!online) return null;
        try {
            return (plu.readAdditionalData(area) >> bit) & 1;
        }
        catch (DriverException e) {
            LOG.log(Level.SEVERE, "Error reading from {0}: {1}", new Object[]{fullName, e});
            setOnline(false);
            return null;
        }
    }


    /**
     *  Reads a signed word (16-bit) value from an additional area.
     *
     *  @param  area   The area number
     *  @param  word   The word number (0 - 1)
     *  @return  The bit value, or null if device offline
     */
    public Integer readAddWord(int area, int word)
    {
        if (!online) return null;
        try {
            int value = plu.readAdditionalData(area);
            return word == 0 ? (short)value : value >> 16;
        }
        catch (DriverException e) {
            LOG.log(Level.SEVERE, "Error reading from {0}: {1}", new Object[]{fullName, e});
            setOnline(false);
            return null;
        }
    }


    /**
     *  Reads the value from an additional area.
     *
     *  @param  area   The area number
     *  @return  The integer value, or null if device offline
     */
    public Integer readAddInt(int area)
    {
        if (!online) return null;
        try {
            return plu.readAdditionalData(area);
        }
        catch (DriverException e) {
            LOG.log(Level.SEVERE, "Error reading from {0}: {1}", new Object[]{fullName, e});
            setOnline(false);
            return null;
        }
    }


    /**
     *  Writes a register value.
     *
     *  @param  area   The area number
     *  @param  reg    The register number
     *  @param  value  The register value
     */
    public void writeRegister(int area, int reg, int value)
    {
        try {
            plu.writeAreaRegister(area, reg, value);
        }
        catch (DriverException e) {
            LOG.log(Level.SEVERE, "Error writing to {0}: {1}", new Object[]{fullName, e});
            setOnline(false);
        }
    }

}
