package org.lsst.ccs.subsystem.common;

import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * Proportional, Integral, Differential controller
 *
 * This PID Controller class implements a proportional, integral, derivative
 * controller for use in the thermal control of the LSST camera electronics.
 *
 * @author CCS Team
 */
public class PIDController {

    private static final Logger LOG = Logger.getLogger(PIDController.class.getName());

    // Parameters
    private double coefP = 3.0;
    private double coefI = 1.0;
    private double coefD = 0.0;
    private double timeConst = 100;         // integration time constant
    private double smoothTime = 30.; // input smoothing time in seconds   
    private double maxOutput = 100.0; // maximum PID output
    private double minOutput = 0.0;   // minimum PID output
    private double maxInput = 0.0;    // maximum input - limit setpoint to this
    private double minInput = 0.0;    // minimum input - limit setpoint to this
    private double setpoint = 0.0;  // the desired average of the input
    private double baseOutput = 0.;   // base output

    // Fields
    private double aveInput;          // average input value
    private double input = 0.;
    private double error = 0.;
    private double integral = 0.;
    private double lastError = 0.;
    private double measTime;
    private double lastTime = 0.;     // previous measurement time
    private double lastOutput = 0.;     // previous output

    /**
     * Allocate a PI object with the given constants for P, I
     *
     */
    public PIDController() {
    }

    /**
     * Return the current PI result. This is constrained by the max and min outs
     *
     * @param input The array of input values
     * @param time The measurement time
     * @return the latest calculated output
     */
    public double performPID(double[] input, double time) {
        aveInput = 0.;
        for (double mInput : input) {
            aveInput += mInput / input.length;
        }
        measTime = time;
        return calculate();
    }

    /**
     * Set the time constant characterizing the integration
     *
     * @param time Integral coefficient (integration time constant)
     */
    public void setTimeConst(double time) {
        timeConst = time;
    }

    /**
     * Sets the smoothing time.
     *
     * @param time the smoothing time
     */
    public void setSmoothTime(double time) {
        smoothTime = time;
    }

    /**
     * Set the Proportional coefficient
     *
     * @param value proportional coefficient
     */
    public void setCoefP(double value) {
        coefP = value;
    }

    /**
     * Set the Integral coefficient
     *
     * @param value integral coefficient
     */
    public void setCoefI(double value) {
        coefI = value;
    }

    /**
     * Set the Derivative coefficient
     *
     * @param value derivative coefficient
     */
    public void setCoefD(double value) {
        coefD = value;
    }

    /**
     * Get the Proportional coefficient
     *
     * @return proportional coefficient
     */
    public double getCoefP() {
        return coefP;
    }

    /**
     * Get the Integral coefficient
     *
     * @return integral coefficient
     */
    public double getCoefI() {
        return coefI;
    }

    /**
     * Get the Derivative coefficient
     *
     * @return derivative coefficient
     */
    public double getCoefD() {
        return coefD;
    }

    /**
     * Get the Integral coefficient
     *
     * @return integral coefficient
     */
    public double getTimeConst() {
        return timeConst;
    }

    /**
     * Sets the base output.
     *
     * @param value the base output value
     */
    public void setBaseOutput(double value) {
        baseOutput = value;
    }

    /**
     * Sets the maximum and minimum values expected from the input.
     *
     * @param minimumInput the minimum value expected from the input
     * @param maximumInput the maximum value expected from the output
     */
    public void setInputRange(double minimumInput, double maximumInput) {
        minInput = minimumInput;
        maxInput = maximumInput;
        setSetpoint(setpoint);
    }

    /**
     * Sets the minimum and maximum values to write.
     *
     * @param minimumOutput the minimum value to write to the output
     * @param maximumOutput the maximum value to write to the output
     */
    public void setOutputRange(double minimumOutput, double maximumOutput) {
        minOutput = minimumOutput;
        maxOutput = maximumOutput;
    }

    /**
     * Set the setpoint for the PIDController
     *
     * @param setpnt the desired setpoint
     */
    public void setSetpoint(double setpnt) {
        setpoint = (maxInput <= minInput) ? setpnt
                : (setpnt > maxInput) ? maxInput
                        : (setpnt < minInput) ? minInput : setpnt;
    }

    /**
     * Returns the current setpoint of the PIDController
     *
     * @return the current setpoint
     */
    public double getSetpoint() {
        return setpoint;
    }

    /**
     * Returns the current difference of the input from the setpoint
     *
     * @return the current error
     */
    public double getError() {
        return error;
    }

    /**
     * Returns the current error integral
     *
     * @return the current error integral
     */
    public double getIntegral() {
        return integral;
    }

    /**
     * Sets the error integral
     *
     * @param value the error integral value to set
     */
    public void setIntegral(double value) {
        integral = value;
    }

    /**
     * Returns the current smoothed input
     *
     * @return the current smoothed input
     */
    public double getInput() {
        return input;
    }

    /**
     * Reset the previous error, the integral, and disable the controller.
     */
    public void reset() {
        integral = 0.;
        lastTime = Double.NaN;
        lastError = Double.NaN;
        lastOutput = 0.;
    }

    /**
     * Smooth the input, perform the PID algorithm. Only called by performPID()
     */
    private double calculate() {
        double deltaIntegral = 0.0;
        double deltaTime;
        double derivative = 0.0;
        double smoothTimeEff = smoothTime;
        double maxStep = (maxOutput - minOutput) / 4.0; // default

        // sanity check constraints
        if (Double.isNaN(lastTime)) {  // re-init after reset
            lastTime = measTime - 1.;
            input = aveInput;
        }
        deltaTime = measTime - lastTime;
        if (smoothTimeEff < deltaTime) {
            smoothTimeEff = deltaTime;
        }
        if (timeConst > smoothTimeEff) {
            maxStep = (maxOutput - minOutput) * smoothTimeEff / timeConst;  // limits ramping
        }

        // smooth the input
        input = (deltaTime * aveInput + (smoothTimeEff - deltaTime) * input) / smoothTimeEff;

        // PID basic terms all in same units and ~order of magnitude as error
        error = setpoint - input;
        if (timeConst > 0.0) {
            deltaIntegral = error * deltaTime / timeConst;
        }
        if (!Double.isNaN(lastError)) {
            derivative = smoothTimeEff * (error - lastError) / deltaTime;
        }

        // calculate output and adjust for min, max, step constraints
        double output = coefP * error + coefI * (integral + deltaIntegral) + coefD * derivative;
        if (output > maxOutput) {
            output = maxOutput;
        } else if (output < minOutput) {
            output = minOutput;
        } else {
            integral += deltaIntegral;  // increment if not pegged, anti-windup
        }
        output = (output > lastOutput + maxStep) ? lastOutput + maxStep : output;  // limit ramp up

        lastError = error;
        lastTime = measTime;
        lastOutput = output;
        double lastDerivative = derivative;
        double lastDeltaIntegral = deltaIntegral;

        // TODO take this out later
        LOG.log(Level.FINE, () -> String.format("raw input      = %.2f", aveInput));
        LOG.log(Level.FINE, () -> String.format("smoothed input = %.2f", input));
        LOG.log(Level.FINE, () -> String.format("error          = %.4f", error));
        LOG.log(Level.FINE, () -> String.format("deltaIntegral  = %.4f", lastDeltaIntegral));
        LOG.log(Level.FINE, () -> String.format("integral       = %.4f", integral));
        LOG.log(Level.FINE, () -> String.format("derivative     = %.4f", lastDerivative));
        LOG.log(Level.FINE, () -> String.format("output         = %.4f", lastOutput));

        // add base output
        return output + baseOutput;
    }

}
