package org.lsst.ccs.subsystem.common.devices.turbopump;

import java.util.HashMap;
import java.util.Map;
import java.util.logging.Logger;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.ConfigurationParameterChanger;
import org.lsst.ccs.drivers.twistorr.TwisTorr84;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.commons.DriverTimeoutException;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.subsystem.common.ErrorUtils;

/* Pre-defined ommand levels */
import static org.lsst.ccs.command.annotations.Command.NORMAL;
import static org.lsst.ccs.command.annotations.Command.ENGINEERING1;
import static org.lsst.ccs.command.annotations.Command.ENGINEERING2;
import static org.lsst.ccs.command.annotations.Command.ENGINEERING3;

/**
 *  Interfaces with driver for Agilent TwisTorr84 turbo pump.
 *
 *  @author Al Eisner
 */
public class TwisTorr84Device extends Device {

   /**
    *  Data fields.
    */

    private final TwisTorr84 turbo = new TwisTorr84();  // Associated driver

   /**
    *  Private lookup maps for driver read commands.  The default
    *  initial capacity of 16 is reasonable for the expected number
    *  of read commands.  The key is a numeric identifier of a monitor
    *  channel defined in the groovy file.
    *
    *  typeMap identifies whether a quantity to be read is of type
    *  logical ("Bool"), integer ("Numeric") or alphanumeric ("Alpha").
    *  It is taken from the type parameter in the groovy file.
    *
    *  itemMap identifies the specific quantity ro be read.  It is
    *  taken from the subtype parameter in the groovy file.
    */
    private final Map<Integer, String> typeMap = new HashMap<>();
    private final Map<Integer, String> itemMap = new HashMap<>();
    private Integer chanKey;

    /* Configuration parameters related to hardware settings */

    @ConfigurationParameter(isFinal = true,
                            description = "serial port")
    protected volatile String devcId;       // Device connection identification

    // The code for the FS84 TurboPump with "Navigator" controller 
    // largely works for the FS304 TurboPump with on-board controller,
    // but some commands are not implemented in the latter.  This
    // parameter allows the code to take both cases into account.
    @ConfigurationParameter(isFinal = true,
                            description = "true/false for model 304/84 pump")
	protected volatile Boolean model304;

    @ConfigurationParameter(isFinal = true,
                            description = "true/false for low/normal speed")
    protected volatile Boolean lowSpeedMode;

    @ConfigurationParameter(isFinal = true, 
                            description = "true/false for water/air cooling")
    protected volatile Boolean waterCooling; 

    @ConfigurationParameter(isFinal = true,
                            description = "true/false for manual/auto")
    protected volatile Boolean ventValveByCmnd;  

    @ConfigurationParameter(isFinal = true,
                            description = "true/false for continuous/impulse")
    protected volatile Boolean interlockType;   // 

    @ConfigurationParameter(description = "Soft-Start mode <true|false>, settable only if pump is stopped")
    protected volatile Boolean softStartMode; 

    @ConfigurationParameter(description = "Active-Stop mode <true|false>, settable only if pump is stopped")
    protected volatile Boolean activeStopMode;

    private static final Logger LOG = Logger.getLogger(TwisTorr84Device.class.getName());

    /* Hard-coded hardware-initialization parameters (depend on pump model)*/

    private final int rotfreq_set_84  = 1350;  // normal rotation frequncy (Hz) 
    private final int rotfreq_set_304 = 1010;
    private final int rotfreq_low_84  = 1100;  // slow rotation frequncy (Hz) 
    private final int rotfreq_low_304 =  900;
    private final Boolean gasType_84 = false;  // true/false for Argon/Nitrogen

    /* Omit serialType, since driver will not allow setting this */
    // private final Boolean serialType = false;  // true/false for RS-485/RS-232

   /**
    *  Performs basic initialization.  
    *  Verify that all ConfigurationParameters are set.
    */
    @Override
    protected void initDevice() {

        chanKey = 0;

        if (devcId == null) {
            ErrorUtils.reportConfigError(LOG, name, "devcId", "is missing");
        }
        fullName = "TwisTorr84 (" + devcId + ")";

        if (model304 == null) {
            ErrorUtils.reportConfigError(LOG, name, "model304", "is missing");
        }
        if (lowSpeedMode == null) {
            ErrorUtils.reportConfigError(LOG, name, "lowSpeedMode", "is missing");
        }
        if (waterCooling == null) {
            ErrorUtils.reportConfigError(LOG, name, "waterCooling", "is missing");
        }
        if (ventValveByCmnd == null) {
            ErrorUtils.reportConfigError(LOG, name, "ventValveByCmnd", "is missing");
        }
        if (interlockType == null) {
            ErrorUtils.reportConfigError(LOG, name, "interlockType", "is missing");
        }
        if (softStartMode == null) {
            ErrorUtils.reportConfigError(LOG, name, "softStartMode", "is missing");
        }
        if (activeStopMode == null) {
            ErrorUtils.reportConfigError(LOG, name, "activeStopMode", "is missing");
        }
    }


   /**
    *  Performs full initialization.
    */
    @Override
    protected void initialize() {

        try {
            turbo.open(devcId);

            /* Hardware initialization */

            turbo.writeBool(TwisTorr84.CmndBool.LOW_SPEED, lowSpeedMode);
            turbo.writeBool(TwisTorr84.CmndBool.VENTVALVE_BY_CMND, ventValveByCmnd);

            /* Settings requiring that pump be in stopped state */
            int status = turbo.readNumeric(TwisTorr84.CmndNumeric.STATUS);
            if (TwisTorr84.PumpStatus.decode(status).equals("STOP")) { 
                turbo.writeBool(TwisTorr84.CmndBool.SOFT_START, softStartMode);
                turbo.writeBool(TwisTorr84.CmndBool.ACTIVE_STOP,activeStopMode);
                turbo.writeBool(TwisTorr84.CmndBool.WATER_COOLING,waterCooling);
                turbo.writeBool(TwisTorr84.CmndBool.INTERLOCK_TYPE, interlockType);

                /* Settings dependent on turbo pump model, FS84 vs. FS304 */
                if (!model304) {
                    turbo.writeBool(TwisTorr84.CmndBool.GAS_TYPE_ARGON, gasType_84);
                    turbo.writeNumeric(TwisTorr84.CmndNumeric.ROTFREQ_SET,rotfreq_set_84);
                    turbo.writeNumeric(TwisTorr84.CmndNumeric.ROTFREQ_LOW,rotfreq_low_84);
                } else {
                    turbo.writeNumeric(TwisTorr84.CmndNumeric.ROTFREQ_SET,rotfreq_set_304);
                    turbo.writeNumeric(TwisTorr84.CmndNumeric.ROTFREQ_LOW,rotfreq_low_304);
                }
            } else {
		LOG.info("TurboPump not in Stopped state - " 
                         + "\n  SOFT_START, ACTIVE_STOP, and a few other settings cannot be changed");
            }
            boolean as = turbo.readBool(TwisTorr84.CmndBool.ACTIVE_STOP);
            boolean ss = turbo.readBool(TwisTorr84.CmndBool.SOFT_START);
            setOnline(true);
            initSensors();
            LOG.info("\n Connected to " + fullName);
            LOG.info("Hardware initialized for " + fullName 
                     + "\n Active-Stop mode = "+ Boolean.toString(as)
                     + ",  Soft-Start mode  = "+ Boolean.toString(ss));
        }

        catch (DriverException e) {
            if (!inited) {
                LOG.severe("Error connecting to or initializaing " 
                          + fullName + ": " + e);
            }
            close();
        }

        inited = true;
    }


   /**
    *  Closes the connection.
    */
    @Override
    protected void close() {
        try {
            turbo.close();
        }
        catch (DriverException e) {
        }
    }

    /**
     * Pump commands and settings
     */

   /**
    *  Select soft start vs. normal start
    *
    *  @param value  True for soft-start mode, false for normal mode.
    *  @throws  DriverException
    */
    @ConfigurationParameterChanger(propertyName = "softStartMode")
    public void setSoftStartMode(Boolean value) throws DriverException
    {
        if (isOnline()) {
            if (value != null) {
                LOG.info(fullName + 
                         "config. change request: set SOFT_START = "+ value);
                turbo.writeBool(TwisTorr84.CmndBool.SOFT_START, value);
                softStartMode = value;
            } else {
                throw new IllegalArgumentException(fullName + ": improper value for softStartMode, not changed");
            }
        } else {                      // initialization call
            if (value != null) {softStartMode = value;}
        }
    }

   /**
    *  Select active stop (with braking) vs. normal stop mode
    *
    *  @param value  True for active-stop mode, false for normal-stop mode.
    *  @throws  DriverException
    */
    @ConfigurationParameterChanger(propertyName = "activeStopMode")
    public void setActiveStopMode(Boolean value) throws DriverException
    {
        if (isOnline()) {
            if (value != null) {
                LOG.info(fullName + 
                         "config. change request: set ACTIVE_STOP = "+ value);
                turbo.writeBool(TwisTorr84.CmndBool.ACTIVE_STOP, value);
                activeStopMode = value;
            } else {
                throw new IllegalArgumentException(fullName + ": improper value for activeStopMode, not changed");
            }
        } else {                      // initialization call
            if (value != null) {activeStopMode = value;}
        }
    }



   /**
    *  Start pump
    *
    *  @throws  DriverException
    */
    @Command(type=Command.CommandType.ACTION, name="startTurboPump", 
             description="Start turbo pump")
    public void startTurboPump() throws DriverException
    {
        turbo.writeBool(TwisTorr84.CmndBool.START_STOP, true);
    }

   /**
    *  Stop pump
    *
    *  @throws  DriverException
    */
    @Command(type=Command.CommandType.ACTION, name="stopTurboPump", 
             description="Stop turbo pump")
    public void stopTurboPump() throws DriverException 
    {
        turbo.writeBool(TwisTorr84.CmndBool.START_STOP, false);
    }

   /**
    *  Open or close the vent valve
    */
   //  To be added later (currently there is no controllable vent valve).

   /**
    *  Checks a channel's parameters for validity.
    *
    *  @param  name     The channel name
    *  @param  hwChan   The hardware channel number
    *  @param  type     The channel type string
    *  @param  subtype  The channel subtype string
    *  @return  A two-element array containing the encoded type [0] and
    *           subtype [1] values.
    *  @throws  Exception if any errors found in the parameters.
    */
    @Override
    protected int[] checkChannel(String name, int hwChan, String type, 
                                 String subtype) throws Exception
    {
        try {
            switch (type) {
                case "Bool":
                    TwisTorr84.CmndBool.valueOf(subtype);
                    break;
                case "Numeric":
                    TwisTorr84.CmndNumeric.valueOf(subtype);
                    break;
                case "Alpha":
                    TwisTorr84.CmndAlpha.valueOf(subtype);
                    break;
                default:
                    ErrorUtils.reportChannelError(LOG, name, "type", type);
            }
        }
        catch (IllegalArgumentException e) {
            ErrorUtils.reportChannelError(LOG, name, type + " subtype ",
					subtype);
        }
        chanKey++;
        typeMap.put(chanKey,type);
        itemMap.put(chanKey,subtype);
        return new int[]{chanKey, 0};
    }


   /**
    *  Reads a channel.
    *
    *  @param  hwChan   The hardware channel number.
    *  @param  type     The encoded channel type returned by checkChannel.
    *  @return  The read value
    */
    @Override
    protected double readChannel(int hwChan, int type)
    {
        double value = super.readChannel(hwChan, type);   //NaN
        String itemType = typeMap.get(type);
        String item = itemMap.get(type);
	// Note:  the framework assures that null "get" returns do not occur

        try {
	    switch (itemType) {
                case "Bool":
                    if (turbo.readBool(TwisTorr84.CmndBool.valueOf(item))) {
                        value = 1.0;
			    } else {
                        value = 0.0;
                    }
                    break;
                case "Numeric":
                    value = turbo.readNumeric(TwisTorr84.CmndNumeric.valueOf(item));
                    break;
                case "Alpha":
                    turbo.readAlpha(TwisTorr84.CmndAlpha.valueOf(item));
                    // Not implemented, relevant hardware not installed
                    break;
	    }
        }        
        catch (DriverTimeoutException et) {
            LOG.severe("TurboPump timeout reading data " + item + ": " + et);
            setOnline(false);
        }
	catch (DriverException e) {
            LOG.severe("TurboPump exception reading data " + item + ": " + e);
        }
        
        return value;
    }

    /* Read-commands for command-line usage */

   /** 
    *  Read and decode pump status
    *
    *  @return  String indicating pump state or failure
    *  @throws  DriverException
    */
    @Command(type=Command.CommandType.QUERY, level=NORMAL, name="readTurboStatus", 
             description="Get pump status, including operating modes and failure")
    public TwisTorr84.PumpStatus readTurboStatus()  throws DriverException
    {
        int status = turbo.readNumeric(TwisTorr84.CmndNumeric.STATUS);
        return TwisTorr84.PumpStatus.decodeStatus(status);
    }

   /** 
    *  Read error code (bits show reason for pump failure)
    *
    *  @return  String giving error code in hexadecimal and decoding it
    *  @throws  DriverException
    */
    @Command(type=Command.CommandType.QUERY, name="readTurboFailCode", 
             description="Find reason(s) for failure", level=NORMAL)
    public String readTurboFailCode()  throws DriverException
    {
        int errcode = turbo.readNumeric(TwisTorr84.CmndNumeric.ERRCODE);
        return Integer.toString(errcode,16) + ", "+ TwisTorr84.decodeError(errcode);
    }

   /** 
    *  Read TurboPump running-time in hours
    *
    *  @return  String giving active time
    *  @throws  DriverException
    */
    @Command(type=Command.CommandType.QUERY,level=ENGINEERING1,
             name="readTurboLife",
             description="show TurboPump running-time in hours")
    public String readTurboLife() throws DriverException
    {
        int life = turbo.readNumeric(TwisTorr84.CmndNumeric.PUMP_HOURS);
        return Integer.toString(life) + " hours";
    }

   /**
    * Read all settings and data from pump controller.
    *
    * Loops over all read commands and returns them in a table format.
    * All DriverExceptions are caught; if one occurs, the data field
    * is replaced by the text (String) associated with the exception.
    *  
    * @return  String reporting all data read and exceptions.
    */
    @Command(type=Command.CommandType.QUERY,level=ENGINEERING1,name="readAll", 
             description="Read all TurboPump controller settings and data")
    public String readAll()
    {
        String table = "Read all turbo-pump settings and data\n" +"\n";

        /* Boolean commands */

        TwisTorr84.CmndBool cmndB[] = TwisTorr84.CmndBool.values();
        int nB = cmndB.length;
        for (int i = 0; i < nB; i++) {
            table += String.format("\n   %-22s", cmndB[i]);
            try {
                boolean respB = turbo.readBool(cmndB[i]);
                table += Boolean.toString(respB);
            } catch (DriverException ex) {
                table += ex.getMessage();
            }
        }
        table += "\n";

        /* Numeric commands */

        TwisTorr84.CmndNumeric cmndN[] = TwisTorr84.CmndNumeric.values();
        int nN = cmndN.length;
        for (int i = 0; i < nN; i++) {
            table += String.format("\n   %-22s", cmndN[i]);
            try {
                int respN = turbo.readNumeric(cmndN[i]);
                table += Integer.toString(respN);
            } catch (DriverException ex) {
                table += ex.getMessage();
            }
        }
        table += "\n";

        TwisTorr84.CmndAlpha cmndA[] = TwisTorr84.CmndAlpha.values();
        int nA = cmndA.length;
        for (int i = 0; i < nA; i++) {
            table += String.format("\n   %-22s", cmndA[i]);
            try {
                String respA = turbo.readAlpha(cmndA[i]);
                table += respA;
            } catch (DriverException ex) {
                table += ex.getMessage();
            }
        }

        return table;
    }
}
