package org.lsst.ccs.subsystem.common.devices.pluto;

import java.util.HashMap;
import java.util.Map;
import java.util.logging.Logger;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.drivers.pluto.Pluto;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.subsystem.common.ErrorUtils;

/**
 *  Handles a Pluto PLC system control channel.
 *
 *  @author Owen Saxton
 */
public class PlutoControl implements HasLifecycle {

   /**
    *  Constants.
    */
    private final static int
        TYPE_BIT  = 0,
        TYPE_REG  = 1;

   /**
    *  Private lookup maps.
    */
    private final static Map<String, Integer> typeMap = new HashMap<>();
    static {
        typeMap.put("BIT", TYPE_BIT);
        typeMap.put("REG", TYPE_REG);
    }

   /**
    *  Data fields.
    */
    @LookupName
    protected String name;
    @LookupField(strategy=LookupField.Strategy.ANCESTORS)
    protected PlutoDevice pDevc;

    protected int hwChan = 0;        // HW channel number
    protected String type;           // type:reg
    protected double offset = 0.0;   // Conversion offset
    protected double scale = 1.0;    // Conversion scale

    private static final Logger LOG = Logger.getLogger(PlutoControl.class.getName());
    private int iType, area;

    @Override
    public void init()
    {
        if (type == null) {
            ErrorUtils.reportConfigError(LOG, name, "type", "not specified");
        }
        Integer iTyp = null;
        String[] typeFields = type.split(":", -1);
        if (typeFields.length == 2) {
            iTyp = typeMap.get(typeFields[0].toUpperCase());
        }
        try {
            area = Integer.decode(typeFields[1]);
        }
        catch (NumberFormatException e) {
            area = -1;
        }
        if (iTyp == null || area < 0 || area >= Pluto.NUM_DTP_AREAS) {
            ErrorUtils.reportConfigError(LOG, name, "type", "has invalid value");
        }
        iType = iTyp;
        int maxChan = iType == TYPE_BIT ? 16 : 2;
        if (hwChan >= maxChan) {
            ErrorUtils.reportConfigError(LOG, name, "hwchan", "has invalid value");
        }
    }


   /**
    *  Writes a value.
    *
    *  @param  value  The value
    */
    public void writeValue(double value)
    {
        int ival = (int)((value - offset) / scale);
        if (iType == TYPE_BIT) {
            pDevc.writeBit(area, hwChan, ival);
        }
        else {
            pDevc.writeRegister(area, hwChan, ival);
        }
    }

}
