package org.lsst.ccs.subsystem.common.devices.dataforth;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.dataforth.Maq20;
import org.lsst.ccs.drivers.dataforth.Maq20Analog;
import org.lsst.ccs.drivers.dataforth.Maq20AnalogIn;
import org.lsst.ccs.drivers.dataforth.Maq20AnalogOut;
import org.lsst.ccs.drivers.dataforth.Maq20Discrete;
import org.lsst.ccs.drivers.dataforth.Maq20DiscreteFreq;
import org.lsst.ccs.drivers.dataforth.Maq20DiscreteIn;
import org.lsst.ccs.drivers.dataforth.Maq20DiscreteOut;
import org.lsst.ccs.drivers.dataforth.Maq20DiscretePWM;
import org.lsst.ccs.drivers.dataforth.Maq20DiscretePulse;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.subsystem.common.ErrorUtils;

/**
 *  Handles a Dataforth Maq20 DAQ system.
 *
 *  @author Owen Saxton
 */
public class Maq20Device extends Device {

   /**
    *  Inner class for holding static module type constants.
    */
    public static class ModuleDef {

        final Maq20.ModuleType type;
        final int opType;
        final int numChan;
        final Map rangeMap;
        final int index;

        ModuleDef(Maq20.ModuleType type, int numChan, Map rangeMap, int index) {
            this.type = type;
            this.opType = Maq20.getModuleOpType(type);
            this.numChan = numChan;
            this.rangeMap = rangeMap;
            this.index = index;
        }

    }

    /**
     *  Inner class for holding dynamic module data.
     */
    public static class ModuleData {

        public String serial;                    // Serial number
        public int modId;                        // Module ID
        public int[] discFunc = new int[2];      // Discrete functions
        public ModuleDef modDef;                 // Module definition data
        public Maq20Analog maqAna;               // Analog module object
        public Maq20AnalogIn maqAnaIn;           // Analog in module object
        public Maq20AnalogOut maqAnaOut;         // Analog out module object
        public Maq20Discrete maqDisc;            // Discrete module object
        public Maq20DiscreteFreq maqDiscFreq;    // Discrete frequency module object
        public Maq20DiscretePulse maqDiscPulse;  // Discrete pulse module object
        public Maq20DiscretePWM maqDiscPWM;;     // Discrete PWM module object
        public Maq20DiscreteIn maqDiscIn;        // Discrete input module object
        public Maq20DiscreteOut maqDiscOut;      // Discrete output module object

    }

    /**
     *  Inner control interface.
     */
    protected static interface Maq20Control {

        void initialize() throws DriverException;

        void close();

    }

    /**
     *  Public constants
     */
    // Hardware (& software) channel counts
    public static final int
        NUM_CHAN_TC = 8,
        NUM_CHAN_RTD = 6,
        NUM_CHAN_IVS = 16,
        NUM_CHAN_IVD = 8,
        NUM_CHAN_IVO = 8,
        NUM_CHAN_DIOL = 5,
        NUM_CHAN_DIOH = 4,
        NUM_CHAN_PULSE = 3,
        NUM_CHAN_PWM = 3,
        NUM_CHAN_FREQ = 1,
        NUM_CHAN_DINP = 20,
        NUM_CHAN_DOUT = 20;

    // Discrete function channel numbers
    public static final int
        CHAN_FREQUENCY = 0,
        CHAN_PULSE_COUNT = 1,
        CHAN_PULSE_RPM = 2,
        CHAN_PWM_PERIOD = 0,
        CHAN_PWM_LOW1 = 1,
        CHAN_PWM_LOW2 = 2;

    // Discrete module functions
    public static final int
        DISC_FUNC_NONE = -1,
        DISC_FUNC_PULSE = 0,
        DISC_FUNC_PWM = 1,
        DISC_FUNC_FREQ = 2;

    /**
    *  Private constants.
    */
    private static final int
        CHAN_TYPE_ANAL_IN = 0,
        CHAN_TYPE_ANAL_OUT = 1,
        CHAN_TYPE_DISC_IN = 2,
        CHAN_TYPE_DISC_OUT = 3,
        CHAN_TYPE_DISC_PULSE = 4,
        CHAN_TYPE_DISC_PWM = 5,
        CHAN_TYPE_DISC_FREQ = 6;

    private static final Map<String, Integer> discFuncMap = new HashMap<>();
    static {
        discFuncMap.put("", DISC_FUNC_NONE);
        discFuncMap.put("PULSE", DISC_FUNC_PULSE);
        discFuncMap.put("PWM", DISC_FUNC_PWM);
        discFuncMap.put("FREQ", DISC_FUNC_FREQ);
    }
    private static final Map<String, Integer> dummyRangeMap = new HashMap<>();
    private static final Map<String, Integer> ttcRangeMap = new HashMap<>();
    static {
        ttcRangeMap.put("T400", Maq20Analog.RANGE_TC_T_400);
        ttcRangeMap.put("T220", Maq20Analog.RANGE_TC_T_220);
    }
    private static final Map<String, Integer> jtcRangeMap = new HashMap<>();
    static {
        jtcRangeMap.put("J760", Maq20Analog.RANGE_TC_J_760);
        jtcRangeMap.put("J393", Maq20Analog.RANGE_TC_J_393);
        jtcRangeMap.put("J199", Maq20Analog.RANGE_TC_J_199);
    }
    private static final Map<String, Integer> ktcRangeMap = new HashMap<>();
    static {
        ktcRangeMap.put("K1350", Maq20Analog.RANGE_TC_K_1350);
        ktcRangeMap.put("K651", Maq20Analog.RANGE_TC_K_651);
        ktcRangeMap.put("K332", Maq20Analog.RANGE_TC_K_332);
    }
    private static final Map<String, Integer> rstcRangeMap = new HashMap<>();
    static {
        rstcRangeMap.put("R1750", Maq20Analog.RANGE_TC_R_1750);
        rstcRangeMap.put("R990", Maq20Analog.RANGE_TC_R_990);
        rstcRangeMap.put("S1750", Maq20Analog.RANGE_TC_S_1750);
        rstcRangeMap.put("S970", Maq20Analog.RANGE_TC_S_970);
    }
    private static final Map<String, Integer> rtdRangeMap = new HashMap<>();
    static {
        rtdRangeMap.put("R850", Maq20Analog.RANGE_RTD_850);
        rtdRangeMap.put("R200", Maq20Analog.RANGE_RTD_200);
        rtdRangeMap.put("R100", Maq20Analog.RANGE_RTD_100);
    }
    private static final Map<String, Integer> voltRangeMap = new HashMap<>();
    static {
        voltRangeMap.put("V60", Maq20Analog.RANGE_VOLT_60);
        voltRangeMap.put("V40", Maq20Analog.RANGE_VOLT_40);
        voltRangeMap.put("V20", Maq20Analog.RANGE_VOLT_20);
        voltRangeMap.put("V10", Maq20Analog.RANGE_VOLT_10);
        voltRangeMap.put("V5", Maq20Analog.RANGE_VOLT_5);
    }
    private static final Map<String, Integer> mvoltRangeMap = new HashMap<>();
    static {
        mvoltRangeMap.put("V2", Maq20Analog.RANGE_MVOLT_2000);
        mvoltRangeMap.put("V1", Maq20Analog.RANGE_MVOLT_1000);
        mvoltRangeMap.put("MV250", Maq20Analog.RANGE_MVOLT_250);
        mvoltRangeMap.put("MV100", Maq20Analog.RANGE_MVOLT_100);
        mvoltRangeMap.put("MV50", Maq20Analog.RANGE_MVOLT_50);
    }
    private static final Map<String, Integer> ampRangeMap = new HashMap<>();
    static {
        ampRangeMap.put("MA0_20", Maq20Analog.RANGE_MAMP_0_20);
        ampRangeMap.put("MA4_20", Maq20Analog.RANGE_MAMP_4_20);
    }
    private static final Map<String, Integer> voutRangeMap = new HashMap<>();
    static {
        voutRangeMap.put("VPM10", Maq20Analog.RANGE_VOUT_PM10);
        voutRangeMap.put("VPM5", Maq20Analog.RANGE_VOUT_PM5);
        voutRangeMap.put("VPM2", Maq20Analog.RANGE_VOUT_PM2);
        voutRangeMap.put("VP10", Maq20Analog.RANGE_VOUT_P10);
        voutRangeMap.put("VP5", Maq20Analog.RANGE_VOUT_P5);
        voutRangeMap.put("VP2", Maq20Analog.RANGE_VOUT_P2);
    }
    private static final Map<String, ModuleDef> typeMap = new HashMap<>();
    private static final List<Maq20.ModuleType> typeList = new ArrayList<>();
    static {
        Maq20.ModuleType type;
        type = Maq20.ModuleType.JTC;
        typeMap.put(type.name(), new ModuleDef(type, NUM_CHAN_TC, jtcRangeMap, typeList.size()));
        typeList.add(type);
        type = Maq20.ModuleType.KTC;
        typeMap.put(type.name(), new ModuleDef(type, NUM_CHAN_TC, ktcRangeMap, typeList.size()));
        typeList.add(type);
        type = Maq20.ModuleType.TTC;
        typeMap.put(type.name(), new ModuleDef(type, NUM_CHAN_TC, ttcRangeMap, typeList.size()));
        typeList.add(type);
        type = Maq20.ModuleType.RSTC;
        typeMap.put(type.name(), new ModuleDef(type, NUM_CHAN_TC, rstcRangeMap, typeList.size()));
        typeList.add(type);
        type = Maq20.ModuleType.RTD;
        typeMap.put(type.name(), new ModuleDef(type, NUM_CHAN_RTD, rtdRangeMap, typeList.size()));
        typeList.add(type);
        type = Maq20.ModuleType.VD;
        typeMap.put(type.name(), new ModuleDef(type, NUM_CHAN_IVD, voltRangeMap, typeList.size()));
        typeList.add(type);
        type = Maq20.ModuleType.VS;
        typeMap.put(type.name(), new ModuleDef(type, NUM_CHAN_IVS, voltRangeMap, typeList.size()));
        typeList.add(type);
        type = Maq20.ModuleType.VO;
        typeMap.put(type.name(), new ModuleDef(type, NUM_CHAN_IVO, voutRangeMap, typeList.size()));
        typeList.add(type);
        type = Maq20.ModuleType.MVD;
        typeMap.put(type.name(), new ModuleDef(type, NUM_CHAN_IVD, mvoltRangeMap, typeList.size()));
        typeList.add(type);
        type = Maq20.ModuleType.ID;
        typeMap.put(type.name(), new ModuleDef(type, NUM_CHAN_IVD, ampRangeMap, typeList.size()));
        typeList.add(type);
        type = Maq20.ModuleType.IS;
        typeMap.put(type.name(), new ModuleDef(type, NUM_CHAN_IVS, ampRangeMap, typeList.size()));
        typeList.add(type);
        type = Maq20.ModuleType.IO;
        typeMap.put(type.name(), new ModuleDef(type, NUM_CHAN_IVO, ampRangeMap, typeList.size()));
        typeList.add(type);
        type = Maq20.ModuleType.DIOH;
        typeMap.put(type.name(), new ModuleDef(type, NUM_CHAN_DIOH, dummyRangeMap, typeList.size()));
        typeList.add(type);
        type = Maq20.ModuleType.DIOL;
        typeMap.put(type.name(), new ModuleDef(type, NUM_CHAN_DIOL, dummyRangeMap, typeList.size()));
        typeList.add(type);
        type = Maq20.ModuleType.DINP;
        typeMap.put(type.name(), new ModuleDef(type, NUM_CHAN_DINP, dummyRangeMap, typeList.size()));
        typeList.add(type);
        type = Maq20.ModuleType.DOUT;
        typeMap.put(type.name(), new ModuleDef(type, NUM_CHAN_DOUT, dummyRangeMap, typeList.size()));
        typeList.add(type);
    }

   /**
    *  Data fields.
    */
    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    protected final Map<String, Maq20Control> controlMap = new HashMap<>();

    @ConfigurationParameter(category="Device", isFinal=true)
    protected volatile String node;     // Network node name
    @ConfigurationParameter(category="Device", maxLength=Maq20.NUM_MODULES, isFinal=true)
    protected volatile String[] serials;  // Serial numbers of DAQ modules

    protected String[] modules;  // Types of DAQ modules

    private static final Logger LOG = Logger.getLogger(Maq20Device.class.getName());
    protected final Maq20 maq = new Maq20();  // Associated Maq20 object
    private ModuleData[] modData;


   /**
    *  Performs configuration.
    */
    @Override
    protected void initDevice()
    {
        if (node == null) {
            ErrorUtils.reportConfigError(LOG, name, "node", "is missing");
        }
        if (modules == null) {
            ErrorUtils.reportConfigError(LOG, name, "modules", "is missing");
        }
        if (serials == null) {
            ErrorUtils.reportConfigError(LOG, name, "serials", "is missing");
        }
        if (modules.length != serials.length) {
            ErrorUtils.reportConfigError(LOG, name, "modules and serials", "have different lengths");
        }
        modData = new ModuleData[modules.length];
        for (int j = 0; j < modules.length; j++) {
            ModuleData mData = null;
            String[] words = modules[j].split(":");
            if (words.length >= 1 && words.length <= 3) {
                ModuleDef mDef = typeMap.get(words[0].toUpperCase());
                if (mDef != null) {
                    mData = new ModuleData();
                    mData.serial = serials[j];
                    mData.modDef = mDef;
                    if (mDef.opType == Maq20.OPER_DISCRETE) {
                        mData.discFunc[0] = DISC_FUNC_NONE;
                        mData.discFunc[1] = DISC_FUNC_NONE;
                        for (int k = 0; k < words.length - 1; k++) {
                            Integer func = discFuncMap.get(words[k + 1].trim().toUpperCase());
                            if (func != null) {
                                mData.discFunc[k] = func;
                            }
                            else {
                                mData = null;
                                break;
                            }
                        }
                    }
                    else {
                        if (words.length != 1) {
                            mData = null;
                        }
                    }
                }
            }
            if (mData == null) {
                ErrorUtils.reportConfigError(LOG, name, "modules", "has invalid item (" + modules[j] +")");
            }
            modData[j] = mData;
        }
        fullName = "Maq20 system (" + node + ")";
    }


   /**
    *  Performs full initialization.
    */
    @Override
    protected void initialize()
    {
        try {
            maq.open(Maq20.ConnType.NET, node);
            for (ModuleData mData : modData) {
                if (mData == null) continue;
                mData.modId = maq.getModuleId(mData.serial);
                if (mData.modId <= 0) {
                    LOG.log(Level.SEVERE, "Module with serial number {0} not found", mData.serial);
                }
                else {
                    Maq20.ModuleType type = maq.getModuleType(mData.modId);
                    if (type != mData.modDef.type) {
                        LOG.log(Level.SEVERE, "Module with serial number {0} has the wrong type ({1})",
                                new Object[]{mData.serial, type});
                        mData.modId = -1;
                    }
                    else {
                        switch (mData.modDef.opType) {
                        case Maq20.OPER_ANALOG:
                            mData.maqAna = mData.maqAnaIn = maq.getAnalIn(mData.modId);
                            break;
                        case Maq20.OPER_ANALOUT:
                            mData.maqAna = mData.maqAnaOut = maq.getAnalOut(mData.modId);
                            break;
                        case Maq20.OPER_DISCRETE:
                            mData.maqDisc = maq.getDiscrete(mData.modId);
                            mData.maqDiscPulse = maq.getDiscPulse(mData.modId);
                            mData.maqDiscPWM = maq.getDiscPWM(mData.modId);
                            mData.maqDiscFreq = maq.getDiscFreq(mData.modId);
                            break;
                        case Maq20.OPER_DISCIN:
                            mData.maqDisc = maq.getDiscIn(mData.modId);
                            mData.maqDiscIn = maq.getDiscIn(mData.modId);
                            break;
                        case Maq20.OPER_DISCOUT:
                            mData.maqDisc = maq.getDiscOut(mData.modId);
                            mData.maqDiscIn = maq.getDiscOut(mData.modId);
                            mData.maqDiscOut = maq.getDiscOut(mData.modId);
                            break;
                        }
                    }
                }
            }
            for (Maq20Control ctrl : controlMap.values()) {
                ctrl.initialize();
            }
            setOnline(true);
            initSensors();
            setOutputLines();
            LOG.log(Level.INFO, "Connected to {0}", fullName);
        }
        catch (DriverException e) {
            if (!inited) {
                LOG.log(Level.SEVERE, "Error connecting to {0}: {1}", new Object[]{fullName, e});
            }
            close();
        }
        inited = true;
    }


   /**
    *  Closes the connection.
    */
    @Override
    protected void close()
    {
        for (Maq20Control ctrl : controlMap.values()) {
            ctrl.close();
        }
        try {
            maq.close();
        }
        catch (DriverException e) {
        }
        for (ModuleData mData : modData) {
            if (mData == null) continue;
            mData.maqAna = null;
            mData.maqAnaIn = null;
            mData.maqAnaOut = null;
            mData.maqDisc = null;
            mData.maqDiscFreq = null;
            mData.maqDiscPWM = null;
            mData.maqDiscPulse = null;
        }
    }


   /**
    *  Checks a channel's parameters for validity.
    *
    *  @param  name     The name of the channel.
    *  @param  hwChan   The hardware channel number.
    *  @param  type     The channel type string.
    *  @param  subtype  The channel subtype string.
    *  @return  A two-element array containing the encoded type [0] and
    *           subtype [1] values.
    *  @throws  Exception if any errors found in the parameters.
    */
    @Override
    protected int[] checkChannel(String name, int hwChan, String type, String subtype) throws Exception
    {
        int modIx = 0, chanType = -1, funcIx = 0;
        Integer iRange = 0;
        ModuleData mData = null;
        try {
            modIx = Integer.decode(type);
            if (modIx >= 0 && modIx < modData.length) {
                mData = modData[modIx];
            }
        }
        catch (NumberFormatException e) {
        }
        if (mData != null) {
            boolean subtypeOK = true;
            int numChan = mData.modDef.numChan;
            if (mData.modDef.opType == Maq20.OPER_DISCRETE) {
                if (subtype.isEmpty()) {
                    chanType = CHAN_TYPE_DISC_IN;
                }
                else if (subtype.toUpperCase().equals("OUT")) {
                    chanType = CHAN_TYPE_DISC_OUT;
                }
                else {
                    funcIx = subtype.equals("0") ? 0 : subtype.equals("1") ? 1 : -1;
                    if (funcIx >= 0) {
                        int discFunc = mData.discFunc[funcIx];
                        if (discFunc == DISC_FUNC_PULSE) {
                            chanType = CHAN_TYPE_DISC_PULSE;
                            numChan = NUM_CHAN_PULSE;
                        }
                        else if (discFunc == DISC_FUNC_PWM) {
                            chanType = CHAN_TYPE_DISC_PWM;
                            numChan = NUM_CHAN_PWM;
                        }
                        else if (discFunc == DISC_FUNC_FREQ) {
                            chanType = CHAN_TYPE_DISC_FREQ;
                            numChan = NUM_CHAN_FREQ;
                        }
                    }
                    else {
                        subtypeOK = false;
                    }
                }
            }
            else if (mData.modDef.opType == Maq20.OPER_ANALOG) {
                if (subtype == null || subtype.isEmpty()) {
                    LOG.log(Level.INFO, "No subtype (range) specified for {0}: using default", name);
                }
                else {
                    Map<String, Integer> rMap = mData.modDef.rangeMap;
                    iRange = rMap.get(subtype.toUpperCase());
                    subtypeOK = iRange != null;
                }
                chanType = CHAN_TYPE_ANAL_IN;
            }
            else {
                chanType = CHAN_TYPE_ANAL_OUT;
            }
            if (!subtypeOK) {
                ErrorUtils.reportChannelError(LOG, name, "subtype (range)", subtype);
            }
            if (hwChan < 0 || hwChan >= numChan) {
                ErrorUtils.reportChannelError(LOG, name, "hwChan", hwChan);
            }
        }
        if (chanType < 0) {
            ErrorUtils.reportChannelError(LOG, name, "type (index)", type);
        }

        return new int[]{(funcIx << 16) | (chanType << 8) | modIx, iRange};
    }


   /**
    *  Initializes a channel.
    *
    *  Much of the testing that is often done in checkChannel() has to be
    *  done here because the device has to be opened first.
    *
    *  @param  name     The channel name
    *  @param  id       The channel ID
    *  @param  hwChan   The hardware channel number.
    *  @param  type     The encoded channel type returned by checkChannel.
    *  @param  subtype  The channel subtype returned by checkChannel.
    */
    @Override
    protected void initChannel(String name, int id, int hwChan, int type, int subtype)
    {
        if (!online) return;
        int modIx = type & 0xff, chanType = type >> 8, range = subtype;
        ModuleData mData = modData[modIx];
        try {
            if (mData.modId <= 0) {
                LOG.log(Level.SEVERE, "Channel {0} initialization error: module failed initialization", name);
                dropChannel(id);
            }
            else if (chanType == CHAN_TYPE_ANAL_IN) {
                mData.maqAnaIn.enable(hwChan, true);
                mData.maqAnaIn.setRange(hwChan, range);
            }
        }
        catch (DriverException e) {
            LOG.log(Level.SEVERE, "Error initializing {0} channel: {1}", new Object[]{name, e});
            setOnline(false);
        }
    }


   /**
    *  Reads a channel.
    *
    *  @param  hwChan   The hardware channel number.
    *  @param  type     The encoded channel type returned by checkChannel.
    *  @return  Channel value
    */
    @Override
    protected double readChannel(int hwChan, int type)
    {
        double value = super.readChannel(hwChan, type);
        if (online) {
            ModuleData mData = modData[type & 0xff];
            int funcNum = type >> 16;
            try {
                switch ((type >> 8) & 0xff) {
                case CHAN_TYPE_ANAL_IN:
                case CHAN_TYPE_ANAL_OUT:
                    value = mData.maqAna.readValue(hwChan);
                    break;
                case CHAN_TYPE_DISC_IN:
                    value = mData.maqDisc.readDiscIn(hwChan);
                    break;
                case CHAN_TYPE_DISC_OUT:
                    value = mData.maqDisc.readDiscOut(hwChan);
                    break;
                case CHAN_TYPE_DISC_PULSE:
                    switch (hwChan) {
                    case CHAN_FREQUENCY:
                        value = mData.maqDiscPulse.readFrequency(funcNum);
                        break;
                    case CHAN_PULSE_COUNT:
                        value = mData.maqDiscPulse.readPulseCount(funcNum);
                        break;
                    case CHAN_PULSE_RPM:
                        //value = mData.maqDiscPulse.readRPM(funcNum);  // Loses precision
                        value = 60.0 * mData.maqDiscPulse.readFrequency(funcNum) / mData.maqDiscPulse.getPulsesPerRevn(funcNum);
                        break;
                    }
                    break;
                case CHAN_TYPE_DISC_PWM:
                    switch (hwChan) {
                    case CHAN_PWM_PERIOD:
                        value = mData.maqDiscPWM.getPeriod(funcNum);
                        break;
                    case CHAN_PWM_LOW1:
                        value = mData.maqDiscPWM.getLowTime1(funcNum);
                        break;
                    case CHAN_PWM_LOW2:
                        value = mData.maqDiscPWM.getLowTime2(funcNum);
                        break;
                    }
                    break;
                case CHAN_TYPE_DISC_FREQ:
                    value = mData.maqDiscFreq.getFrequency(funcNum);
                    break;
                }
            }
            catch (DriverException e) {
                LOG.log(Level.SEVERE, "Error reading {0} channel: {1}", new Object[]{name, e});
                setOnline(false);
            }
        }
        return value;
    }


    /**
     *  Gets the number of defined modules.
     *
     *  @return  The number of modules
     */
    public int getModuleCount()
    {
        return modData.length;
    }


    /**
     *  Gets the module data with the specified index.
     *
     *  @param  index  The module index
     *  @return  The module data, or null if index is invalid
     */
    public ModuleData getModuleData(int index)
    {
        return (index >= 0 && index < modData.length) ? modData[index] : null;
    }

}
