package org.lsst.ccs.subsystem.common.devices.dataforth;

import java.util.logging.Logger;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.dataforth.Maq20;
import org.lsst.ccs.drivers.dataforth.Maq20AnalogOut;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.subsystem.common.ErrorUtils;

/**
 *  Handles a Dataforth Maq20 system analog output module.
 *
 *  @author Owen Saxton
 */
public class Maq20AnalogControl implements HasLifecycle, Maq20Device.Maq20Control {

    @LookupName
    protected String name;
    @LookupField(strategy=LookupField.Strategy.ANCESTORS)
    protected Maq20Device maqDevc;

    private Integer modIndex;  // The module index in the defined set of modules
    private String[] ranges;   // Ranges for the channels (up to 8)

    private static final Logger LOG = Logger.getLogger(Maq20AnalogControl.class.getName());
    private Maq20Device.ModuleData modData;
    private Integer[] rangeValues;


    /**
     *  Life-cycle initialization.
     * 
     *  Checks the configuration
     */
    @Override
    public void init()
    {
        if (modIndex == null) {
            ErrorUtils.reportConfigError(LOG, name, "modIndex", "not specified");
        }
        modData = maqDevc.getModuleData(modIndex);
        if (modData == null) {
            ErrorUtils.reportConfigError(LOG, name, "modIndex (" + modIndex + ")", "specifies non-existent module");
        }
        Maq20Device.ModuleDef modDef = modData.modDef;
        if (modDef.opType != Maq20.OPER_ANALOUT) {
            ErrorUtils.reportConfigError(LOG, name, "modIndex (" + modIndex + ")", "specifies non-analog output module");
        }
        if (ranges != null && ranges.length > modDef.numChan) {
            ErrorUtils.reportConfigError(LOG, name, "ranges", "has more than " + modDef.numChan + " elements");
        }
        rangeValues = new Integer[modDef.numChan];
        if (ranges != null) {
            for (int chan = 0; chan < ranges.length; chan++) {
                String range = ranges[chan];
                if (range == null || range.trim().isEmpty()) continue;
                Integer rangeValue = (Integer)modDef.rangeMap.get(range);
                if (rangeValue == null) {
                    ErrorUtils.reportConfigError(LOG, name, "range " + range, "not valid for " + modDef.type + " module");
                }
                rangeValues[chan] = rangeValue;
            }
        }
    }


    /**
     *  Post-connection initialization.
     *
     *  @throws  DriverException
     */
    @Override
    public void initialize() throws DriverException
    {
        Maq20AnalogOut anaOut = getAnaOut();
        for (int chan = 0; chan < modData.modDef.numChan; chan++) {
            if (rangeValues[chan] != null) {
                anaOut.setRange(chan, rangeValues[chan]);
            }
        }
    }


    /**
     *  Pre-closure cleanup.
     */
    @Override
    public void close()
    {
    }


    /**
     *  Writes an analog value.
     *
     *  @param  chan   The channel number
     *  @param  value  The value
     *  @throws  DriverException
     */
    public void writeValue(int chan, double value) throws DriverException
    {
        getAnaOut().writeValue(chan, value);
    }


    /**
     *  Reads an analog value.
     *
     *  @param  chan   The channel number
     *  @return  The value
     *  @throws  DriverException
     */
    public double readValue(int chan) throws DriverException
    {
        return getAnaOut().readValue(chan);
    }


    /**
     *  Gets the module index.
     *
     *  @return  The module index
     */
    public int getModIndex()
    {
        return modIndex == null ? -1 : modIndex;
    }


    /**
     *  Gets the analog output object.
     *
     *  @return  The analog output object
     *  @throws  DriverException  if connection not open
     */
    private Maq20AnalogOut getAnaOut() throws DriverException
    {
        if (modData.maqAnaOut == null) {
            throw new DriverException("Connection not open");
        }
        return modData.maqAnaOut;
    }

}
