package org.lsst.ccs.subsystem.common.devices.dataforth;

import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.dataforth.Maq20;
import org.lsst.ccs.drivers.dataforth.Maq20Discrete;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.subsystem.common.ErrorUtils;

/**
 *  Handles a Dataforth Maq20 system discrete output module.
 *
 *  @author Owen Saxton
 */
public class Maq20DiscControl implements HasLifecycle {

    /**
     *  Data fields.
     */
    @LookupName
    protected String name;
    @LookupField(strategy=LookupField.Strategy.ANCESTORS)
    protected Maq20Device maqDevc;

    private Integer modIndex;  // Module index

    private static final Logger LOG = Logger.getLogger(Maq20DiscControl.class.getName());
    private Maq20Device.ModuleData modData;
    private boolean haveError = false;


    /**
     *  Life-cycle initialization.
     * 
     *  Checks the configuration
     */
    @Override
    public void init()
    {
        if (modIndex == null) {
            ErrorUtils.reportConfigError(LOG, name, "modIndex", "not specified");
        }
        modData = maqDevc.getModuleData(modIndex);
        if (modData == null) {
            ErrorUtils.reportConfigError(LOG, name, "modIndex (" + modIndex + ")", "specifies non-existent module");
        }
        if (modData.modDef.opType != Maq20.OPER_DISCRETE) {
            ErrorUtils.reportConfigError(LOG, name, "modIndex (" + modIndex + ")", "specifies non-discrete module");
        }
    }


    /**
     *  Sets the state of an output line.
     *
     *  These lines use inverted logic: 0 = on, 1 = off
     *
     *  @param  chan  The channel number
     *  @param  on    Whether to set on
     *  @throws  DriverException
     */
    public void setLineOn(int chan, boolean on) throws DriverException
    {
        getDisc().writeDisc(chan, on ? 0 : 1);
    }


    /**
     *  Gets the state of an output line.
     *
     *  @param  chan  The channel number
     *  @return  Whether line is on
     */
    public Boolean isLineOn(int chan)
    {
        Boolean on = null;
        try {
            on = getDisc().readDiscOut(chan) == 0;
            haveError = false;
        }
        catch (DriverException e) {
            if (!haveError) {
                LOG.log(Level.SEVERE, "Error reading channel {0} state: {1}", new Object[]{chan, e});
                haveError = true;
            }
        }
        return on;
    }


    /**
     *  Gets discrete I/O object.
     *
     *  @return  Discrete I/O object
     *  @throws  DriverException  if connection not open
     */
    private Maq20Discrete getDisc() throws DriverException
    {
        if (modData.maqDisc == null) {
            throw new DriverException("Connection not open");
        }
        return modData.maqDisc;
    }

}
