package org.lsst.ccs.subsystem.common.devices;

import java.util.logging.Logger;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.ConfigurationParameterChanger;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.dataforth.Maq20DiscretePulse;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.subsystem.common.ErrorUtils;

/**
 *  Handles a Dataforth Maq20 system pulse control channel.
 *
 *  @author Owen Saxton
 */
public class Maq20PulseControl implements HasLifecycle, Maq20Device.Maq20DiscControl {

    /**
     *  Data fields.
     */
    @LookupName
    private String name;
    @LookupField(strategy=LookupField.Strategy.ANCESTORS)
    private Maq20Device maqDevc;

    @ConfigurationParameter
    private int pulsesPerRevn = 1;

    private String type;

    private static final Logger LOG = Logger.getLogger(Maq20PulseControl.class.getName());
    private Maq20Device.ModuleData modData;
    private Maq20DiscretePulse pulse;
    private int funcNum;
    private boolean inited = false;

    @Override
    public void init()
    {
        if (type == null) {
            ErrorUtils.reportConfigError(LOG, name, "type", "not specified");
        }
        String[] words = type.split(":");
        if (words.length != 2) {
            ErrorUtils.reportConfigError(LOG, name, "type", "doesn't have exactly two elements");
        }
        try {
            int modIx = Integer.decode(words[0]);
            modData = maqDevc.getModuleData(modIx);
            if (modData == null) {
                ErrorUtils.reportConfigError(LOG, name, "module " + modIx, "doesn't exist");
            }
        }
        catch (NumberFormatException ex) {
            ErrorUtils.reportConfigError(LOG, name, "module index (" + words[0] + ")", "is not numeric");
        }
        if (!words[1].equals("0") && !words[1].equals("1")) {
            ErrorUtils.reportConfigError(LOG, name, "function number (" + words[1] + ")", "is not 0 or 1");
        }
        funcNum = Integer.decode(words[1]);
    }


    /**
     *  Performs initialization.
     *
     *  @throws DriverException
     */
    @Override
    public void initialize() throws DriverException
    {
        pulse = modData.maqDiscPulse;
        inited = true;
        checkPulse();
        pulse.enable(funcNum);
        setPulsesPerRevn(pulsesPerRevn);
    }


    /**
     *  Setter for the pulses/revolution.
     *
     *  @param  pulses  The number of pulses per revolution
     *  @throws DriverException
     */
    @ConfigurationParameterChanger
    public void setPulsesPerRevn(int pulses) throws DriverException
    {
        pulsesPerRevn = pulses;
        if (inited) {
            checkPulse();
            pulse.setPulsesPerRevn(funcNum, pulsesPerRevn);
            pulse.armFunction(funcNum);
        }
    }


    /**
     *  Getter for pulses/revolution.
     *
     *  @return  The number of pulses/revn
     */
    @Command(description = "Get number of pulses per revolution")
    public int getPulsesPerRevn()
    {
        return pulsesPerRevn;
    }


    /**
     *  Checks whether pulse operations are accessible.
     *
     *  @throws DriverException
     */
    private void checkPulse() throws DriverException
    {
        if (pulse == null) {
            throw new DriverException("Cannot access pulse function");
        }
    }

}
