package org.lsst.ccs.subsystem.common.devices;

import java.util.logging.Logger;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.ConfigurationParameterChanger;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.dataforth.Maq20DiscretePWM;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.subsystem.common.ErrorUtils;

/**
 *  Handles a Dataforth Maq20 system PWM control channel.
 *
 *  @author Owen Saxton
 */
public class Maq20PWMControl implements HasLifecycle, Maq20Device.Maq20DiscControl {

    /**
     *  Data fields.
     */
    @LookupName
    private String name;
    @LookupField(strategy=LookupField.Strategy.ANCESTORS)
    private Maq20Device maqDevc;

    @ConfigurationParameter
    private double frequency = 1000.0;
    @ConfigurationParameter
    private double dutyCycle1 = 0.0;
    @ConfigurationParameter
    private double dutyCycle2 = 0.0;

    private String type;

    private static final Logger LOG = Logger.getLogger(Maq20PWMControl.class.getName());
    private Maq20Device.ModuleData modData;
    private Maq20DiscretePWM pwm;
    private int funcNum;
    private boolean inited = false;


    @Override
    public void init() {
        if (type == null) {
            ErrorUtils.reportConfigError(LOG, name, "type", "not specified");
        }
        String[] words = type.split(":");
        if (words.length != 2) {
            ErrorUtils.reportConfigError(LOG, name, "type", "doesn't have exactly two elements");
        }
        try {
            int modIx = Integer.decode(words[0]);
            modData = maqDevc.getModuleData(modIx);
            if (modData == null) {
                ErrorUtils.reportConfigError(LOG, name, "module " + modIx, "doesn't exist");
            }
        }
        catch (NumberFormatException ex) {
            ErrorUtils.reportConfigError(LOG, name, "module index (" + words[0] + ")", "is not numeric");
        }
        if (!words[1].equals("0") && !words[1].equals("1")) {
            ErrorUtils.reportConfigError(LOG, name, "function number (" + words[1] + ")", "is not 0 or 1");
        }
        funcNum = Integer.decode(words[1]);
    }


    /**
     *  Performs initialization.
     *
     *  @throws DriverException
     */
    @Override
    public void initialize() throws DriverException
    {
        pwm = modData.maqDiscPWM;
        inited = true;
        checkPWM();
        if (pwm.isEnabled(funcNum)) {
            fetch();
        }
        else {
            pwm.enable(funcNum);
        }
        load();
    }


    /**
     *  Setter for frequency.
     *
     *  @param  freq  The frequency
     *  @throws DriverException
     */
    @ConfigurationParameterChanger
    public void setFrequency(double freq) throws DriverException
    {
        frequency = freq > 0.0 && freq <= 1000.0 ? freq : 1000.0;
        if (inited) {
            load();
        }
    }


    /**
     *  Getter for frequency.
     *
     *  @return  The frequency
     */
    @Command(description = "Get frequency")
    public double getFrequency()
    {
        return frequency;
    }


    /**
     *  Setter for duty cycle 1
     *
     *  @param  duty  The duty cycle (0 - 1)
     *  @throws DriverException
     */
    @ConfigurationParameterChanger
    public void setDutyCycle1(double duty) throws DriverException
    {
        dutyCycle1 = Math.min(Math.max(duty, 0.0), 1.0);
        if (inited) {
            checkPWM();
            int period = (int)(1_000_000 / frequency);
            pwm.setLowTime1(funcNum, (int)(period * (1.0 - dutyCycle1)));
            pwm.armFunction(funcNum);
        }
    }


    /**
     *  Getter for duty cycle 1
     *
     *  @return  The duty cycle (0 - 1)
     */
    @Command(description = "Get duty cycle 1")
    public double getDutyCycle1()
    {
        return dutyCycle1;
    }


    /**
     *  Setter for duty cycle 2
     *
     *  @param  duty  The duty cycle (0 - 1)
     *  @throws DriverException
     */
    @ConfigurationParameterChanger
    public void setDutyCycle2(double duty) throws DriverException
    {
        dutyCycle2 = Math.min(Math.max(duty, 0.0), 1.0);
        if (inited) {
            checkPWM();
            int period = (int)(1_000_000 / frequency);
            pwm.setLowTime2(funcNum, (int)(period * (1.0 - dutyCycle2)));
            pwm.enableOutput2(funcNum, dutyCycle2 != 0.0);
            pwm.armFunction(funcNum);
        }
    }


    /**
     *  Getter for duty cycle 2
     *
     *  @return  The duty cycle (0 - 1)
     */
    @Command(description = "Get duty cycle 2")
    public double getDutyCycle2()
    {
        return dutyCycle2;
    }


    /**
     *  Loads the registers.
     *
     *  @throws DriverException
     */
    private void load() throws DriverException
    {
        checkPWM();
        pwm.setTimebase(funcNum, Maq20DiscretePWM.TIMEBASE_USECS);
        double period = 1_000_000 / frequency;
        pwm.setPeriod(funcNum, (int)period);
        pwm.setLowTime1(funcNum, (int)(period * (1.0 - dutyCycle1)));
        pwm.setLowTime2(funcNum, (int)(period * (1.0 - dutyCycle2)));
        pwm.enableOutput2(funcNum, dutyCycle2 != 0.0);
        pwm.armFunction(funcNum);
    }


    /**
     *  Fetches the registers.
     *
     *  @throws DriverException
     */
    private void fetch() throws DriverException
    {
        checkPWM();
        int timeBase = pwm.getTimebase(funcNum);
        int mult = timeBase == Maq20DiscretePWM.TIMEBASE_USECS ? 1
                     : timeBase == Maq20DiscretePWM.TIMEBASE_MSECS ? 1000 : 1_000_000;
        double period = mult * pwm.getPeriod(funcNum);
        setDutyCycle1(1.0 - mult * pwm.getLowTime1(funcNum) / period);
        setDutyCycle2(1.0 - mult * pwm.getLowTime2(funcNum) / period);
    }


    /**
     *  Checks whether PWM operations are accessible.
     *
     *  @throws DriverException
     */
    private void checkPWM() throws DriverException
    {
        if (pwm == null) {
            throw new DriverException("Cannot access PWM function");
        }
    }

}
