package org.lsst.ccs.subsystem.common.devices;

import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.subsystem.common.ErrorUtils;

/**
 *  Handles a Dataforth Maq20 system output control channel.
 *
 *  @author Owen Saxton
 */
public class Maq20Control implements HasLifecycle {

    /**
     *  Data fields.
     */
    @LookupName
    private String name;
    @LookupField(strategy=LookupField.Strategy.ANCESTORS)
    private Maq20Device maqDevc;

    private String type;

    private static final Logger LOG = Logger.getLogger(Maq20Control.class.getName());
    private Maq20Device.ModuleData modData;
    private boolean haveError = false;

    @Override
    public void init()
    {
        if (type == null) {
            ErrorUtils.reportConfigError(LOG, name, "type", "not specified");
        }
        String[] words = type.split(":");
        if (words.length != 1) {
            ErrorUtils.reportConfigError(LOG, name, "type", "doesn't have exactly one element");
        }
        try {
            int modIx = Integer.decode(words[0]);
            modData = maqDevc.getModuleData(modIx);
            if (modData == null) {
                ErrorUtils.reportConfigError(LOG, name, "module " + modIx, "doesn't exist");
            }
        }
        catch (NumberFormatException ex) {
            ErrorUtils.reportConfigError(LOG, name, "module index (" + words[0] + ")", "is not numeric");
        }
    }


    /**
     *  Sets the state of an output line.
     *
     *  These lines use inverted logic: 0 = on, 1 = off
     *
     *  @param  chan  The channel number
     *  @param  on    Whether to set on
     *  @throws  DriverException
     */
    public void setLineOn(int chan, boolean on) throws DriverException
    {
        if (modData.maqDisc != null) {
            modData.maqDisc.writeDisc(chan, on ? 0 : 1);
        }
        else {
            throw new DriverException("Error setting line: device " + maqDevc.getName() + " is offline");
        }
    }


    /**
     *  Gets the state of an output line.
     *
     *  @param  chan  The channel number
     *  @return  Whether line is on
     */
    public Boolean isLineOn(int chan)
    {
        Boolean on = null;
        try {
            if (modData.maqDisc != null) {
                on = modData.maqDisc.readDiscOut(chan) == 0;
                haveError = false;
            }
            else {
                throw new DriverException(maqDevc.getName() + " is offline");
            }
        }
        catch (DriverException e) {
            if (!haveError) {
                LOG.log(Level.SEVERE, "Error reading channel {0} state: {1}", new Object[]{chan, e});
                haveError = true;
            }
        }
        return on;
    }

}
