package org.lsst.ccs.subsystem.common.devices;

import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.monitor.MonitorLogUtils;
import org.lsst.ccs.utilities.logging.Logger;

/**
 *  Handles a Dataforth Maq20 system output control channel.
 *
 *  @author Owen Saxton
 */
public class Maq20Control implements HasLifecycle {

    /**
     *  Data fields.
     */
    private String type;

    private static final Logger LOG = Logger.getLogger(Maq20Control.class.getName());
    @LookupField(strategy=LookupField.Strategy.ANCESTORS)
    private Maq20Device maqDevc;
    private Maq20Device.ModuleData modData;
    private boolean haveError = false;
    @LookupName
    private String name;

    @Override
    public void init()
    {
        if (type == null) {
            MonitorLogUtils.reportConfigError(LOG, name, "type", "not specified");
        }
        String[] words = type.split(":");
        if (words.length != 1) {
            MonitorLogUtils.reportConfigError(LOG, name, "type", "doesn't have exactly one element");
        }
        try {
            int modIx = Integer.decode(words[0]);
            modData = maqDevc.getModuleData(modIx);
            if (modData == null) {
                MonitorLogUtils.reportConfigError(LOG, name, "module", "doesn't exist");
            }
        }
        catch (NumberFormatException ex) {
            MonitorLogUtils.reportConfigError(LOG, name, "module index", "is not numeric");
        }
    }


    /**
     *  Sets the state of an output line.
     *
     *  These lines use inverted logic: 0 = on, 1 = off
     *
     *  @param  chan  The channel number
     *  @param  on    Whether to set on
     *  @throws  DriverException
     */
    public void setLineOn(int chan, boolean on) throws DriverException
    {
        if (modData.maqDisc != null) {
            modData.maqDisc.writeDisc(chan, on ? 0 : 1);
        }
        else {
            throw new DriverException("Error setting line: device " + maqDevc.getName() + " is offline");
        }
    }


    /**
     *  Gets the state of an output line.
     *
     *  @param  chan  The channel number
     *  @return  Whether line is on
     */
    public Boolean isLineOn(int chan)
    {
        Boolean on = null;
        try {
            if (modData.maqDisc != null) {
                on = modData.maqDisc.readDiscOut(chan) == 0;
                haveError = false;
            }
            else {
                throw new DriverException(maqDevc.getName() + " is offline");
            }
        }
        catch (DriverException e) {
            if (!haveError) {
                LOG.severe("Error reading channel " + chan + " state: " + e);
                haveError = true;
            }
        }
        return on;
    }

}
