package org.lsst.ccs.subsystem.common.devices.vacuum;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.mks.GP390;
import org.lsst.ccs.drivers.mks.GP835;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.monitor.MonitorLogUtils;
import org.lsst.ccs.subsystem.common.devices.vacuum.states.VacuumState;
import org.lsst.ccs.subsystem.common.devices.vacuum.states.VDevState;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * Device class for the GP835 driver
 *
 * @author homer
 */
public class GPVacMon835Device extends Device implements VQMDevice {

    String serialdev;

    private static final Logger LOG = Logger.getLogger(GPVacMon835Device.class.getName());
    private final GP835 vqmDev = new GP835();
    VDevState.vacstates vstate = VDevState.vacstates.NOTCONFIGURED;
    double lastPres = 0.0;
    long last_disconnect = 0;

    /**
     * Initializes the device.
     * 
     * Used to check static device parameter values
     */
    @Override
    public void initDevice() {
        if (serialdev == null) {
            MonitorLogUtils.reportConfigError(LOG, name, "serialdev", "not specified");
        }
        fullName = "GP835 VQM device";
    }

    /**
     * Initializes the connection.
     */
    @Override
    protected void initialize() {
        try {
            vqmDev.open(serialdev);
            setOnline(true);
            initSensors();
            vstate = VDevState.vacstates.OK;
            LOG.info("Connected to " + fullName);
        }
        catch (DriverException e) {
            if (!inited) {
                LOG.error("Error connecting to " + fullName + ": " + e);
            }
        }
        inited = true;
    }

    /**
     * Closes the connection.
     */
    @Override
    protected void close() {
        try {
            vqmDev.close();
        }
        catch (DriverException e) {
            LOG.error("Error disconnecting from " + fullName + ": " + e);
        }
    }

    /**
     * Checks a channel's parameters for validity.
     *
     * @param name
     * @param hwChan
     * @param type
     * @param subtype
     * @return 
     * @throws Exception 
     */
    @Override
    protected int[] checkChannel(String name, int hwChan, String type, String subtype) throws Exception {
        if (hwChan != 0) {
            MonitorLogUtils.reportError(LOG, name, "hwChan", hwChan);
        }
        return new int[]{0, 0};
    }

    /**
     * Initializes a channel.
     *
     * Not needed in this case.
     *
     * @param chan
     * @param type
     * @param subtype
     */
    @Override
    protected void initChannel(int chan, int type, int subtype) {
    }

    /**
     * Reads a channel.
     *
     * @param chan
     * @param type
     * @return 
     */
    @Override
    protected double readChannel(int chan, int type) {
        double value = super.readChannel(chan, type);
        try {
            value = readPressure();
        } catch (Exception e) {
            LOG.error("Error reading VQM pressure: " + e);
            setOnline(false);
        }
        return value;
    }

    /**
     * Reconnects to the device.
     * 
     * This just forces a disconnect: monitoring will attempt the reconnect
     */
    @Override
    @Command(description = "Reconnect to the VQM")
    public void reconnect() {
        setOnline(false);
    }

    /**
     * Is the vacuum gauge on?
     * 
     * @return
     * @throws DriverException
     */
    @Override
    @Command(description = "Returns whether the GP835 device is active")
    public boolean isVacuumGaugeOn() throws DriverException {
        try {
            boolean state = vqmDev.isGaugeOn();
            if (!vstate.equals(VDevState.vacstates.TRIPPED)) {
                vstate = state ? VDevState.vacstates.ON : VDevState.vacstates.OFF;
            }
            return state;
        } catch (DriverException e) {
            LOG.error("Error reading vacuum gauge state: " + e);
            throw e;
        }
    }

    /**
     * Sets the vacuum gauge on
     * 
     * @throws DriverException
     */
    @Override
    @Command(description = "Turn on power to the vacuum gauge")
    public void setGaugeOn() throws DriverException {
        try {
            vqmDev.setGaugeOn();
        } catch (DriverException e) {
            LOG.error("Error setting VQM gauge on: " + e);
            throw e;
        }
    }

    /**
     * Sets the vacuum gauge off
     * 
     * @throws DriverException
     */
    @Override
    @Command(description = "Turn off power to the vacuum gauge")
    public void setGaugeOff() throws DriverException {
        try {
            vqmDev.setGaugeOff();
        } catch (DriverException e) {
            LOG.error("Error setting VQM gauge off: " + e);
            throw e;
        }
    }

    /**
     * Is the vacuum gauge filament on?
     * 
     * @return
     * @throws DriverException
     */
    @Command(description = "Returns whether the filament is on")
    public boolean isFilamenteOn() throws DriverException {
        try {
            return vqmDev.isFilamentOn();
        } catch (DriverException e) {
            LOG.error("Error reading vacuum gauge filament state: " + e);
            throw e;
        }
    }

    /**
     * Sets the vacuum gauge filament on
     * 
     * @throws DriverException
     */
    @Command(description = "Turn on the vacuum gauge filament")
    public void setFilamentOn() throws DriverException {
        try {
            vqmDev.setFilamentOn();
        } catch (DriverException e) {
            LOG.error("Error setting VQM gauge filament on: " + e);
            throw e;
        }
    }

    /**
     * Sets the vacuum gauge filament off
     * 
     * @throws DriverException
     */
    @Command(description = "Turn off the vacuum gauge filament")
    public void setFilamentOff() throws DriverException {
        try {
            vqmDev.setFilamentOff();
        } catch (DriverException e) {
            LOG.error("Error setting VQM gauge filament off: " + e);
            throw e;
        }
    }

    /**
     * Returns the pressure read from the GP835 device
     * 
     * @return
     * @throws DriverException
     */
    @Override
    @Command(type = Command.CommandType.QUERY, description = "returns the pressure read from the GP835 device")
    public double readPressure() throws DriverException {
        try {
            return lastPres = vqmDev.getPressure();
        } catch (DriverException e) {
            LOG.error("Error reading VQM pressure: " + e);
            throw e;
        }
    }

    /**
     * Enables a trip relay
     * 
     * @param relay The relay number (1-2)
     * @param limit The high pressure limit value
     * @throws DriverException
     */
    @Command(description="Enable a trip relay")
    public void enableRelay(@Argument(description="Relay number") int relay,
                            @Argument(description="High pressure limit") double limit) throws DriverException {
        vqmDev.setRelayTrip(relay, GP390.RELAY_ACT, limit);
        vqmDev.setRelayTrip(relay, GP390.RELAY_DEACT, 0.94 * limit);  // Can't be closer than 5% of the limit
        vqmDev.assignRelay(relay, GP390.RELAY_ABS);
        vqmDev.enableRelay(relay, true);
    }

    /**
     * Disables a trip relay
     * 
     * @param relay The relay number (1-2)
     * @throws DriverException
     */
    @Command(description="Disable a trip relay")
    public void disableRelay(@Argument(description="Relay number") int relay) throws DriverException {
        vqmDev.enableRelay(relay, false);
    }

    /**
     * Gets a trip relay limit value
     * 
     * @param relay The relay number (1-2)
     * @return The high pressure limit value
     * @throws DriverException
     */
    @Command(type=Command.CommandType.QUERY, description="Get a trip relay limit value")
    public double getRelayLimit(@Argument(description="Relay number") int relay) throws DriverException {
        return vqmDev.getRelayTrip(relay, GP390.RELAY_ACT);
    }

    /**
     * Gets a trip relay enabled state
     * 
     * @param relay The relay number (1-2)
     * @return Whether enabled
     * @throws DriverException
     */
    @Command(type=Command.CommandType.QUERY, description="Get a trip relay enabled state")
    public boolean isRelayEnabled(@Argument(description="Relay number") int relay) throws DriverException {
        return vqmDev.isRelayEnabled(relay);
    }

    /**
     * Gets a trip relay active state
     * 
     * @param relay The relay number (1-2)
     * @return Whether active
     * @throws DriverException
     */
    @Command(type=Command.CommandType.QUERY, description="Get a trip relay active state")
    public boolean isRelayActive(@Argument(description="Relay number") int relay) throws DriverException {
        return vqmDev.isRelayActive(relay);
    }

    /**
     * Returns the identification read from the GP835 device
     * 
     * @return
     * @throws DriverException
     */
    @Override
    @Command(type = Command.CommandType.QUERY, description = "Returns the ident read from the GP835 device")
    public String getIdent() throws DriverException {
        try {
            String[] ident = vqmDev.getIdentification();
            return ident[0] + "," + ident[1] + "," + ident[2] + "," + ident[3];
        } catch (DriverException e) {
            LOG.error("Error getting VQM device identification: " + e);
            throw e;
        }
    }

    /**
     * Returns the test report from the GP835 device
     * 
     * @return
     * @throws DriverException
     */
//    @Override
    @Command(type = Command.CommandType.QUERY, description = "Returns the test report from the GP835 device")
    public String readReport() throws DriverException {
        try {
            return vqmDev.getTestReport();
        } catch (DriverException e) {
            LOG.error("Error getting VQM test report : " + e);
            throw e;
        }
    }

    /**
     * Returns the last pressure read
     * 
     * @return
     */
    @Override
    public double getLastPres() {
        return lastPres;
    }

    /**
     * Reads the AMU data
     *
     * @return
     * @throws DriverException
     */
//    @Override
    @Command(type = Command.CommandType.QUERY, description = "read the buffer")
    public double[][] readAMU() throws DriverException {
        try {
            return vqmDev.readAMU();
        } catch (DriverException e) {
            LOG.error("Error reading AMU report: " + e);
            throw e;
        }
    }

    /**
     * Reads the AMU data and dumps it to a file
     *
     * @param filename
     * @return
     * @throws DriverException
     */
    @Command(type = Command.CommandType.QUERY, description = "read the buffer and dump to file")
    public double[][] readAMU(@Argument(name = "filename", description = "output filename for the VQM buffered values") String filename)
                              throws DriverException {
        double[][] buff = readAMU();
        FileWriter fstream = null;
        File pdFl = new File(filename);
//        if (pdFl.exists()) {
        try {
            if (pdFl.exists()) {
                pdFl.delete();
            }
            pdFl.createNewFile();
        } catch (Exception e) {
            LOG.error("Unable to create file (" + filename + ") for reason " + e);
        }
        try {
            fstream = new FileWriter(pdFl);
        } catch (IOException e) {
            LOG.error("Failed to open writer stream for file (" + filename + ") for reason " + e);
        }
        try {
            if (fstream != null) {
                LOG.error("writing file of buffered values from either Bias or PhotoDiode device");
//            BufferedWriter out = new BufferedWriter(new OutputStreamWriter(fstream));
                BufferedWriter out = new BufferedWriter(fstream);

                String line = null;
                for (int i = 0; i < buff[0].length; i++) {
                    line = buff[1][i] + " " + buff[0][i];
                    try {
                        out.write(line);
                        out.newLine();
                        out.flush();
                    } catch (IOException e) {
                        e.printStackTrace();
                    }

                }
                try {
                    out.close();
                } catch (IOException e) {
                    LOG.error("Failed to close file!");
                }

            }
        } catch (Exception ee) {
            LOG.error("Failed to remove and create a new file.!");

        }
        return buff;
    }

    /**
     * Reads the AMU data and returns it as a string
     * 
     * @return
     * @throws DriverException 
     */
    @Command(description = "Read the buffer and returns as a string")
    public String readAMUStr() throws DriverException {
        double data[][] = vqmDev.readAMU();
        return "Count = " + getString(data[0]) + "\n"
                + " AMUs = " + getString(data[1]) + "\n";
    }


    /**
     * Sets vacuum device status
     * 
     * @param istate 
     */
    @Override
    @Command(description = "Set vacuum device status")
    public void setVacState(@Argument(description="State to set") int istate) {
        vstate = VDevState.vacstates.values()[istate];
    }

    /**
     * Gets vacuum device status
     * 
     * @return 
     */
    @Override
    @Command(description = "Set vacuum device status")
    public int getVacState() {
        return vstate.ordinal();
    }

    /**
     * Converts an array of numbers to a string.
     * 
     * @param values
     * @return
     */
    private static StringBuilder getString(double[] values) {
        StringBuilder text = new StringBuilder();
        text.append(values[0]);
        for (int j = 1; j < values.length; j++) {
            text.append(", ").append(values[j]);
        }

        return text;
    }

}
