package org.lsst.ccs.subsystem.common.devices;

import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.dataforth.Maq20Discrete;
import org.lsst.ccs.monitor.Control;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.monitor.Monitor;
import org.lsst.ccs.monitor.MonitorLogUtils;
import org.lsst.ccs.utilities.logging.Logger;

/**
 *  Handles a Dataforth Maq20 system output control channel.
 *
 *  @author Owen Saxton
 */
public class Maq20Control extends Control {

    /**
     *  Data fields.
     */
    private String type;

    private static final Logger LOG = Logger.getLogger(Maq20Control.class.getName());
    private Maq20Device maqDevc;
    private Maq20Device.ModuleData modData;
    private boolean haveError = false;


    /**
     *  Performs configuration.
     *
     *  @param  mon   The associated monitor
     *  @param  devc  The associated device
     */
    @Override
    protected void configure(Monitor mon, Device devc)
    {
        super.configure(mon, devc);
        maqDevc = (Maq20Device)devc;
        if (type == null) {
            MonitorLogUtils.reportConfigError(LOG, getName(), "type", "not specified");
        }
        String[] words = type.split(":");
        if (words.length != 1) {
            MonitorLogUtils.reportConfigError(LOG, getName(), "type", "doesn't have exactly one element");
        }
        try {
            int modIx = Integer.decode(words[0]);
            modData = maqDevc.getModuleData(modIx);
            if (modData == null) {
                MonitorLogUtils.reportConfigError(LOG, getName(), "module", "doesn't exist");
            }
        }
        catch (NumberFormatException ex) {
            MonitorLogUtils.reportConfigError(LOG, getName(), "module index", "is not numeric");
        }
    }


    /**
     *  Sets the state of an output line.
     *
     *  These lines use inverted logic: 0 = on, 1 = off
     *
     *  @param  chan  The channel number
     *  @param  on    Whether to set on
     *  @throws  DriverException
     */
    public void setLineOn(int chan, boolean on) throws DriverException
    {
        if (modData.maqDisc != null) {
            modData.maqDisc.writeDisc(chan, on ? 0 : 1);
        }
        else {
            throw new DriverException("Error setting line: device " + maqDevc.getName() + " is offline");
        }
    }


    /**
     *  Gets the state of an output line.
     *
     *  @param  chan  The channel number
     *  @return  Whether line is on
     */
    public Boolean isLineOn(int chan)
    {
        Boolean on = null;
        try {
            if (modData.maqDisc != null) {
                on = modData.maqDisc.readDiscOut(chan) == 0;
                haveError = false;
            }
            else {
                throw new DriverException(maqDevc.getName() + " is offline");
            }
        }
        catch (DriverException e) {
            if (!haveError) {
                LOG.severe("Error reading channel " + chan + " state: " + e);
                haveError = true;
            }
        }
        return on;
    }

}
