package org.lsst.ccs.subsystem.common.devices.refrigeration;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.ametek.AvcCooler;
import org.lsst.ccs.drivers.commons.DriverConstants;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.monitor.MonitorLogUtils;
import org.lsst.ccs.subsystem.common.devices.refrigeration.data.RefrigerationConfig;
import org.lsst.ccs.subsystem.common.devices.refrigeration.states.CryoDevState;
import org.lsst.ccs.utilities.logging.Logger;
//import org.lsst.ccs.drivers.ascii.Ascii;


/**
 * Device class for the AVCDRV driver
 *
 * @author homer
 */
public class AVCCryoDevice extends Device {
// implements CryoTelDevice {

    String serialdev;
    DriverConstants.ConnType connType;
    RefrigerationConfig.CryoTelType devType = null;
    
    private RefrigerationConfig.CryoTelType localdevType = null;
    private String localname = null;

    private static final Logger LOG = Logger.getLogger(AVCCryoDevice.class.getName());

    String ident;
    double T_OFFSET = 273.0; // for choosing between celsius or Kelvin
    int baud = 9600;
    int dataChar = 0;
    //                   encoded data characteristics for FTDI or serial:
    //                     0 sets 8-bit, no parity, 1 stop bit, no flow control

    private final AvcCooler AVCDRV = new AvcCooler();
    RefrigerationConfig cfg = new RefrigerationConfig();
    CryoDevState.cryostates cooler_state = CryoDevState.cryostates.NOTCONFIGURED;
    long last_disconnect = 0;

    @ConfigurationParameter(category = "Limits")
    private double temp_Min_OFF = -140.; // temperature at which cryotel should be shut off to avoid getting too cold
    @ConfigurationParameter(category = "Limits")
    private double temp_Min_ON = -120.; // temperature at which the cryotel can be turned back on after auto shut-off

    private boolean auto_shutoff_state = false; // has device been automatically shut off due to dangerous temperature
    private boolean commanded_on = false; // was a request made to turn on the CryoTel
    
    
    /**
     * Initializes the device.
     *
     * Used to check static device parameter values
     */
//    @Override
    public void initDevice() {
        if (serialdev == null) {
            MonitorLogUtils.reportConfigError(LOG, name, "serialdev", "not specified");
        }
        fullName = "AVCDRV CryoTel device";
    }

    /**
     * Initializes the connection.
     */
//    @Override
    protected void initialize() {
        this.localname = name;
        this.localdevType = devType;
        try {
//        super.open(CONN_TYPE_SERIAL, serialname, baud);

//            AVCDRV.open(connType.SERIAL, serialdev, baud, dataChar);
            AVCDRV.openSerial(serialdev, baud);

            //DriverConstants.ConnType.TELNET, node, user, passwd);
            //DriverConstants.ConnType.NET, name, lineState, lineWarm);
            setOnline(true);
            initSensors();
            cooler_state = CryoDevState.cryostates.OK;
            LOG.info("Connected to " + fullName + " at " + serialdev + " and name " + name);
        } catch (DriverException e) {
            if (!inited) {
                LOG.error("Error connecting to " + fullName + " at " + serialdev + " and name " + name + ": " + e);
            }
        }
        // always start with the CryoTel is OFF mode ... TBD
//        AVCDRV.setMode(AvcCooler.Mode.OFF);
        // always start with the CryoTel is OFF mode ... TBD
//        AVCDRV.setMode(AvcCooler.Mode.OFF);

        inited = true;
    }

    /**
     * Closes the connection.
     */
//    @Override
    protected void close() {
        try {
            AVCDRV.close();
        } catch (DriverException e) {
            LOG.error("Error disconnecting from " + fullName + ": " + e);
        }
    }

    /**
     * Checks a channel's parameters for validity.
     *
     * @param name
     * @param hwChan
     * @param type
     * @param subtype
     * @return
     * @throws Exception
     */
//    @Override
    protected int[] checkChannel(String name, int hwChan, String type, String subtype) throws Exception {
        if (hwChan > 6) {
            MonitorLogUtils.reportError(LOG, name, "hwChan", hwChan);
        }
        return new int[]{0, 0};
    }

    /**
     * Initializes a channel.
     *
     * Not needed in this case.
     *
     * @param chan
     * @param type
     * @param subtype
     */
//    @Override
    protected void initChannel(int chan, int type, int subtype) {
    }

    /**
     * Reads a channel.
     *
     * @param chan
     * @param type
     * @return
     */
//    @Override
    protected double readChannel(int chan, int type) {
        double value = super.readChannel(chan, type);
        //	double value = 0.;
        if (chan == 0) {
            try {
                value = this.getTemperature();
            } catch (Exception e) {
                LOG.error("Error reading Temperature: " + e);
                //		setOnline(false);
            }
        }
        if (chan == 1) {
            try {
                value = this.getSetPoint();
            } catch (Exception e) {
                LOG.error("Error reading Temperature Set Point: " + e);
                //		setOnline(false);
            }
        }
        if (chan == 2) {
            try {
                value = this.getPower();
            } catch (Exception e) {
                LOG.error("Error reading Power setting: " + e);
                //		setOnline(false);
            }
        }
        if (chan == 3) {
            try {
                value = this.readPower();
            } catch (Exception e) {
                LOG.error("Error reading Power: " + e);
                //		setOnline(false);
            }
        }
        if (chan == 4) {
            try {
                value = this.showAuto_shutoff_state() ? 1.0 : 0.0 ;
            } catch (Exception e) {
                LOG.error("Error getting auto shutoff state: " + e);
                //		setOnline(false);
            }
        }
        if (chan == 5) {
                value = this.getTemp_Min_OFF() ;
        }
        if (chan == 6) {
                value = this.getTemp_Min_ON() ;
        }

        return value;
    }

    /**
     * Reconnects to the device.
     *
     * This just forces a disconnect: monitoring will attempt the reconnect
     */
//    @Override
    @Command(description = "Reconnect to the cooler")
    public void reconnect() {
        setOnline(false);
    }

    /**
     * Get the product type
     */
//    @Override
    @Command(description = "Show the product type")
    public String getProduct() throws DriverException {
        return AVCDRV.getProductType();
    }

//    @Override
    @Command(description = "Show the firmware version")
    public String getVersion() throws DriverException {
        return AVCDRV.getFWVersion();
    }

//    @Override
    @Command(description = "Show the cooler mode")
    public String getMode() throws DriverException {
        return AVCDRV.getMode().toString();
    }

// @Override
    @Command(description = "Set the cooler mode")
    public void setMode(String mode) throws DriverException {
        AVCDRV.setMode(AvcCooler.Mode.valueOf(mode));
        commanded_on = mode.toUpperCase().contains("ON");
        auto_shutoff_state = false;

    }

//   @Override
    @Command(description = "Show the target temperature")
    public double getSetPoint() throws DriverException {
        return AVCDRV.getTemperature() - T_OFFSET;
    }

//    @Override
    @Command(description = "Set the target temperature")
    public void setSetPoint(double temp) throws DriverException {
        AVCDRV.setTemperature(temp + T_OFFSET);
    }

//    @Override
    @Command(description = "Get the cooler power")
    public double getPower() throws DriverException {
        return AVCDRV.getPower();
    }

//    @Override
    @Command(description = "Read the cooler power")
    public double readPower() throws DriverException {
        return AVCDRV.readPower();
    }

//    @Override
    @Command(description = "Set the cooler power")
    public void setPower(double pwr) throws DriverException {
        AVCDRV.setPower(pwr);
    }

//    @Override
    @Command(description = "Show the temperature")
    public double getTemperature() throws DriverException {
        double temp = AVCDRV.readTemperature() - T_OFFSET;

        if (temp < this.temp_Min_OFF && !auto_shutoff_state) {
            this.setMode("OFF");
            LOG.info("Cryotel temperature "+temp+" C is below shutoff limit of "+temp_Min_OFF+" C . Cryotel mode set to OFF");
            auto_shutoff_state = true; // NOTE: This has to be after the setMode
        }
// In order to set mode to ON :
// * the flag indicating that the mode had been automatically shutoff must be true AND
// * the flag indicating that the user had specified that the MODE can be turned to ON must be true
//
        if (temp > this.temp_Min_ON && auto_shutoff_state && commanded_on) {
            LOG.info("Cryotel temperature "+temp+" C is above turn on limit of "+temp_Min_ON+" C after an auto shutoff. Cryotel mode set to ON");
            this.setMode("ON");
            auto_shutoff_state = false;
        }
        return temp;
    }

    /**
     * ************************************************************************
     */
    /**
     * Sets cryo device status
     *
     * @param istate
     */
//    @Override
    @Command(description = "Set cryo device status")
    public void setState(@Argument(description = "State to set") int istate) {
        cooler_state = CryoDevState.cryostates.values()[istate];
    }

    @Command(description = "Show the state of the auto shutoff")
    public boolean showAuto_shutoff_state() {
        return auto_shutoff_state;
    }

    /**
     * Gets cryo device status
     *
     * @return
     */
//    @Override
    @Command(description = "Set cryo device status")
    public int getState() {
        return cooler_state.ordinal();
    }

    /**
     * Converts an array of numbers to a string.
     *
     * @param values
     * @return
     */
    private static StringBuilder getString(double[] values) {
        StringBuilder text = new StringBuilder();
        text.append(values[0]);
        for (int j = 1; j < values.length; j++) {
            text.append(", ").append(values[j]);
        }

        return text;
    }
    @Command(description = "Get serial dev specification")
    public String getSerialdev() {
        return serialdev;
    }

    @Command(description = "Get device identifier")
    public String getIdent() {
        return ident;
    }

    @Command(description = "Get cryotel state")
    public CryoDevState.cryostates getCooler_state() {
        return cooler_state;
    }

    @Command(description = "Get temperature at which cryotel will be shutoff automatically.")
    public double getTemp_Min_OFF() {
        return temp_Min_OFF;
    }

    @Command(description = "Set temperature at which cryotel will be shutoff automatically.")
    public String setTemp_Min_OFF(double temp_Min_OFF) {
        this.temp_Min_OFF = temp_Min_OFF;
        return("This is a configurable value that should be changed via: Ex: CryoTelCryo change temp_Min_OFF -145.");
    }

    @Command(description = "Get temperature at which cryotel will be turned on automatically after an automatic shutoff.")
    public double getTemp_Min_ON() {
        return temp_Min_ON;
    }

    @Command(description = "Set temperature at which cryotel will be turned on automatically after an automatic shutoff.")
    public String setTemp_Min_ON(double temp_Min_ON) {
        this.temp_Min_ON = temp_Min_ON;
        return("This is a configurable value that should be changed via: Ex: CryoTelCryo change temp_Min_ON -120.");
    }

    @Command(description = "Return name of the device")
    public String getName() {
        return localname;
    }

    @Command(description = "Get the most recent error")
    public int getError() throws DriverException {
        return AVCDRV.getError();
    }
    
    @Command(description = "Get the most recent error")
    public double getRejectTemp() throws DriverException {
        return AVCDRV.readRejectTemp();
    }

    @Command(description = "Return the devType")
    public RefrigerationConfig.CryoTelType getDevType() {
        return localdevType;
    }
    
    
    
}
