package org.lsst.ccs.subsystem.common.devices.power.distribution;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.lsst.ccs.Agent;
import org.lsst.ccs.StateChangeListener;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.subsystem.common.devices.power.distribution.state.PduOutletState;
import org.lsst.ccs.subsystem.common.devices.power.distribution.state.PduState;
import org.lsst.ccs.subsystem.common.devices.power.distribution.state.PduProperties;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupPath;
import org.lsst.ccs.description.ComponentLookup;
import org.lsst.ccs.description.ComponentNode;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.apcpdu.APC7900;
import org.lsst.ccs.drivers.apcpdu.APC7900B;
import org.lsst.ccs.drivers.apcpdu.APC7900Series;
import org.lsst.ccs.drivers.apcpdu.APC7900Sim;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.monitor.MonitorLogUtils;
import org.lsst.ccs.services.AgentStateService;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.utilities.logging.Logger;

/**
 *  APC7900 device class
 *
 *  @author: The LSST CCS Team
 */
public class APC7900Device extends Device implements StateChangeListener {

    public final static int
        TYPE_POWER = 1,
        CHAN_CURRENT = 0,
        CHAN_POWER = 1;

    final static Map<String, Integer> typeMap = new HashMap<>();
    static {
        typeMap.put("POWER", TYPE_POWER);
    }

    @ConfigurationParameter(isFinal = true)
    private String type = "APC7900";

    @ConfigurationParameter(isFinal = true)
    protected String node;

    @ConfigurationParameter(isFinal = true)
    protected int numberOfOutlets = 8;

    @ConfigurationParameter(isFinal = true)
    protected List<String> outlets = new ArrayList<>();

    private final List<PduOutlet> listOfOutlets = new ArrayList();
    
    protected Map<String,PduOutlet> mapOfOutlets = new HashMap<>();
    
    String user = "apc";
    String passwd = "apc";

    protected final Logger sLog = Logger.getLogger(getClass().getPackage().getName());
    private APC7900Series pdu;
    
    @LookupField(strategy = LookupField.Strategy.TREE)
    private Agent a;
    
    private ComponentLookup componentLookup;
    
    @LookupField(strategy = LookupField.Strategy.TREE)
    protected AgentStateService stateService;

    @LookupField(strategy = LookupField.Strategy.TREE)
    protected AlertService alertService;

    @LookupPath
    private String path;
    
    @Override
    public void build() {
        super.build(); //To change body of generated methods, choose Tools | Templates.
        
        if ( ! outlets.isEmpty() ) {
            if ( outlets.size() != numberOfOutlets ) {
                throw new IllegalArgumentException("The size of the provided outlets ("+outlets.size()+") must match the provided numberOfOutlets "+numberOfOutlets);
            }
        }
        
        componentLookup = a.getComponentLookup();
        ComponentNode thisNode = componentLookup.getComponentNodeForObject(this);
        for ( int i = 1; i <= numberOfOutlets; i++ ) {
            PduOutlet outlet = new PduOutlet(i);
            if ( ! outlets.isEmpty() ) {
                outlet.setName(outlets.get(i-1));
            }
            mapOfOutlets.put(outlet.getName(), outlet);
            listOfOutlets.add(outlet);
            //Add the Pdu Outlet node and register its state.
            componentLookup.addComponentNodeToLookup(thisNode, new ComponentNode(thisNode, outlet.getName(), outlet));
        }

        a.setAgentProperty(PduProperties.HAS_PDU, "true");
        String components = a.getAgentProperty(PduProperties.PDU_COMPONENTS);
        if (components == null) {
            components = "";
        }
        if ( !components.isEmpty() ) {
            components += ",";
        }
        components += path;
        a.setAgentProperty(PduProperties.PDU_COMPONENTS, components);        
    }

    @Override
    public void init() {
        
        if ( type == "APC7900" ) {
            pdu = new APC7900();
        } else if ( type == "APC7900B") {
            pdu = new APC7900B();            
        } else if ( type == "APC7900Sim") {
            pdu = new APC7900Sim(outlets);
        }
            
        //Register the Pdu State for this component
        stateService.registerState(PduState.class, "The state of the Pdu", this);        
        //And set its initial state
        stateService.updateAgentComponentState(this, PduState.NOTCONFIGURED);
                
        for ( PduOutlet outlet : mapOfOutlets.values() ) {
            stateService.registerState(PduOutletState.class, "The state of a Pdu Outlet: ON/OFF", outlet);            
        }
        
        //Add state change listener to raise alerts if/when outlets are turned off.
        stateService.addStateChangeListener(this, PduOutletState.class, PduState.class);

        super.init();
    }
    
    
    /**
     *  Performs configuration.
     */
    @Override
    protected void initDevice()
    {
        if (node == null) {
            MonitorLogUtils.reportConfigError(sLog, name, "node", "is not specified");
        }
        fullName = "APC7900 PDU (" + node + ")";
    }


    /**
     *  Initializes the device.
     */
    @Override
    protected void initialize()
    {
        try {
            pdu.open(APC7900.ConnType.TELNET, node, user, passwd);
            Map<String,Integer> outNumMap = pdu.getOutletNumberMap();

            if ( outNumMap.size() != numberOfOutlets ) {
                throw new RuntimeException("Unexpected number of outlets: "+outNumMap.size()+" it should have been "+numberOfOutlets+".");
            }
            
            for ( String outletName : outNumMap.keySet() ) {
                int index = outNumMap.get(outletName);
                if ( outlets.isEmpty() ) {
                    listOfOutlets.get(index-1).setName(outletName);
                } else {
                    String providedOutletName = outlets.get(index -1);
                    if ( ! outletName.equals(providedOutletName) ) {
                        log.warning("Changing the description for outlet "+index+" from "+outletName+" to "+providedOutletName);
                        pdu.setOutletName(index, providedOutletName);
                    }
                }       
            }
                        
            initSensors();
            setOnline(true);
            stateService.updateAgentComponentState(this, PduState.OK);
            sLog.info("Connected to " + fullName);
        }
        catch (DriverException e) {
            if (!inited) {
                sLog.error("Error connecting to " + fullName + ": " + e);
            }
            close();
        }
        inited = true;
    }


    /**
     *  Closes the connection.
     */
    @Override
    protected void close()
    {
        try {
            pdu.close();
        }
        catch (DriverException e) {
        }
        stateService.updateAgentComponentState(this, PduState.NOTCONFIGURED);
    }

    @Override
    public void stateChanged(Object changedObj, Enum newState, Enum oldState) {
        if ( changedObj instanceof PduOutlet ) {            
            if ( newState == PduOutletState.OFF ) {
                PduOutlet outlet = (PduOutlet)changedObj;
                alertService.raiseAlert(new Alert(PduProperties.PDU_OUTLET_ALERT,"Outlet turned off"), AlertState.ALARM, "Outlet "+outlet.getName()+" has been turned off");
            }
        }
        if ( changedObj == this && newState != PduState.OK ) {
            alertService.raiseAlert(new Alert(PduProperties.PDU_ALERT,"PDU problem"), AlertState.ALARM, "PDU "+getName()+" is in state "+newState);            
        }
    }

   /**
    *  Checks a channel's parameters for validity.
    *
    *  @param  name     The channel name
    *  @param  hwChan   The hardware channel
    *  @param  type     The channel type string
    *  @param  subtype  The channel subtype string
    *  @return  Two-element array containing the encoded type [0] and subtype [1] 
    *  @throws  Exception  If parameters are invalid
    */
    @Override
    protected int[] checkChannel(String name, int hwChan, String type, String subtype) throws Exception
    {
        Integer iType = typeMap.get(type.toUpperCase());
        int index = 0;
        if (iType == null) {
            MonitorLogUtils.reportError(sLog, name, "channel type", type);
        }
        if (hwChan != CHAN_CURRENT && hwChan != CHAN_POWER) {
            MonitorLogUtils.reportError(sLog, name, "hardware channel", hwChan);
        }
        
        return new int[]{iType | (index << 16), 0};
    }


    /**
     *  Reads a channel.
     *
     *  @param  hwChan  The hardware channel number
     *  @param  type    The encoded channel type
     *  @return  The read value
     */
    @Override
    protected double readChannel(int hwChan, int type)
    {
        double value = 0;
        switch (type & 0xffff) {
            case TYPE_POWER:
                try {
                    value = hwChan == CHAN_POWER ? pdu.readPower() : pdu.readCurrent()[0];
                } catch (DriverException e) {
                    sLog.error("Error reading power/current value: " + e);
                    setOnline(false);
                }
        }
        return value;
    }

    @Override
    protected void readChannelGroup() {
        for (PduOutlet outlet : mapOfOutlets.values()) {
            try {
                double value = pdu.isOutletOn(outlet.getIndex()) ? 1.0 : 0.0;
                stateService.updateAgentComponentState(outlet, value == 1 ? PduOutletState.ON : PduOutletState.OFF);
            } catch (DriverException e) {
            }
        }
    }

    /**
     *  Disables the PDU.
     *
     *  Needed to relinquish control to the web or another telnet client
     */
    @Override
    @Command(name="disable", description="Disable the connection to the PDU")
    public void disable()
    {
        super.disable();
        setOnline(false);
    }


    /**
     *  Enables the PDU.
     */
    @Override
    @Command(name="enable", description="Enable the connection to the PDU")
    public void enable()
    {
        super.enable();
    }


    /**
     *  Gets the list of outlet names.
     *
     *  @return  The list of names
     */
    @Command(name="getOutletNames", description="Get the list of outlet names")
    public List<String> getOutletNames()
    {
        List<String> result = new ArrayList<>();
        for ( PduOutlet outlet: listOfOutlets ) {
            result.add(outlet.getName());
        }
        return result;
    }


    /**
     *  Gets the map of outlet on states.
     *
     *  @return  The map of outlet names to on states (true or false)
     *  @throws  DriverException
     */
    @Command(name="getOutletOnStateMap", description="Get the map of outlet on states")
    public Map<String, Boolean> getOutletOnStateMap() throws DriverException
    {
        return pdu.getOutletOnStateMap();
    }


    /**
     *  Turns an outlet on.
     *
     *  @param  name  The outlet name
     *  @throws  DriverException
     */
    @Command(name="outletOn", description="Turn outlet on")
    public void outletOn(@Argument(name="name", description="Outlet name")
                         String name) throws DriverException
    {
        pdu.delayedOutletOn(getOutletNumber(name));        
    }


    /**
     *  Turns an outlet off.
     *
     *  @param  name  The outlet name
     *  @throws  DriverException
     */
    @Command(name="outletOff", description="Turn outlet off")
    public void outletOff(@Argument(name="name", description="Outlet name")
                          String name) throws DriverException
    {
        pdu.delayedOutletOff(getOutletNumber(name));
    }


    /**
     *  Forces an outlet to turn on.
     *
     *  @param  name  The outlet name
     *  @throws  DriverException
     */
    @Command(name="forceOutletOn", description="Force outlet to turn on")
    public void forceOutletOn(@Argument(name="name", description="Outlet name")
                              String name) throws DriverException
    {
        pdu.setOutletOn(getOutletNumber(name));
    }


    /**
     *  Forces an outlet to turn off.
     *
     *  @param  name  The outlet name
     *  @throws  DriverException
     */
    @Command(name="forceOutletOff", description="Force outlet to turn off")
    public void forceOutletOff(@Argument(name="name", description="Outlet name")
                               String name) throws DriverException
    {
        pdu.setOutletOff(getOutletNumber(name));
    }


    /**
     *  Gets whether an outlet is on.
     *
     *  @param  name  The outlet name
     *  @return  Whether outlet is turned on
     *  @throws  DriverException
     */
    @Command(name="isOutletOn", description="Get whether outlet is on")
    public boolean isOutletOn(@Argument(name="name", description="Outlet name")
                              String name) throws DriverException
    {
        return pdu.isOutletOn(getOutletNumber(name));
    }


    /**
     *  Change the state of an outlet
     *
     *  @param name  The outlet name
     *  @param state The desired PduOutletState for the outlet.
     *  @param force true if the state change is to be forced.
     *  @throws  DriverException
     */
    @Command(name="setOutletState", description="Set the outlet state to a given OutletState")
    public void setOutletState(
            @Argument(allowedValueProvider = "getOutletNames", name="name", description="Outlet name") String name, 
            @Argument(name="state", description="The desired PduOutletState") PduOutletState state, 
            @Argument(name="force", description="true to force the state change", defaultValue = "false") boolean force 
            ) throws DriverException
    {
        PduOutlet outlet = mapOfOutlets.get(name);
        if ( outlet == null ) {
            throw new IllegalArgumentException("Outlet name: "+name+" is illegal. The chosen outlet does not exist.");
        }
        if ( force ) {
            if ( state == PduOutletState.OFF ) {
                pdu.setOutletOff(name);
            } else {
                pdu.setOutletOn(name);                
            }
        } else {
            if ( state == PduOutletState.OFF ) {
                pdu.delayedOutletOff(name);
            } else {
                pdu.delayedOutletOn(name);                
            }            
        }
        stateService.updateAgentComponentState(outlet, state);
    }


    
    
    
    
    /**
     *  Gets an outlet number from its name.
     *
     *  This routine should be unnecessary since the PDU handles names directly,
     *  but there's a bug in the PDU parser which causes it to report an error
     *  if the name is hyphenated.
     *
     *  @param  name  The outlet name
     *  @return  The outlet number
     *  @throws  DriverException
     */
    int getOutletNumber(String name) throws DriverException
    {
        for ( PduOutlet outlet : listOfOutlets) {
            if ( outlet.getName().equals(name) ) {
                return outlet.getIndex();
            }
        }
        throw new DriverException("Invalid outlet name "+name);
    }    
}
