package org.lsst.ccs.subsystem.common.thermalsim;

import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;

/**
 * A generic thermal element.
 * 
 * This can be a link, a bulk element, or a composite. Handle the thermal loop.
 * 
 * Element should be connected via the connect method of ThermalModel which
 * handles properly all the possibilities.
 * 
 * Bulk elements have a temperature. Composite elements can have a temperature.
 * 
 * @author aubourg
 *
 */

public abstract class ThermalSimElement {

    protected volatile long timeStep = 20; // in milliseconds

    protected volatile long lastVisit = 0; // millis

   //protected volatile long dt = 0;

    protected volatile ScheduledFuture<?> future = null;

    ThermalModel model;

    String name;

    public ThermalSimElement(ThermalModel model, String name) {
        this.model = model;
        this.name = name;

        model.addElement(this); // will not be used until built, no leak issue
    }

    public void setTimeStep(long timeStep) {
        this.timeStep = timeStep;
        if (future != null) {
            future.cancel(false);
            start();
        }
    }

    public String getName() {
        return name;
    }

    void start() {
        long ts = timeStep / model.speedUp;
        if (ts<1) ts=1;
        
        future = model.execService.scheduleAtFixedRate(() -> doThermalLoop(), 0, ts, TimeUnit.MILLISECONDS);
    }

    void stop() {
        if (future != null) {
            future.cancel(false);
            future = null;
        }
    }

    public long getTimeStep() {
        return timeStep;
    }

    private void doThermalLoop() {
        long t = System.currentTimeMillis();
        long dt;
        if (lastVisit == 0) {
            dt = 0;
        } else {
            dt = t - lastVisit;
        }
        dt *= model.speedUp;
        lastVisit = t;
        thermalLoop(dt);
    }

    protected ThermalSimElement getActualRight() {
        return this;
    }

    protected ThermalSimElement getActualLeft() {
        return this;
    }

    protected abstract void thermalLoop(long dt);

    public void dump() {
        if (getTemperature() > 0)
            System.out.printf("%-15s %-30s %5.2f %40s\n", name, getClass().getSimpleName(), getTemperature(), getExtraInfo());
        else
            System.out.printf("%-15s %-30s       %40s\n", name, getClass().getSimpleName(), getExtraInfo());
    }

    protected String getExtraInfo() {
        return "";
    }

    public boolean isActive() {
        return true;
    }

    public double getTemperature() {
        return 0;
    }

}
