package org.lsst.ccs.subsystem.common.thermalsim;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;

/**
 * A thermal model that manages a set of thermal elements (ThermalSimElement).
 * 
 * Handles connecting the elements, and starting the whole simulation.
 * 
 */

public class ThermalModel {

    ScheduledExecutorService execService = Executors.newScheduledThreadPool(java.lang.Runtime.getRuntime().availableProcessors() / 3 + 1);

    Map<String, ThermalSimElement> elements = new ConcurrentHashMap<String, ThermalSimElement>();

    
    int speedUp = 1;

    boolean started = false;

    public ThermalModel() {

    }

    void addElement(ThermalSimElement e) {
        if (elements.containsKey(e.getName()))
            throw new RuntimeException("duplicate element name " + e.getName());
        elements.put(e.getName(), e);
    }

    public void start() {
        elements.values().stream().filter(ThermalSimElement::isActive).forEach(ThermalSimElement::start);
        started = true;
    }

    public void stop() {
        elements.values().stream().forEach(e -> e.stop());
        started = false;
    }

    public void dump() {
        elements.values().stream().forEach(e -> e.dump());
    }

    public void dumpActive() {
        elements.values().stream().filter(ThermalSimElement::isActive).forEach(ThermalSimElement::dump);
    }

    public void setSpeedUp(int speedUp) {
        boolean restart = started;
        if (started)
            stop();
        this.speedUp = speedUp;
        if (restart)
            start();
    }

    public void connect(ThermalSimElement left, ThermalSimElement right) {
        if (started) {
            throw new RuntimeException("Cannot modify network when simulation has started");
        }
        System.out.println("connect " + left + " " + right);
        left = left.getActualRight();
        right = right.getActualLeft();
        System.out.println(">connect " + left + " " + right);

        boolean connected = false;
        if (left instanceof ThermalLink && right instanceof ThermalCapacitance) {
            ((ThermalLink) left).setRight((ThermalCapacitance) right);
            connected = true;
        }
        if (right instanceof ThermalLink && left instanceof ThermalCapacitance) {
            ((ThermalLink) right).setLeft((ThermalCapacitance) left);
            connected = true;
        }
        if (right instanceof ResistiveThermalLink && left instanceof ResistiveThermalLink) {
            // we need to merge the links and sum the resistances.
            ResistiveThermalLink l = (ResistiveThermalLink) left;
            ResistiveThermalLink r = (ResistiveThermalLink) right;
            MergedResistiveThermalLink merge = null;

            if (l.mergedLink == null && r.mergedLink == null) {
                merge = new MergedResistiveThermalLink(l.model, l.name + "+" + r.name, l.res + r.res);
                merge.links.add(l);
                merge.links.add(r);
                l.setMergedLink(merge);
                r.setMergedLink(merge);
                if (r.right != null)
                    merge.setRight(r.right);
                if (l.left != null)
                    merge.setLeft(l.left);
            } else if (l.mergedLink != null && r.mergedLink == null) {
                merge = l.mergedLink;
                merge.name += "+" + r.name;
                merge.res += r.res;
                merge.links.add(r);
                r.setMergedLink(merge);
                if (r.right != null)
                    merge.setRight(r.right);
            } else if (l.mergedLink == null && r.mergedLink != null) {
                merge = r.mergedLink;
                merge.name = l.name + "+" + merge.name;
                merge.res += l.res;
                merge.links.add(0, l);
                l.setMergedLink(merge);
                if (l.left != null)
                    merge.setLeft(l.left);
            } else {
                merge = l.mergedLink;
                merge.name += "+" + r.mergedLink.name;
                merge.res += r.mergedLink.res;
                merge.links.addAll(r.mergedLink.links);
                for (ResistiveThermalLink lk : r.mergedLink.links) {
                    lk.setMergedLink(merge);
                }
                if (r.mergedLink.right != null)
                    merge.setRight(r.mergedLink.right);
            }

            connected = true;
        }
        if (!connected)
            throw new RuntimeException("cannot connect " + left + " with " + right);

    }

}
