package org.lsst.ccs.subsystem.common.thermalsim;


/**
 * A composite link + bulk element.
 * 
 * Side (connectable) elements are resistive thermal links with R/2, connected to a central
 * capacitance C.
 * 
 * @author aubourg
 *
 */
public class ThermalCapacitanceLink extends ThermalLink {

    // This should be a capacitance + a resistance, attached, and delegating
    // to have links on both side we do R/2 - C - R/2

    public ThermalCapacitanceLink(ThermalModel model, String name, double resistance, double capacitance) {
        super(model, name);
        resLeft = new ResistiveThermalLink(model, name + "_rl", resistance / 2);
        resRight = new ResistiveThermalLink(model, name + "_rr", resistance / 2);
        cap = new ThermalCapacitance(model, name + "_c", capacitance);
        resLeft.setRight(cap);
        resRight.setLeft(cap);
    }

    ResistiveThermalLink resLeft, resRight;
    ThermalCapacitance cap;

    public void setTemperature(double t) {
        cap.setTemperature(t);
    }

    public double getTemperature() {
        return cap.getTemperature();
    }

    @Override
    protected ThermalSimElement getActualLeft() {
        return resLeft;
    }

    @Override
    protected ThermalSimElement getActualRight() {
        return resRight;
    }

    @Override
    void setLeft(ThermalCapacitance left) {
        super.setLeft(left);
        resLeft.setLeft(left);
    }

    @Override
    void setRight(ThermalCapacitance right) {
        super.setRight(right);
        resRight.setRight(right);
    }

    @Override
    protected void thermalLoop(long dt) {
        // NOOP will be done by delegates
    }

    @Override
    public double power(double tLeft, double tRight) {
        return 0; // will be done by delegates
    }

    @Override
    protected String getExtraInfo() {
        return String.format(" c=%5.2f r=%5.2f", cap.thermalCap, resLeft.res*2);

    }
    
    public boolean isActive() {
        return false;
    }


}
