package org.lsst.ccs.subsystem.common.thermalsim;

import org.python.google.common.util.concurrent.AtomicDouble;


/**
 * A bulk element, that will adjust its temperature upon heat transfer.
 * 
 * Capacitance is given in J/K.
 * 
 * @author aubourg
 *
 */
public class ThermalCapacitance extends ThermalSimElement {

    public ThermalCapacitance(ThermalModel model, String name, double capacitance) {
        super(model, name);
        thermalCap = capacitance;
    }

    protected AtomicDouble temperature = new AtomicDouble(); // Kelvins
    protected AtomicDouble qAccumulated = new AtomicDouble();

    double thermalCap; // J/K

    @Override
    public double getTemperature() {
        return temperature.get();
    }

    public void setTemperature(double t) {
        temperature.set(t);
    }

    public double increaseTemperature(double dt) {
        return temperature.addAndGet(dt);
    }

    public synchronized void heat(double q) {
        qAccumulated.addAndGet(q);
    }

    @Override
    protected void thermalLoop(long dt) {
        double q = qAccumulated.get();
        qAccumulated.addAndGet(-q);
        double deltaT = q / thermalCap;
        if (deltaT > 50)
            deltaT = 50;
        if (deltaT < -50)
            deltaT = -50;
        increaseTemperature(deltaT);
        // System.out.printf("cap %-10s Q %6.4f deltaT %6.4f\n", name, q, q /
        // thermalCap);

    }

    @Override
    protected String getExtraInfo() {
        return String.format(" c=%5.2f", thermalCap);
    }

}
