package org.lsst.ccs.subsystem.common.thermalsim;

import static java.lang.Math.pow;

/**
 * A radiative thermal link (Stefan's law). Area is given in m^2. Emissivity can
 * be adjusted and is dimensionless.
 * 
 * @author aubourg
 *
 */
public class RadiativeThermalLink extends ThermalLink {

    public RadiativeThermalLink(ThermalModel model, String name, double area) {
        super(model, name);
        this.area = area;
    }

    private final double sigma = 5.67E-8; // Stefan-Boltzman const. (W m−2 K−4)

    private double area; // area (m^2)

    private double emis1 = 0.06, emis2 = 0.06; // emissivity of each material.
    // Copper 0.06, Al 0.09

    @Override
    public double power(double t1, double t2) {
        return ((emis2 * emis2) / (emis1 + emis2 - emis1 * emis2)) * sigma * area * (pow(t1, 4) - pow(t2, 4));
    }

    public void setLeft(ThermalCapacitance e1, double emissivity) {
        super.setLeft(e1);
        emis1 = emissivity;
    }

    public void setRight(ThermalCapacitance e2, double emissivity) {
        super.setRight(e2);
        emis2 = emissivity;
    }

    public void setEmis1(double emis1) {
        this.emis1 = emis1;
    }

    public void setEmis2(double emis2) {
        this.emis2 = emis2;
    }

    @Override
    protected String getExtraInfo() {
        return String.format(" a=%5.2f", area);
    }

}
