package org.lsst.ccs.subsystem.common.thermalsim;

/**
 * A lumped bulk element with capacitance and resistance.
 * 
 * Builds a series of n RC circuits.
 * 
 * The leftmost element is the first capacitance, the rightmost element the last
 * resistance.
 * 
 * @author aubourg
 *
 */

public class LumpedThermalCapacitanceLink extends ThermalSimElement {

    public LumpedThermalCapacitanceLink(ThermalModel model, String name, double resistance, double capacitance, int nParts) {
        super(model, name);
        this.nParts = nParts;
        caps = new ThermalCapacitance[nParts];
        links = new ResistiveThermalLink[nParts];
        for (int i = 0; i < nParts; i++) {
            caps[i] = new ThermalCapacitance(model, name + "_c" + i, capacitance / nParts);
            links[i] = new ResistiveThermalLink(model, name + "_r" + i, resistance / nParts);
            // parts[i] = new ThermalCapacitanceLink(model, name + "_" + i,
            // resistance / nParts, capacitance / nParts);
            links[i].setLeft(caps[i]);
            if (i > 0) {
                links[i - 1].setRight(caps[i]);
            }
        }
    }

    int nParts;
    ThermalCapacitance[] caps;
    ResistiveThermalLink[] links;

    @Override
    protected ThermalSimElement getActualRight() {
        return links[nParts - 1];
    }

    @Override
    protected ThermalSimElement getActualLeft() {
        return caps[0];
    }

    @Override
    protected void thermalLoop(long dt) {
        // Don't do anything, the subparts will do it all
    }

    @Override
    public double getTemperature() {
        // The average temperature of the parts.
        double t = 0;
        for (ThermalCapacitance c : caps) {
            t += c.getTemperature();
        }
        return t / nParts;
    }

    public void setTemperature(double t) {
        for (ThermalCapacitance c : caps) {
            c.setTemperature(t);
        }
    }

    @Override
    protected String getExtraInfo() {
        return String.format(" c=%5.2f r=%5.2f n=%3d", caps[0].thermalCap * nParts, links[0].res * nParts, nParts);
    }

}
