package org.lsst.ccs.subsystem.common.thermalcontrol;

import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.function.DoubleConsumer;
import java.util.function.DoubleSupplier;

/**
 * PID Loop controller
 * 
 * The supplier provides a temperature (in K) The consumer accepts a power (in
 * W)
 * 
 * @author aubourg
 *
 */

public class PIDController {

    public PIDController(DoubleSupplier temperatureProbe, DoubleConsumer heater) {
        this.temperatureProbe = temperatureProbe;
        this.heater = heater;
    }

    DoubleSupplier temperatureProbe;
    DoubleConsumer heater;
    
    public DoubleSupplier getTemperatureProbe() {
        return temperatureProbe;
    }

    double target;

    public void setTarget(double target) {
        changeSetup();
        this.target = target;
    }

    public double getTarget() {
        return target;
    }

    double gain = 0.01;
    double derivativeTime = 1;
    double integralTime = 4 * derivativeTime;

    double maxCorr = 50;
    double minCorr = 0;

    double lastCorr = 0;
    long setupChangeTime = 0;

    boolean freezeOnUpdate = true;
    boolean progressiveParmUpdates = false; // TODO

    ScheduledExecutorService execService = Executors.newScheduledThreadPool(1);
    protected volatile ScheduledFuture<?> future = null;

    public void setFreezeOnUpdate(boolean freezeOnUpdate) {
        this.freezeOnUpdate = freezeOnUpdate;
    }

    public boolean isFreezeOnUpdate() {
        return freezeOnUpdate;
    }

    public void changeSetup() {
        setupChangeTime = System.currentTimeMillis();
        resetIntegral();
    }

    public double getGain() {
        return gain;
    }

    public void setGain(double gain) {
        this.gain = gain;
        changeSetup();
    }

    public double getIntegralTime() {
        return integralTime;
    }

    public void setIntegralTime(double integralTime) {
        this.integralTime = integralTime;
        changeSetup();
    }

    public double getDerivativeTime() {
        return derivativeTime;
    }

    public void setDerivativeTime(double derivativeTime) {
        this.derivativeTime = derivativeTime;
    }

    // derivative buffering ?

    double lastTime = 0;
    double lastErrorValue;
    double integral = 0;

    long minDelay = 2;

    long loopTime = 100; // ms

    public void resetIntegral() {
        integral = 0;
    }

    public void setMinCorr(double minCorr) {
        this.minCorr = minCorr;
    }

    public void setMaxCorr(double maxCorr) {
        this.maxCorr = maxCorr;
    }

    public double getMinCorr() {
        return minCorr;
    }

    public double getMaxCorr() {
        return maxCorr;
    }

    public void start() {
        future = execService.scheduleAtFixedRate(() -> process(), 0, loopTime, TimeUnit.MILLISECONDS);
    }

    public void stop() {
        if (future != null) {
            future.cancel(false);
            future = null;
        }
    }

    public void process() {
        long timeStamp = System.currentTimeMillis();
        if (timeStamp - lastTime < minDelay)
            return;
        double t = temperatureProbe.getAsDouble();
        double error = t - target;
        integral += error;
        double der = (error - lastErrorValue) / (timeStamp - lastTime);

        double corr = -gain * (error + integral / integralTime + der * derivativeTime);

        if (corr > maxCorr) {
            corr = maxCorr;
        }
        if (corr < minCorr) {
            corr = minCorr;
        }

        lastTime = timeStamp;
        lastErrorValue = error;

        if (freezeOnUpdate && (timeStamp - setupChangeTime < integralTime * 1000L / 4 || timeStamp - setupChangeTime < derivativeTime * 1000L / 4)) {
            corr = lastCorr;
        } else {
            lastCorr = corr;
        }

        System.out.printf("T %5.2f  ΔT %5.2f p %5.2f\n", t, error, corr);

        heater.accept(corr);
    }

}
