package org.lsst.ccs.subsystem.monitor;

import java.io.IOException;
import java.util.List;
import java.util.Map;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;

/**
 * A Class containing Monitor level commands. 
 * This class is intended to be added to an Agent command dictionary to provide
 * basic Monitor commands to interact with the underlying Monitor framework.
 * 
 * @author turri
 */
public class MonitorCommands {
    
    private final Monitor mon;
    
    /**
     * Public constructor for MonitoringCommands.
     * 
     * @param mon The Monitor object on which the commands will be executed.
     */
    public MonitorCommands(Monitor mon) {
        this.mon = mon;
    }
    
   /**
    ***************************************************************************
    **
    **  Saves the monitoring limits.
    **
    **  @throws  IOException
    **
    ***************************************************************************
    */
    @Command(type=Command.CommandType.ACTION, description="Save the monitoring limits")
    public void saveLimits() throws IOException
    {
        mon.saveLimits();
    }

    /**
    ***************************************************************************
    **
    **  Gets the list of all device names.
    **
    **  @return  The list of device names
    **
    ***************************************************************************
    */
    @Command(type=Command.CommandType.QUERY, description="Get the list of all devices")
    public List<String> getDeviceNames()
    {
        return mon.getDeviceNames();
    }

   /**
    ***************************************************************************
    **
    **  Gets the list of channel names.
    **
    **  @return  The list of channel names
    **
    ***************************************************************************
    */
    @Command(type=Command.CommandType.QUERY, description="Get the list of channel names")
    public List<String> getChannelNames()
    {
        return mon.getChannelNames();
    }

   /**
    ***************************************************************************
    **
    **  Gets a channel value.
    **
    **  @param  name  The channel name
    **
    **  @return  The last-read channel value
    **
    **  @throws  IllegalArgumentException if the channel doesn't exist
    **
    ***************************************************************************
    */
    @Command(type=Command.CommandType.QUERY, description="Get a channel value")
    public double getChannelValue(@Argument(name="name", description="Channel name")
                                  String name)
    {
        Double value = mon.getChannelValue(name);
        if (value == null) {
            throw new IllegalArgumentException("Unknown channel name: " + name);
        }
        return value;
    }

   /**
    ***************************************************************************
    **
    **  Reads a channel value.
    **
    **  @param  name  The channel name
    **
    **  @return  The read channel value, or NaN if the data isn't available
    **
    **  @throws  IllegalArgumentException if the channel doesn't exist
    **
    ***************************************************************************
    */
    @Command(type=Command.CommandType.QUERY, description="Read a channel value")
    public double readChannelValue(@Argument(name="name", description="Channel name")
                                   String name)
    {
        Double value = mon.readChannelValue(name);
        if (value == null) {
            throw new IllegalArgumentException("Unknown channel name: " + name);
        }
        return value;
    }

   /**
    ***************************************************************************
    **
    **  Gets the map of configured values from all devices.
    **
    **  @return  The map of configured values
    **
    ***************************************************************************
    */
    @Command(type=Command.CommandType.QUERY,
             description="Get the map of all configured values")
    public Map<String, Object> getConfigurationMap()
    {
        return mon.getConfigurationMap();
    }

}
