package org.lsst.ccs.subsystem.monitor;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 *****************************************
 *
 *  "Device" for performing calculations.
 *
 *  @author Owen Saxton
 *
 *****************************************
 */
public class CalcDevice extends Device {

   /**
    *  Private fields.
    */
    private final static Map<String, Integer> typeMap = new HashMap<>();
    static {
        typeMap.put("SUM",  Channel.TYPE_SUM);
        typeMap.put("DIFF", Channel.TYPE_DIFF);
        typeMap.put("PROD", Channel.TYPE_PROD);
        typeMap.put("QUOT", Channel.TYPE_QUOT);
    }
    private final List<String> subtypeList = new ArrayList<>();
    private final List<Integer> chanList = new ArrayList<>();
    private int keyValue = 0;


   /**
    *  Performs basic initialization.
    */
    @Override
    protected void initialize()
    {
        initSensors();
        setOnline(true);
    }


   /**
    *  Closes the connection.
    */
    @Override
    protected void close()
    {
    }


   /**
    *  Checks a channel's parameters for validity.
    *
    *  @param  name     The name of the channel.
    *
    *  @param  hwChan   The hardware channel number.
    *
    *  @param  type     The channel type string.
    *
    *  @param  subtype  The channel subtype string.
    *
    *  @return  A two-element array containing the encoded type [0] and
    *           subtype [1].
    *
    *  @throws  Exception if any errors found in the parameters
    */
    @Override
    protected int[] checkChannel(String name, int hwChan, String type,
                                 String subtype) throws Exception
    {
        Integer iType = typeMap.get(type.toUpperCase());
        if (iType == null) {
            mon.reportError(name, "type", type);
        }
        subtypeList.add(keyValue, subtype);

        return new int[]{(iType << 16) | keyValue++, 0};
    }


   /**
    *  Completes the initialization of a calculation channel.
    *
    *  Has to be done here, after all channels have been configured
    *
    *  @param  name     The channel name
    *
    *  @param  id       The channel ID
    *
    *  @param  hwChan   The hardware channel number.
    *
    *  @param  type     The encoded channel type returned by checkChannel.
    *
    *  @param  subtype  The channel subtype returned by checkChannel.
    */
    @Override
    protected void initChannel(String name, int id, int hwChan, int type,
                               int subtype)
    {
        int key = type & 0xffff;
        String subtypeS = subtypeList.get(key);
        String[] names = subtypeS.split(":", -1);
        try {
            int id0 = 0, id1 = 0;
            if (names.length != 2
                  || (id0 = mon.getChannelId(names[0])) < 0
                  || (id1 = mon.getChannelId(names[1])) < 0) {
                mon.reportError(name, "subtype", subtypeS);
            }
            chanList.add(key, (id0 << 16) | id1);
        }
        catch (Exception e) {
            dropChannel(id);
        }
    }


   /**
    *  Reads a channel.
    *
    *  @param  hwChan   The hardware channel number.
    *
    *  @param  type     The encoded channel type.
    *
    *  @return  The value read from the channel
    */
    @Override
    protected double readChannel(int hwChan, int type)
    {
        int chans = chanList.get(type & 0xffff);
        Channel chan1 = mon.getChannel(chans >> 16);
        Channel chan2 = mon.getChannel(chans & 0xffff);
        double value = Double.NaN;
        if (chan1.getState() != Channel.STATE_OFFLINE
              && chan2.getState() != Channel.STATE_OFFLINE) {
            switch (type >> 16) {
            case Channel.TYPE_SUM:
                value = chan1.getValue() + chan2.getValue();
                break;
            case Channel.TYPE_DIFF:
                value = chan1.getValue() - chan2.getValue();
                break;
            case Channel.TYPE_PROD:
                value = chan1.getValue() * chan2.getValue();
                break;
            case Channel.TYPE_QUOT:
                value = chan1.getValue() / chan2.getValue();
                break;
            }
        }

        return value;
    }

}
