package org.lsst.ccs.subsystem.monitor;

import java.lang.reflect.Constructor;
import java.util.BitSet;
import java.util.LinkedHashMap;
import java.util.HashMap;
import java.util.Map;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.command.annotations.Command.CommandType;
import org.lsst.ccs.framework.Configurable.Environment;
import org.lsst.ccs.framework.ConfigurableComponent;
import org.lsst.ccs.utilities.logging.Logger;

/**
 ******************************************************************************
 **
 **  Handles hardware devices
 **
 **  @author Owen Saxton
 **
 ******************************************************************************
 */
public abstract class Device extends ConfigurableComponent {

   /**
    ***************************************************************************
    **
    **  Data fields.
    **
    ***************************************************************************
    */
    protected String     fullName;   // Full device name
    protected BitSet     chanMask;   // Channel IDs on this device
    protected boolean    disabled;   // True if disabled
    protected boolean    inited;     // True if initialized
    protected boolean    online;     // True if online (running successfully)
    protected Monitor    mon;        // Associated monitor object
    protected Logger     log;        // The logger
    protected Map<String, Control>
                         ctlChans;   // Control channels
    protected int        lineMask;   // Mask of used lines
    protected int        lineWarm;   // Mask of warm-start lines
    protected int        lineState;  // Mask of set lines


   /**
    ***************************************************************************
    **
    **  Gets the full name.
    **
    **  @return  The full name of the device
    **
    ***************************************************************************
    */
    public String getFullName()
    {
        return fullName;
    }


   /**
    ***************************************************************************
    **
    **  Gets the online state.
    **
    **  @return  Whether or not the device is online
    **
    ***************************************************************************
    */
    public boolean isOnline()
    {
        return online;
    }


   /**
    ***************************************************************************
    **
    **  Disables the device.
    **
    ***************************************************************************
    */
    public void disable()
    {
        disabled = true;
        setOnline(false);
    }


   /**
    ***************************************************************************
    **
    **  Enables the device.
    **
    ***************************************************************************
    */
    public void enable()
    {
        disabled = false;
        checkOnline();
    }


   /**
    ***************************************************************************
    **
    **  Performs common configuration.
    **
    **  Can be overridden if necessary, but must be called by overriding code.
    **
    **  @param  mon  The associated monitor
    **
    ***************************************************************************
    */
    protected void configure(Monitor mon)
    {
        this.mon = mon;
        log = mon.log;
        chanMask = new BitSet(mon.getNumChans());
        Environment ce = getEnvironment();
        if (ce != null) {
            ctlChans = getChildren(Control.class);
            for (Control ctl : ctlChans.values()) {
                ctl.configure(mon, this);
            }
        }
        else {
            ctlChans = new HashMap(); 
        }
    }


   /**
    ***************************************************************************
    **
    **  Gets the map of configured items.
    **
    **  @return  A map of the configured items linking name and value.
    **
    ***************************************************************************
    */
    @Command(type=CommandType.QUERY,
             description="Gets map of configured items")
    public Map<String, Object> getConfigValues()
    {
        Map<String, Object> map = new LinkedHashMap<>();
        addConfigValues(map);
        for (Control ctl : ctlChans.values()) {
            ctl.addConfigValues(map);
        }

        return map;
    }


   /**
    ***************************************************************************
    **
    **  Checks a monitor channel's parameters for validity.
    **
    **  This method is to be overridden for devices with monitor channels that
    **  have descriptive information to be checked and encoded.
    **
    **  @param  name     The name of the channel.
    **
    **  @param  hwChan   The hardware channel number.
    **
    **  @param  type     The channel type string.
    **
    **  @param  subtype  The channel subtype string.
    **
    **  @return  A two-element array containing the encoded type [0] and
    **           subtype [1].
    **
    **  @throws  Exception if any errors found in the parameters
    **
    ***************************************************************************
    */
    protected int[] checkChannel(String name, int hwChan, String type,
                                 String subtype) throws Exception
    {
        return new int[]{0, 0};
    }


   /**
    ***************************************************************************
    **
    **  Performs hardware initialization.
    **
    ***************************************************************************
    */
    protected abstract void initialize();


   /**
    ***************************************************************************
    **
    **  Closes the connection.
    **
    ***************************************************************************
    */
    protected abstract void close();


   /**
    ***************************************************************************
    **
    **  Initializes all attached sensors.
    **
    ***************************************************************************
    */
    protected void initSensors()
    {
        for (int id = chanMask.nextSetBit(0); id >= 0;
             id = chanMask.nextSetBit(id + 1)) {
            mon.getChannel(id).initSensor();
        }
    }


   /**
    ***************************************************************************
    **
    **  Initializes a monitor channel.
    **
    **  This method is to be overridden for devices with monitor channels that
    **  need to be configured in the hardware.
    **
    **  @param  hwChan   The hardware channel number.
    **
    **  @param  type     The encoded channel type returned by checkChannel.
    **
    **  @param  subtype  The channel subtype returned by checkChannel.
    **
    ***************************************************************************
    */
    protected void initChannel(int hwChan, int type, int subtype)
    {
    }


   /**
    ***************************************************************************
    **
    **  Initializes a monitor channel.
    **
    **  This method is to be overridden for devices with monitor channels that
    **  need to be configured in the hardware.
    **
    **  @param  name     The channel name
    **
    **  @param  id       The channel ID
    **
    **  @param  hwChan   The hardware channel number.
    **
    **  @param  type     The encoded channel type returned by checkChannel.
    **
    **  @param  subtype  The channel subtype returned by checkChannel.
    **
    ***************************************************************************
    */
    protected void initChannel(String name, int id, int hwChan, int type,
                               int subtype)
    {
        initChannel(hwChan, type, subtype);  // Call the old form by default
    }


   /**
    ***************************************************************************
    **
    **  Adds a channel to the device's list of monitored channels.
    **
    **  @param  id  The id of the channel.
    **
    ***************************************************************************
    */
    protected void addChannel(int id)
    {
        chanMask.set(id);
    }


   /**
    ***************************************************************************
    **
    **  Drops a channel from the device's list of monitored channels.
    **
    **  The monitor's list must be updated as well, and the channel itself
    **  made invalid.
    **
    **  @param  id  The id of the channel.
    **
    ***************************************************************************
    */
    protected void dropChannel(int id)
    {
        chanMask.clear(id);
        BitSet mask = new BitSet(id);
        mask.set(id);
        mon.setOnline(mask, false);
        mon.getChannel(id).setValid(false);
    }


   /**
    ***************************************************************************
    **
    **  Checks the online state.
    **
    **  If offline, an attempt is made to re-initialize the device.
    **
    ***************************************************************************
    */
    public void checkOnline()
    {
        if (online || disabled) return;
        initialize();
    }


   /**
    ***************************************************************************
    **
    **  Tests the online state.
    **
    **  If offline, an error message is logged.
    **
    **  @return  Whether or not the device is online.
    **
    ***************************************************************************
    */
    protected boolean testOnline()
    {
        if (!online) {
            log.error("Device " + fullName + " is offline");
        }
        return online;
    }


   /**
    ***************************************************************************
    **
    **  Sets the online state.
    **
    **  If the device is set offline, an error message is logged.
    **
    **  @param  online  The online state to set: true or false
    **
    ***************************************************************************
    */
    protected void setOnline(boolean online)
    {
        if (this.online == online) return;
        this.online = online;
        mon.setOnline(chanMask, online);
        if (!online) {
            if (!disabled) {
                log.error("Disconnected from " + fullName);
            }
            close();
        }
    }


   /**
    ***************************************************************************
    **
    **  Reads all attached sensors (monitor channels).
    **
    ***************************************************************************
    */
    protected void readSensors()
    {
        readChannelGroup();
        for (int id = chanMask.nextSetBit(0); id >= 0;
             id = chanMask.nextSetBit(id + 1)) {
            mon.getChannel(id).readSensor();
        }
    }


   /**
    ***************************************************************************
    **
    **  Reads the monitor channel group.
    **
    **  This method is to be overridden for devices that reads some (or all)
    **  of its monitor channels at one time.
    **
    ***************************************************************************
    */
    protected void readChannelGroup()
    {
    }


   /**
    ***************************************************************************
    **
    **  Reads a monitor channel.
    **
    **  This method is to be overridden for devices with monitor channels.
    **
    **  @param  hwChan   The hardware channel number.
    **
    **  @param  type     The encoded channel type returned by checkChannel.
    **
    **  @return  The value read from the channel
    **
    ***************************************************************************
    */
    protected double readChannel(int hwChan, int type)
    {
        return Double.NaN;
    }


   /**
    ***************************************************************************
    **
    **  Reads a monitor channel immediately.
    **
    **  This method is to be overridden for devices with monitor channels.
    **  It is used when the instantaneous value of the channel is needed
    **  rather than the one obtained during a previous readChannelGroup call.
    **
    **  @param  hwChan   The hardware channel number.
    **
    **  @param  type     The encoded channel type returned by checkChannel.
    **
    **  @return  The value read from the channel
    **
    ***************************************************************************
    */
    protected double readChannelNow(int hwChan, int type)
    {
        return readChannel(hwChan, type);
    }


   /**
    ***************************************************************************
    **
    **  Adds to the map of configured items.
    **
    **  This method is to be overridden if necessary
    **
    **  @param  map  The map of configured items to be added to
    **
    ***************************************************************************
    */
    protected void addConfigValues(Map<String, Object> map)
    {
    }


   /**
    ***************************************************************************
    **
    **  Checks whether exception is a timeout.
    **
    **  If it is a timeout, the device is set offline.  In all cases an
    **  exception of the supplied class is thrown, containing the message of
    **  the original exception.
    **
    **  @param  e       The exception to be checked.
    **
    **  @param  eClass  The class of the exception to be thrown.
    **
    **  @throws  Exception
    **
    ***************************************************************************
    */
    protected void checkTimeout(Exception e, Class<?> eClass) throws Exception
    {
        if (isTimeout(e)) {
            setOnline(false);
        }
        Constructor<?> eCon = eClass.getConstructor(String.class);
        throw (Exception)eCon.newInstance(e.getMessage());
    }


   /**
    ***************************************************************************
    **
    **  Tests whether exception is a timeout.
    **
    **  @param  e  The exception to be tested.
    **
    **  @return  Whether or not the exception is a timeout.
    **
    ***************************************************************************
    */
    protected boolean isTimeout(Exception e)
    {
        return e.getMessage().matches(".*time.*out.*");
    }


   /**
    ***************************************************************************
    **
    **  Checks the validity of an output line.
    **
    **  @param  name  The name of the output line.
    **
    **  @param  line  The hardware line number of the output line.
    **
    **  @throws  Exception if the line number is invalid.
    **
    ***************************************************************************
    */
    protected void checkLine(String name, int line) throws Exception
    {
        checkHwLine(name, line);
    }


   /**
    ***************************************************************************
    **
    **  Checks the validity of an output line.
    **
    **  This method is to be overridden for devices with output lines.
    **
    **  @param  name  The name of the output line.
    **
    **  @param  line  The hardware line number of the output line.
    **
    **  @throws  Exception if the line number is invalid.
    **
    ***************************************************************************
    */
    protected void checkHwLine(String name, int line) throws Exception
    {
        mon.reportError(name, "device", getName());
    }


   /**
    ***************************************************************************
    **
    **  Sets the warm-start state of an output line.
    **
    **  @param  line  The output line number.
    **
    **  @param  on    The warm-start on state to set: true or false
    **
    ***************************************************************************
    */
    protected void setLineWarm(int line, boolean on)
    {
        if (on) {
            lineWarm |= (1 << line);
        }
        else {
            lineWarm &= ~(1 << line);
        }
    }


   /**
    ***************************************************************************
    **
    **  Sets an output line on or off.
    **
    **  @param  line  The output line number.
    **
    **  @param  on    The on state to set: true or false
    **
    ***************************************************************************
    */
    protected void setLine(int line, boolean on)
    {
        if (on) {
            lineState |= (1 << line);
        }
        else {
            lineState &= ~(1 << line);
        }
        setHwLine(line, on);
    }


   /**
    ***************************************************************************
    **
    **  Sets an output line on or off.
    **
    **  This method is to be overridden for devices with output lines.
    **
    **  @param  line  The output line number.
    **
    **  @param  on    The on state to set: true or false
    **
    ***************************************************************************
    */
    protected void setHwLine(int line, boolean on)
    {
    }


   /**
    ***************************************************************************
    **
    **  Gets an output line set state.
    **
    **  @param  line  The output line number.
    **
    **  @return  Whether or not the line is set.
    ** 
    ***************************************************************************
    */
    protected boolean isLineSet(int line)
    {
        return isHwLineSet(line);
    }


   /**
    ***************************************************************************
    **
    **  Gets an output line set state.
    **
    **  This method is to be overridden for devices with output lines.
    **
    **  @param  line  The output line number.
    **
    **  @return  Whether or not the line is set.
    ** 
    ***************************************************************************
    */
    protected boolean isHwLineSet(int line)
    {
        return false;
    }


   /**
    ***************************************************************************
    **
    **  Sets all output lines from the saved state.
    **
    ***************************************************************************
    */
    protected void setOutputLines()
    {
        for (int mask = lineMask, line = 0; mask != 0; mask >>>= 1, line++) {
            if ((mask & 1) != 0) {
                if ((lineWarm & 1 << line) == 0) {
                    setLine(line, (lineState & 1 << line) != 0);
                }
                else {
                    setLine(line, isHwLineSet(line));
                }
            }
        }
    }


   /**
    ***************************************************************************
    **
    **  Gets the states of all output lines and saves them.
    **
    ***************************************************************************
    */
    protected void getOutputLines()
    {
        lineState = 0;
        for (int mask = lineMask, line = 0; mask != 0; mask >>>= 1, line++) {
            if ((mask & 1) != 0) {
                lineState |= isHwLineSet(line) ? 1 << line : 0;
            }
        }
    }


   /**
    ***************************************************************************
    **
    **  Adds an output line to the known list.
    **
    **  @param  line  The output line number.
    **
    ***************************************************************************
    */
    protected void addLine(int line)
    {
        lineMask |= 1 << line;
    }

}
