package org.lsst.ccs.subsystem.monitor;

import java.util.BitSet;
import java.util.HashMap;
import java.util.IllegalFormatException;
import java.util.Map;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.data.KeyValueDataList;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.ConfigurationParameterChanger;
import org.lsst.ccs.framework.ConfigurableComponent;

/**
 ******************************************************************************
 **
 **  Handles a monitoring channel.
 **
 **  @author Owen Saxton
 **
 ******************************************************************************
 */
public class Channel extends ConfigurableComponent {

   /**
    ***************************************************************************
    **
    **  Constants.
    **
    ***************************************************************************
    */
    public final static int
        TYPE_UNKNOWN = -1,
        TYPE_TEMP    = 0,
        TYPE_VOLTS   = 1,
        TYPE_POWER   = 2,
        TYPE_SWITCH  = 3,
        TYPE_SUM     = 4,
        TYPE_DIFF    = 5,
        TYPE_PROD    = 6,
        TYPE_QUOT    = 7;
    
    public final static int
        STATE_OFFLINE = -1,
        STATE_GOOD    = 0,
        STATE_WARNING = 1,
        STATE_ERROR   = 2;

    private final static int
        LIMIT_CHECK_NONE  = 0,   // Don't check limit
        LIMIT_CHECK_FLAG  = 1,   // Flag if limit exceeded
        LIMIT_CHECK_ALARM = 2;   // Alarm if limit exceeded

    private final String DEFAULT_FORMAT = "%.2f ";

   /**
    ***************************************************************************
    **
    **  Private lookup maps.
    **
    ***************************************************************************
    */
    private final static Map<String,Integer> checkMap = new HashMap<>();
    static {
        checkMap.put("NONE",  LIMIT_CHECK_NONE);
        checkMap.put("FLAG",  LIMIT_CHECK_FLAG);
        checkMap.put("ALARM", LIMIT_CHECK_ALARM);
    }

   /**
    ***************************************************************************
    **
    **  Data fields.
    **
    ***************************************************************************
    */
    // Supplied immutable fields
    private String  description = "";   // Channel description
    private String  units = "";         // Units name
    private String  devcName;           // The name of the associated device (required)
    private int     hwChan = 0;         // Hardware channel
    private String  type = "";          // Channel type string (TEMP, VOLTS, POWER...)
    private String  subtype = "";       // Channel subtype string (extra config info)
    private double  offset = 0.0;       // Value offset
    private double  scale = 1.0;        // Value scale
    private String  checkLo = "FLAG";   // Low limit check option string
    private String  alarmLo;            // Low limit alarm name (null = none)
    private double  dbandLo = 0.0;      // Low limit deadband
    private String  checkHi = "FLAG";   // High limit check option string
    private String  alarmHi;            // High limit alarm name (null = none)
    private double  dbandHi = 0.0;      // High limit deadband

    // Supplied changeable or configurable fields
    private String  format = DEFAULT_FORMAT;  // Format string for display
    @ConfigurationParameter(name="limitLo", category=Monitor.LIMITS)
    private double  limitLo = 0.0;      // Low limit value
    @ConfigurationParameter(name="limitHi", category=Monitor.LIMITS)
    private double  limitHi = 0.0;      // High limit value

    // Derived immutable fields
    private String  name;          // Channel name
    private Device  devc;          // The associated device
    private int     typeI;         // Encoded channel type
    private int     subtypeI;      // Encoded channel subtype
    private int     checkLoI;      // Low limit check option
    private Alarm   alarmLoA;      // Low limit alarm (null = none)
    private int     checkHiI;      // High limit check option
    private Alarm   alarmHiA;      // High limit alarm (null = none)
    private int     id;            // External ID
    private boolean valid;         // True if configured correctly
    private Monitor mon;           // Associated monitor object

    // Volatile fields
    private double  value;         // Last value read from sensor
    private long    timestamp;     // Time of last successful read
    private int     state = STATE_OFFLINE; // State of the channel
    private double  resetLo = 0.0; // Low limit reset value
    private double  resetHi = 0.0; // High limit reset value
    private boolean ignrAlarmLo;   // True if low limit alarm ignored
    private boolean ignrAlarmHi;   // True if high limit alarm ignored


   /**
    ***************************************************************************
    **
    **  Constructor.
    **
    **  @param  desc        The descriptive name of the channel.
    **
    **  @param  fmt         The format string for displaying the value.
    **
    **  @param  unit        The units name for the channel.
    **
    **  @param  devName     The name of the device containing the channel.
    **
    **  @param  chan        The hardware channel number of the channel.
    **
    **  @param  type        The type of the channel.
    **
    **  @param  subtype     The sub-type of the channel.
    **
    **  @param  offs        The offset to be applied to the read value.
    **
    **  @param  scal        The scale factor for the read value.
    **
    **  @param  checkLo     The type of check against the low limit.
    **
    **  @param  limLo       The low limit value.
    **
    **  @param  deadbandLo  The low limit deadband value.
    **
    **  @param  alarmLo     The name of the alarm for the low limit.
    **
    **  @param  checkHi     The type of check against the high limit.
    **
    **  @param  limHi       The high limit value.
    **
    **  @param  deadbandHi  The high limit deadband value.
    **
    **  @param  alarmHi     The name of the alarm for the high limit.
    **
    ***************************************************************************
    */
    @Deprecated
    public Channel(String desc, String fmt, String unit, String devName,
                   int chan, String type, String subtype, double offs,
                   double scal, String checkLo, double limLo, double deadbandLo,
                   String alarmLo, String checkHi, double limHi,
                   double deadbandHi, String alarmHi)
    {
        description = desc;
        setFormat(fmt);
        units       = unit;
        setDevcName(devName);
        hwChan      = chan;
        this.type       = type;
        this.subtype    = subtype;
        offset      = offs;
        scale       = scal;
        this.checkLo    = checkLo;
        limitLo     = limLo;
        dbandLo     = deadbandLo;
        this.alarmLo    = alarmLo;
        this.checkHi    = checkHi;
        limitHi     = limHi;
        dbandHi     = deadbandHi;
        this.alarmHi    = alarmHi;
    }


    public Channel() {
    }
    

   /**
    ***************************************************************************
    **
    **  Constructor.
    **
    **  This is the old version, which doesn't set the format string.
    **
    **  @param  desc        The descriptive name of the channel.
    **
    **  @param  unit        The units name for the channel.
    **
    **  @param  devName     The name of the device containing the channel.
    **
    **  @param  chan        The hardware channel number of the channel.
    **
    **  @param  type        The type of the channel.
    **
    **  @param  subtype     The sub-type of the channel.
    **
    **  @param  offs        The offset to be applied to the read value.
    **
    **  @param  scal        The scale factor for the read value.
    **
    **  @param  checkLo     The type of check against the low limit.
    **
    **  @param  limLo       The low limit value.
    **
    **  @param  deadbandLo  The low limit deadband value.
    **
    **  @param  alarmLo     The name of the alarm for the low limit.
    **
    **  @param  checkHi     The type of check against the high limit.
    **
    **  @param  limHi       The high limit value.
    **
    **  @param  deadbandHi  The high limit deadband value.
    **
    **  @param  alarmHi     The name of the alarm for the high limit.
    **
    ***************************************************************************
    */
    @Deprecated
    public Channel(String desc, String unit, String devName, int chan,
                   String type, String subtype, double offs, double scal,
                   String checkLo, double limLo, double deadbandLo,
                   String alarmLo, 
                   String checkHi, double limHi, double deadbandHi,
                   String alarmHi)
    {
        this(desc, null, unit, devName, chan, type, subtype, offs, scal,
             checkLo, limLo, deadbandLo, alarmLo, checkHi, limHi, deadbandHi,
             alarmHi);
    }

    // Setter methods used when building the object


    /**
     * Sets the format string for this Channel.
     * 
     * @param fmt The provided format to be set
     */
    public final void setFormat(String fmt) {
        format = (fmt == null || fmt.isEmpty()) ? DEFAULT_FORMAT
                   : ((fmt.charAt(0) != '%') ? "%" + fmt : fmt) + " ";        
    }
    

    /**
     * Sets the name of the Device to corresponding to this Channel.
     * 
     * @param devName The name of the Device as defined in the groovy description node.
     */
    public final void setDevcName(String devName) {
        devcName = devName == null ? Monitor.CALC_DEVICE_NAME : devName;        
    }


    /**
    ***************************************************************************
    **
    **  Sets the low limit.
    **
    **  @param  limit  The low limit value to set.
    **
    ***************************************************************************
    */
    @ConfigurationParameterChanger
    public void setLimitLo(double limit)
    {
        if (limit == limitLo) return;
        limitLo = limit;
        setResetLo();
        if (valid) {
            mon.loLimChange.set(id);
            KeyValueDataList data = new KeyValueDataList(name);
            data.addData("alarmLow", String.valueOf(limit),
                         KeyValueData.KeyValueDataType.KeyValueMetaData);
            mon.subsys.publishSubsystemDataOnStatusBus(data);
            mon.publishState();
        }
    }


   /**
    ***************************************************************************
    **
    **  Sets the high limit.
    **
    **  @param  limit  The high limit value to set.
    **
    ***************************************************************************
    */
    @ConfigurationParameterChanger
    public void setLimitHi(double limit)
    {
        if (limit == limitHi) return;
        limitHi = limit;
        setResetHi();
        if (valid) {
            mon.hiLimChange.set(id);
            KeyValueDataList data = new KeyValueDataList(name);
            data.addData("alarmHigh", String.valueOf(limit),
                         KeyValueData.KeyValueDataType.KeyValueMetaData);
            mon.subsys.publishSubsystemDataOnStatusBus(data);
            mon.publishState();
        }
    }


    /**
     * Sets the low deadband value
     * 
     * @param  value  The value to set
     */
    public void setDbandLo(double value) {
        dbandLo = value;
        setResetLo();
    }


    /**
     * Sets the high deadband value
     * 
     * @param  value  The value to set
     */
    public void setDbandHi(double value) {
        dbandHi = value;
        setResetHi();
    }


    /**
     * Sets the low limit reset value
     */
    private void setResetLo() {
        resetLo = dbandLo <= 0 ? limitLo : limitLo + dbandLo;
    }


    /**
     * Sets the high limit reset value
     */
    private void setResetHi() {
        resetHi = dbandHi <= 0 ? limitHi : limitHi - dbandHi;
    }
    

   /**
    ***************************************************************************
    **
    **  Configures channel description.
    **
    ***************************************************************************
    */
    void configure(Monitor mon, int id)
    {
        name = getName();
        this.mon = mon;
        this.id = id;

        checkLoI = LIMIT_CHECK_FLAG;
        checkHiI = LIMIT_CHECK_FLAG;
        typeI    = TYPE_UNKNOWN;
        subtypeI = -1;

        try {
            if (devcName == null) {
                mon.reportError(name, "device name");
            }
            devc = mon.getDevice(devcName);
            if (devc == null) {
                mon.reportError(name, "device name", devcName);
            }

            String origFmt = format;
            try {
                String temp = String.format(format, 0.0);
            }
            catch (IllegalFormatException e) {
                format = DEFAULT_FORMAT;
                mon.reportError(name, "display format", origFmt);
            }

            Integer iCheck = checkMap.get(checkLo.toUpperCase());
            if (iCheck == null) {
                mon.reportError(name, "low check option", checkLo);
            }
            checkLoI = iCheck;
            iCheck = checkMap.get(checkHi.toUpperCase());
            if (iCheck == null) {
                mon.reportError(name, "high check option", checkHi);
            }
            checkHiI = iCheck;

            if (alarmLo != null) {
                alarmLoA = mon.getAlarm(alarmLo);
            }
            if (alarmLoA == null && (checkLoI == LIMIT_CHECK_ALARM || alarmLo != null)) {
                mon.reportError(name, "low alarm name", alarmLo);
            }
            if (alarmHi != null) {
                alarmHiA = mon.getAlarm(alarmHi);
            }
            if (alarmHiA == null && (checkHiI == LIMIT_CHECK_ALARM || alarmHi != null)) {
                mon.reportError(name, "high alarm name", alarmHi);
            }

            int[] types = devc.checkChannel(name, hwChan, type, subtype);
            typeI = types[0];
            subtypeI = types[1];

            valid = true;
            devc.addChannel(id);
        }
        catch (Exception e) {
            // Always thrown by mon.reportError
        }
    }


   /**
    ***************************************************************************
    **
    **  Completes configuration of calculation channel description.
    **
    **  This has to be done after all other channel configuration.
    **
    ***************************************************************************
    */
    void configCalc()
    {
        String[] names = subtype.split(":", -1);
        int id0 = 0, id1 = 0;
        try {
            if (names.length != 2
                  || (id0 = mon.getChannelId(names[0])) < 0
                  || (id1 = mon.getChannelId(names[1])) < 0) {
                mon.reportError(name, "subtype", subtype);
            }
            subtypeI = (id0 << 16) | id1;
        }
        catch (Exception e) {
            valid = false;
            devc.dropChannel(id);
        }
    }


   /**
    ***************************************************************************
    **
    **  Initializes the hardware sensor.
    **
    ***************************************************************************
    */
    void initSensor()
    {
        if (!valid) return;
        devc.initChannel(name, id, hwChan, typeI, subtypeI);
    }


   /**
    ***************************************************************************
    **
    **  Reads the hardware sensor.
    **
    ***************************************************************************
    */
    void readSensor()
    {
        double val;
        if (!valid || !devc.isOnline()) {
            val = Double.NaN;
        }
        else {
            val = devc.readChannel(hwChan, typeI);
        }
        if (!Double.isNaN(val)) {
            value = offset + scale * val;
            timestamp = System.currentTimeMillis();
            state = STATE_GOOD;
        }
        else {
            state = STATE_OFFLINE;
        }
    }


   /**
    ***************************************************************************
    **
    **  Checks whether current value is within limits.
    **
    **  @param  cState  The channel state to be updated
    **
    ***************************************************************************
    */
    void checkLimits(BitSet cState)
    {
        if (!valid || state == STATE_OFFLINE) return;
        boolean ok = true;
        if (value < limitLo) {
            if (checkLoI != LIMIT_CHECK_NONE) {
                ok = false;
                if (checkLoI == LIMIT_CHECK_ALARM && !ignrAlarmLo) {
                    alarmLoA.updateState(Alarm.STATE_ERROR);
                }
            }
        }
        else if (value > limitHi) {
            if (checkHiI != LIMIT_CHECK_NONE) {
                ok = false;
                if (checkHiI == LIMIT_CHECK_ALARM && !ignrAlarmHi) {
                    alarmHiA.updateState(Alarm.STATE_ERROR);
                }
            }
        }
        else {
            if (checkLoI == LIMIT_CHECK_ALARM && !ignrAlarmLo) {
                alarmLoA.updateState(value >= resetLo ? Alarm.STATE_GOOD
                                                      : Alarm.STATE_DEADBAND);
            }
            if (checkHiI == LIMIT_CHECK_ALARM && !ignrAlarmHi) {
                alarmHiA.updateState(value <= resetHi ? Alarm.STATE_GOOD
                                                      : Alarm.STATE_DEADBAND);
            }
        }
        if (ok) {
            cState.set(id);
            state = STATE_GOOD;
        }
        else {
            state = STATE_ERROR;
        }
    }


   /**
    ***************************************************************************
    **
    **  Checks whether channel is online.
    **
    **  @param  cState  The channel state to be updated
    **
    ***************************************************************************
    */
    void checkOnline(BitSet cState)
    {
        cState.set(id, valid && state != STATE_OFFLINE);
    }


   /**
    ***************************************************************************
    **
    **  Gets the channel description.
    **
    **  @return  The channel description
    **
    ***************************************************************************
    */
    String getDescription()
    {
        return description;
    }


   /**
    ***************************************************************************
    **
    **  Gets the display format.
    **
    **  @return  The channel description
    **
    ***************************************************************************
    */
    String getFormat()
    {
        return format;
    }


   /**
    ***************************************************************************
    **
    **  Gets the channel units.
    **
    **  @return  The channel units
    **
    ***************************************************************************
    */
    String getUnits()
    {
        return units;
    }


   /**
    ***************************************************************************
    **
    **  Gets the channel low limit.
    **
    **  @return  The channel low limit
    **
    ***************************************************************************
    */
    double getLimitLo()
    {
        return limitLo;
    }


   /**
    ***************************************************************************
    **
    **  Gets the channel low limit alarm name.
    **
    **  @return  The channel low limit alarm name
    **
    ***************************************************************************
    */
    String getAlarmNameLo()
    {
        return checkLoI == LIMIT_CHECK_ALARM ? alarmLo
                 : checkLoI == LIMIT_CHECK_FLAG ? "" : null;
    }


   /**
    ***************************************************************************
    **
    **  Gets the channel high limit.
    **
    **  @return  The channel high limit
    **
    ***************************************************************************
    */
    double getLimitHi()
    {
        return limitHi;
    }


   /**
    ***************************************************************************
    **
    **  Gets the channel high limit alarm name.
    **
    **  @return  The channel high limit alarm name
    **
    ***************************************************************************
    */
    String getAlarmNameHi()
    {
        return checkHiI == LIMIT_CHECK_ALARM ? alarmHi
                 : checkHiI == LIMIT_CHECK_FLAG ? "" : null;
    }


   /**
    ***************************************************************************
    **
    **  Gets the channel ID.
    **
    **  @return  The channel ID
    **
    ***************************************************************************
    */
    int getId()
    {
        return id;
    }


   /**
    ***************************************************************************
    **
    **  Enables or disables an alarm.
    **
    **  @param  isLow   True for low limit alarm, false for high.
    **
    **  @param  enable  True to enable the alarm, false to disable it.
    **
    ***************************************************************************
    */
    public void enableAlarm(boolean isLow, boolean enable)
    {
        if (isLow) {
            ignrAlarmLo = !enable;
        }
        else {
            ignrAlarmHi = !enable;
        }
    }


   /**
    ***************************************************************************
    **
    **  Gets the channel value.
    **
    **  @return  The last value successfully read from the channel.
    **
    ***************************************************************************
    */
    public double getValue()
    {
        return value;
    }


   /**
    ***************************************************************************
    **
    **  Gets the channel timestamp.
    **
    **  @return  The timestamp of the last successful read.
    **
    ***************************************************************************
    */
    public long getTimestamp()
    {
        return timestamp;
    }


   /**
    ***************************************************************************
    **
    **  Gets the channel state.
    **
    **  @return  The channel state.
    **
    ***************************************************************************
    */
    public int getState()
    {
        return state;
    }


   /**
    ***************************************************************************
    **
    **  Sets whether the channel is valid.
    **
    **  @param  isValid  Value to set
    **
    ***************************************************************************
    */
    public void setValid(boolean isValid)
    {
        valid = isValid;
    }


   /**
    ***************************************************************************
    **
    **  Reads the value directly from the hardware.
    **
    **  @return  The value read or NaN if the device is offline
    **
    ***************************************************************************
    */
    public double readValue()
    {
        if (!valid || !devc.isOnline()) {
            return Double.NaN;
        }
        else {
           return offset + scale * devc.readChannelNow(hwChan, typeI);
        }
    }

}
