package org.lsst.ccs.subsystem.monitor;

import java.util.ArrayList;
import java.util.BitSet;
import java.util.List;
import java.util.Map;
import java.util.Timer;
import java.util.TimerTask;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.data.KeyValueDataList;
import org.lsst.ccs.framework.Module;
import org.lsst.ccs.subsystem.monitor.data.MonitorChan;
import org.lsst.ccs.subsystem.monitor.data.MonitorFullState;
import org.lsst.ccs.subsystem.monitor.data.MonitorState;
import org.lsst.ccs.utilities.logging.Logger;

/**
 ******************************************************************************
 **
 **  Implements monitoring.
 **
 **  @author Owen Saxton
 **
 ******************************************************************************
 */
public class Monitor {

   /**
    ***************************************************************************
    **
    **  Constants.
    **
    ***************************************************************************
    */
    private final static int
        DEFAULT_UPDATE_PERIOD = 1000,
        DEFAULT_CHECK_PERIOD  = 10000;

    final static String
        CALC_DEVICE_NAME = "Calc",
        LIMITS = "Limits";

   /**
    ***************************************************************************
    **
    **  Data fields.
    **
    ***************************************************************************
    */
    private final Module module;
    private final Map<String, Device> devcMap;
    private final Map<String, Channel> chanMap;
    private final Map<String, Alarm> alarmMap;
    private final Map<String, Line> lineMap;
    private final List<Device> devcList = new ArrayList<>();
    private final List<Alarm> alarmList = new ArrayList<>();
    private final BitSet goodChans = new BitSet(), onlineChans = new BitSet();

    final AlarmHandler alarmH;
    final Logger log;
    final CalcDevice calcDevc = new CalcDevice();
    final Subsystem subsys;
    final BitSet loLimChange = new BitSet(), hiLimChange = new BitSet();

    private int numChans;
    private Channel[] chanData;
    private boolean doPublishData;


   /**
    ***************************************************************************
    **
    **  Interface for handling alarm events.
    **
    ***************************************************************************
    */
    public interface AlarmHandler {

        void processAlarm(int event, int parm);

    }


   /**
    ***************************************************************************
    **
    **  Inner class to implement a timer task to update the channel state
    **  in a timely manner.
    **
    ***************************************************************************
    */
    private class UpdateState extends TimerTask {

        @Override
        public void run()
        {
            /*
            **  Read the sensor values
            */
            readSensors();

            /*
            **  Check the limits and set the channel state
            */
            updateState();
        }

    }

                
   /**
    ***************************************************************************
    **
    **  Inner class to implement a timer task to check hardware devices state.
    **
    ***************************************************************************
    */
    private class CheckDevices extends TimerTask {

        @Override
        public void run()
        {
            devcList.stream().forEach((devc) -> {
                devc.checkOnline();
            });
        }

    }


   /**
    ***************************************************************************
    **
    **  Constructors.
    **
    **  @param  mod        The associated module
    **
    **  @param  alarmHand  The associated alarm handler
    **
    **  @param  logger     The associated logger
    **
    ***************************************************************************
    */
    public Monitor(Module mod, AlarmHandler alarmHand, Logger logger)
    {
        module = mod;
        alarmH = alarmHand;
        log = logger;
        subsys = module.getSubsystem();
        devcMap = module.getChildren(Device.class);
        chanMap = module.getChildren(Channel.class);
        alarmMap = module.getChildren(Alarm.class);
        lineMap = module.getChildren(Line.class);
    }


   /**
    ***************************************************************************
    **
    **  Initializes the monitoring configuration.
    **
    ***************************************************************************
    */
    public void initConfiguration()
    {
        devcMap.put(CALC_DEVICE_NAME, calcDevc);
        for (Device devc : devcMap.values()) {
            devcList.add(devc);
            devc.configure(this);
        }

        for (Line line : lineMap.values()) {
            line.configure(this);
        }

        for (Alarm alarm : alarmMap.values()) {
            alarmList.add(alarm);
            alarm.configure(this);
        }

        chanData = new Channel[chanMap.size()];
        int id = 0;
        for (Channel ch : chanMap.values()) {
            ch.configure(this, id);
            chanData[id++] = ch;
        }
        calcDevc.configChans();
    }


   /**
    ***************************************************************************
    **
    **  Initializes the monitoring sensors.
    **
    ***************************************************************************
    */
    public void initSensors()
    {
        initSensors(DEFAULT_CHECK_PERIOD);
    }

    public void initSensors(int checkPeriod)
    {
        /*
        **  Set default check period if necessary
        */
        if (checkPeriod <= 0) {
            checkPeriod = DEFAULT_CHECK_PERIOD;
        }

        /*
        **  Perform full initialization of all enabled devices
        */
        for (Device devc : devcList) {
            if (!devc.disabled) {
                devc.initialize();
            }
        }

        /*
        **  Initialize the alarms
        */
        for (Alarm alarm : alarmList) {
            alarm.initialize();
        }

        /*
        **  Start the timer task which checks the device state periodically
        */
        (new Timer("CheckDevices", true)).schedule(new CheckDevices(), 0L,
                                                   checkPeriod);
    }


   /**
    ***************************************************************************
    **
    **  Reads the sensor data.
    **
    ***************************************************************************
    */
    public void readSensors()
    {
        for (Device devc : devcList) {
            devc.readSensors();
        }
    }


   /**
    ***************************************************************************
    **
    **  Starts monitoring.
    **
    ***************************************************************************
    */
    public void start()
    {
        start(DEFAULT_UPDATE_PERIOD);
    }
    
    public void start(int period)
    {
        (new Timer("UpdateState", true)).schedule(new UpdateState(), 0L, period);
    }


   /**
    ***************************************************************************
    **
    **  Gets the full monitoring state.
    **
    **  @return  The full monitoring state, as needed by GUIs
    **
    ***************************************************************************
    */
    public MonitorFullState getFullState()
    {
        MonitorFullState state = new MonitorFullState();
        for (Alarm alarm : alarmList) {
            state.addAlarm(alarm.getName(), alarm.getDescription());
        }
        for (Channel ch : chanData) {
            state.addChannel(new MonitorChan(ch.getName(), ch.getDescription(),
                                             ch.getFormat(), ch.getUnits(),
                                             ch.getLimitLo(),
                                             ch.getAlarmNameLo(),
                                             ch.getLimitHi(),
                                             ch.getAlarmNameHi(),
                                             ch.getValue()));
        }
        state.setMonitorState(new MonitorState(goodChans, onlineChans,
                                               loLimChange, hiLimChange));
        return state;
    }


   /**
    ***************************************************************************
    **
    **  Publishes the state of the monitoring module.
    **
    **  This is intended to be called whenever any element of the state is
    **  changed.
    **
    ***************************************************************************
    */
    public void publishState()
    {
        MonitorState state = new MonitorState(goodChans, onlineChans,
                                              loLimChange, hiLimChange);
        KeyValueData data = new KeyValueData(MonitorState.KEY, state);
        subsys.publishSubsystemDataOnStatusBus(data);
    }    


   /**
    ***************************************************************************
    **
    **  Publishes all the data values of the monitoring module.
    **
    ***************************************************************************
    */
    public void publishData()
    {
        KeyValueDataList dataList = new KeyValueDataList();
        for (Channel ch : chanData) {
            if (ch.getState() != Channel.STATE_OFFLINE) {
                dataList.addData(ch.getName(), ch.getValue(), ch.getTimestamp());
            }
        }
        if (!dataList.getListOfKeyValueData().isEmpty()) {
            subsys.publishSubsystemDataOnStatusBus(dataList);
        }
    }


   /**
    ***************************************************************************
    **
    **  Publishes all the limit values of the monitoring module.
    **
    **  This is intended to be called at startup time.
    **
    ***************************************************************************
    */
    public void publishLimits()
    {
        for (Channel ch : chanData) {        
            KeyValueDataList data = new KeyValueDataList(ch.getName());
            data.addData("alarmLow", String.valueOf(ch.getLimitLo()),
                         KeyValueData.KeyValueDataType.KeyValueMetaData);
            data.addData("alarmHigh", String.valueOf(ch.getLimitHi()),
                         KeyValueData.KeyValueDataType.KeyValueMetaData);
            subsys.publishSubsystemDataOnStatusBus(data);
        }
    }    


   /**
    ***************************************************************************
    **
    **  Updates the state of the monitoring system.
    **
    **  This checks whether all current channel values are within limits,
    **  activates any resulting alarms, checks for any changes in online
    **  channels, and reports any changes on the status bus.
    **
    ***************************************************************************
    */
    public void updateState()
    {
        /*
        **  Generate the current channel good and online states
        */
        BitSet goodState = new BitSet(numChans),
               onlineState = new BitSet(numChans);
        for (Channel ch : chanData) {
            ch.checkLimits(goodState);
            ch.checkOnline(onlineState);
        }
        for (Alarm alarm : alarmList) {
            alarm.setState();
        }
                
        /*
        **  If state changed, publish it and log channel data
        */
        goodState.xor(goodChans);
        goodChans.xor(goodState);
        onlineState.xor(onlineChans);
        onlineChans.xor(onlineState);
        if (!goodState.isEmpty() || !onlineState.isEmpty()) {
            publishState();
        }
        if (!goodState.isEmpty() || doPublishData) {
            publishData();
            doPublishData = false;
        }
    }


    @Deprecated
    public void checkLimits()
    {
        updateState();
    }


   /**
    ***************************************************************************
    **
    **  Sets the publish data flag.
    **
    ***************************************************************************
    */
    public void setPublishData()
    {
        doPublishData = true;
    }


   /**
    ***************************************************************************
    **
    **  Clears limits-changed indicators.
    **
    ***************************************************************************
    */
    public void clearLimitChanges()
    {
        loLimChange.clear();
        hiLimChange.clear();
    }


   /**
    ***************************************************************************
    **
    **  Gets the device with the given name.
    **
    **  @param  name  The device name
    **
    **  @return  The matching device, or null if not found
    **
    ***************************************************************************
    */
    public Device getDevice(String name)
    {
        return devcMap.get(name);
    }


   /**
    ***************************************************************************
    **
    **  Gets the line with the given name.
    **
    **  @param  name  The line name
    **
    **  @return  The matching line, or null if not found
    **
    ***************************************************************************
    */
    public Line getLine(String name)
    {
        return lineMap.get(name);
    }


   /**
    ***************************************************************************
    **
    **  Gets the alarm with the given name.
    **
    **  @param  name  The alarm name
    **
    **  @return  The matching alarm, or null if not found
    **
    ***************************************************************************
    */
    public Alarm getAlarm(String name)
    {
        return alarmMap.get(name);
    }


   /**
    ***************************************************************************
    **
    **  Gets the channel with the given name.
    **
    **  @param  name  The channel name
    **
    **  @return  The matching channel, or null if not found
    **
    ***************************************************************************
    */
    public Channel getChannel(String name)
    {
        return chanMap.get(name);
    }


   /**
    ***************************************************************************
    **
    **  Gets the channel with the given ID.
    **
    **  @param  id  The channel ID
    **
    **  @return  The matching channel
    **
    ***************************************************************************
    */
    public Channel getChannel(int id)
    {
        return chanData[id];
    }


   /**
    ***************************************************************************
    **
    **  Gets the ID of the channel with the given name.
    **
    **  @param  name  The channel name
    **
    **  @return  The ID of the matching channel, or -1 if not found
    **
    ***************************************************************************
    */
    public int getChannelId(String name)
    {
        Channel chan = chanMap.get(name);
        return chan == null ? -1 : chan.getId();
    }


   /**
    ***************************************************************************
    **
    **  Sets channels on- or off-line.
    **
    **  @param  mask    Mask of affected channels
    **
    **  @param  online  True if channels to be set online, false otherwise
    **
    ***************************************************************************
    */
    public void setOnline(BitSet mask, boolean online)
    {
        BitSet prevOnline = (BitSet)onlineChans.clone();
        if (online) {
            onlineChans.or(mask);
        }
        else {
            onlineChans.andNot(mask);
        }
        if (!onlineChans.equals(prevOnline)) {
            publishState();
        }
    }


   /**
    ***************************************************************************
    **
    **  Gets the number of channels.
    **
    **  @return  The number of channels
    **
    ***************************************************************************
    */
    public int getNumChans()
    {
        return numChans;
    }


   /**
    ***************************************************************************
    **
    **  Gets the associated module.
    **
    **  @return  The module
    **
    ***************************************************************************
    */
    public Module getModule()
    {
        return module;
    }


   /**
    ***************************************************************************
    **
    **  Saves the limit values.
    **
    ***************************************************************************
    */
    public void saveLimits()
    {
        module.getEnvironment().saveChangesForCategories(LIMITS);
        clearLimitChanges();
        publishState();         // Must always do this
    }


   /**
    ***************************************************************************
    **
    **  Reports parameter errors.
    **
    **  @param  cName   The component name
    **
    **  @param  pName   The parameter name
    **
    **  @param  pValue  The parameter value
    **
    **  @throws  Exception, always
    **
    ***************************************************************************
    */
    public void reportError(String cName, String pName, Object pValue)
        throws Exception
    {
        log.error("Invalid " + pName + " (" + pValue + ") for " + cName);
        throw new Exception();
    }


   /**
    ***************************************************************************
    **
    **  Reports configuration parameter errors and exits.
    **
    **  @param  cName    The component name
    **
    **  @param  pName    The parameter name
    **
    **  @param  message  The message to log
    **
    ***************************************************************************
    */
    public void reportConfigError(String cName, String pName, String message)
    {
        log.error("Configuration parameter " + pName + " for " + cName + " "
                    + message);
        throw new RuntimeException("Fatal configuration error");
    }

}
