package org.lsst.ccs.subsystem.monitor;

import java.io.IOException;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.command.annotations.Command.CommandType;
import org.lsst.ccs.framework.HardwareController;
import org.lsst.ccs.framework.Module;
import org.lsst.ccs.framework.TreeWalkerDiag;
import org.lsst.ccs.subsystem.monitor.data.MonitorFullState;
import org.lsst.ccs.subsystem.monitor.data.MonMainFullState;
import org.lsst.ccs.subsystem.monitor.data.MonMainState;
import org.lsst.ccs.utilities.logging.Logger;

/**
 ******************************************************************************
 **
 **  Implements a monitor-only modular subsystem.
 **
 **  @author Owen Saxton
 **
 ******************************************************************************
 */
public class MonMain extends Module implements HardwareController {

   /**
    ***************************************************************************
    **
    **  Data fields.
    **
    ***************************************************************************
    */
    private final int checkMillis;
    private final Logger sLog;
    private Monitor mon;


   /**
    ***************************************************************************
    **
    **  Main constructor.
    **
    **  @param  name             The subsystem name.
    **
    **  @param  tickMillis       The tick period (ms).
    **
    **  @param  checkMillis      The hardware check period (ms).
    **
    ***************************************************************************
    */
    public MonMain(String name, int tickMillis, int checkMillis)
    {
        super(name, tickMillis);
        this.checkMillis = checkMillis;
        sLog = Logger.getLogger("org.lsst.ccs.subsystem.monitor");
    }


   /**
    ***************************************************************************
    **
    **  Initializes the subsystem.
    **
    ***************************************************************************
    */
    @Override
    public void initModule()
    {
        /*
        **  Initialize all monitoring configuration data
        */
        mon = new Monitor(this, null, sLog);
        mon.initConfiguration();
    }


   /**
    ***************************************************************************
    **
    **  Initializes the hardware.
    **
    **  @return  GO
    **
    ***************************************************************************
    */
    @Override
    public TreeWalkerDiag checkHardware()
    {
        /*
        **  Initialize the hardware
        */
        mon.initSensors(checkMillis);

        return TreeWalkerDiag.GO;
    }


   /**
    ***************************************************************************
    **
    **  Checks whether hardware is started.
    **
    ***************************************************************************
    */
    @Override
    public void checkStarted()
    {
        System.out.println("checkStarted was called");
    }


   /**
    ***************************************************************************
    **
    **  Checks whether hardware is stopped.
    **
    ***************************************************************************
    */
    @Override
    public void checkStopped()
    {
        System.out.println("checkStopped was called");
    }


   /**
    ***************************************************************************
    **
    **  Starts the subsystem.
    **
    ***************************************************************************
    */
    @Override
    public void startTicking()
    {
        mon.start();
        super.startTicking();
        publishState();
        mon.publishState();    // For any GUIs
        mon.publishLimits();   // For GUIs and the trending database
        System.out.println("Monitoring test started");
    }


   /**
    ***************************************************************************
    **
    **  Performs periodic trending data broadcast.
    **
    ***************************************************************************
    */
    @Override
    public void tick()
    {
        mon.publishData();
    }


   /**
    ***************************************************************************
    **
    **  Sets the tick period.
    **
    **  Overrides the method in Module in order to publish a status update.
    **
    **  @param  value  The tick period (milliseconds) to set.
    **
    ***************************************************************************
    */
    @Command(type=CommandType.ACTION, description="Set the tick interval")
    @Override
    public void setTickMillis(int value)
    {
        super.setTickMillis(value);
        publishState();
    }


   /**
    ***************************************************************************
    **
    **  Saves the configuration data.
    **
    **  @throws  IOException
    **
    ***************************************************************************
    */
    @Command(type=CommandType.ACTION,
             description="Save the current configuration")
    public void saveConfig() throws IOException
    {
        getEnvironment().saveAllChanges();
        mon.clearLimitChanges();
        mon.publishState();     // Must always do this
    }


   /**
    ***************************************************************************
    **
    **  Gets the full state of the monitoring module.
    **
    **  This is intended to be called by GUIs during initialization
    **
    **  @return  The full generic monitoring state
    **
    ***************************************************************************
    */
    @Command(type=CommandType.QUERY,
             description="Get the full generic monitoring state")
    public MonMainFullState getFullState()
    {
        MonMainState mmState = new MonMainState(getTickMillis());
        MonitorFullState monState = mon.getFullState();
        return new MonMainFullState(mmState, monState);
    }    


   /**
    ***************************************************************************
    **
    **  Publishes the state of the monitor module.
    **
    **  This is intended to be called whenever any element of the state is
    **  changed.
    **
    ***************************************************************************
    */
    private void publishState()
    {
        KeyValueData data = new KeyValueData(MonMainState.KEY,
                                             new MonMainState(getTickMillis()));
        getSubsystem().publishSubsystemDataOnStatusBus(data);
    }    

}
