package org.lsst.ccs.subsystem.monitor;

import java.util.BitSet;
import java.util.HashMap;
import java.util.IllegalFormatException;
import java.util.Map;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.data.KeyValueDataList;
import org.lsst.ccs.framework.ConfigurableComponent;
import org.lsst.ccs.framework.annotations.ConfigChanger;
import org.lsst.ccs.utilities.logging.Logger;

/**
 ******************************************************************************
 **
 **  Handles a monitoring channel.
 **
 **  @author Owen Saxton
 **
 ******************************************************************************
 */
public class Channel extends ConfigurableComponent {

   /**
    ***************************************************************************
    **
    **  Constants.
    **
    ***************************************************************************
    */
    public final static int
        TYPE_UNKNOWN = -1,
        TYPE_TEMP    = 0,
        TYPE_VOLTS   = 1,
        TYPE_POWER   = 2,
        TYPE_SWITCH  = 3,
        TYPE_DIFF    = 4;
    
    public final static int
        STATE_OFFLINE = -1,
        STATE_GOOD    = 0,
        STATE_WARNING = 1,
        STATE_ERROR   = 2;

    private final static int
        LIMIT_CHECK_NONE  = 0,   // Don't check limit
        LIMIT_CHECK_FLAG  = 1,   // Flag if limit exceeded
        LIMIT_CHECK_ALARM = 2;   // Alarm if limit exceeded

    private final String DEFAULT_FORMAT = "%.2f ";

   /**
    ***************************************************************************
    **
    **  Private lookup maps.
    **
    ***************************************************************************
    */
    private final static Map<String,Integer> checkMap = new HashMap<>();
    static {
        checkMap.put("NONE",  LIMIT_CHECK_NONE);
        checkMap.put("FLAG",  LIMIT_CHECK_FLAG);
        checkMap.put("ALARM", LIMIT_CHECK_ALARM);
    }

   /**
    ***************************************************************************
    **
    **  Data fields.
    **
    ***************************************************************************
    */
    // Supplied immutable fields
    private final String  description;   // Channel description
    private final String  units;         // Units name
    private final String  devcName;      // The name of the associated device
    private final int     hwChan;        // Hardware channel
    private final String  typeS;         // Channel type string (TEMP, VOLTS, POWER...)
    private final String  subtypeS;      // Channel subtype string (extra config info)
    private final double  offset;        // Value offset
    private final double  scale;         // Value scale
    private final String  checkLoS;      // Low limit check option string
    private final String  alarmLoS;      // Low limit alarm name (null = none)
    private final double  dbandLo;       // Low limit deadband
    private final String  checkHiS;      // High limit check option string
    private final String  alarmHiS;      // High limit alarm name (null = none)
    private final double  dbandHi;       // High limit deadband

    // Supplied changeable or configurable fields
    private String        format;        // Format string for diaplay (less "%")
    private double        limitLo;       // Low limit value
    private double        limitHi;       // High limit value

    // Derived immutable fields
    private String        name;          // Channel name
    private Device        devc;          // The associated device
    private int           type;          // Encoded channel type
    private int           subtype;       // Encoded channel subtype
    private int           checkLo;       // Low limit check option
    private Alarm         alarmLo;       // Low limit alarm (null = none)
    private int           checkHi;       // High limit check option
    private Alarm         alarmHi;       // High limit alarm (null = none)
    private int           id;            // External ID
    private boolean       valid;         // True if configured correctly
    private Monitor       mon;           // Associated monitor object
    private Logger        log;           // The logger

    // Volatile fields
    private double        value;         // Last value read from sensor
    private long          timestamp;     // Time of last successful read
    private int           state;         // State of the channel
    private double        resetLo;       // Low limit reset value
    private double        resetHi;       // High limit reset value
    private boolean       ignrAlarmLo;   // True if low limit alarm ignored
    private boolean       ignrAlarmHi;   // True if high limit alarm ignored


   /**
    ***************************************************************************
    **
    **  Constructor.
    **
    **  @param  desc        The descriptive name of the channel.
    **
    **  @param  fmt         The format string for displaying the value.
    **
    **  @param  unit        The units name for the channel.
    **
    **  @param  devName     The name of the device containing the channel.
    **
    **  @param  chan        The hardware channel number of the channel.
    **
    **  @param  type        The type of the channel.
    **
    **  @param  subtype     The sub-type of the channel.
    **
    **  @param  offs        The offset to be applied to the read value.
    **
    **  @param  scal        The scale factor for the read value.
    **
    **  @param  checkLo     The type of check against the low limit.
    **
    **  @param  limLo       The low limit value.
    **
    **  @param  deadbandLo  The low limit deadband value.
    **
    **  @param  alarmLo     The name of the alarm for the low limit.
    **
    **  @param  checkHi     The type of check against the high limit.
    **
    **  @param  limHi       The high limit value.
    **
    **  @param  deadbandHi  The high limit deadband value.
    **
    **  @param  alarmHi     The name of the alarm for the high limit.
    **
    ***************************************************************************
    */
    public Channel(String desc, String fmt, String unit, String devName,
                   int chan, String type, String subtype, double offs,
                   double scal, String checkLo, double limLo, double deadbandLo,
                   String alarmLo, String checkHi, double limHi,
                   double deadbandHi, String alarmHi)
    {
        description = desc;
        format      = (fmt == null || fmt.isEmpty()) ? DEFAULT_FORMAT
                        : ((fmt.charAt(0) != '%') ? "%" + fmt : fmt) + " ";
        units       = unit;
        devcName    = devName == null ? Monitor.CALC_DEVICE_NAME : devName;
        hwChan      = chan;
        typeS       = type;
        subtypeS    = subtype;
        offset      = offs;
        scale       = scal;
        checkLoS    = checkLo;
        limitLo     = limLo;
        dbandLo     = deadbandLo;
        resetLo     = dbandLo <= 0 ? limitLo : limitLo + dbandLo;
        alarmLoS    = alarmLo;
        checkHiS    = checkHi;
        limitHi     = limHi;
        dbandHi     = deadbandHi;
        resetHi     = dbandHi <= 0 ? limitHi : limitHi - dbandHi;
        alarmHiS    = alarmHi;
        state       = STATE_OFFLINE;
    }


   /**
    ***************************************************************************
    **
    **  Constructor.
    **
    **  This is the old version, which doesn't set the format string.
    **
    **  @param  desc        The descriptive name of the channel.
    **
    **  @param  unit        The units name for the channel.
    **
    **  @param  devName     The name of the device containing the channel.
    **
    **  @param  chan        The hardware channel number of the channel.
    **
    **  @param  type        The type of the channel.
    **
    **  @param  subtype     The sub-type of the channel.
    **
    **  @param  offs        The offset to be applied to the read value.
    **
    **  @param  scal        The scale factor for the read value.
    **
    **  @param  checkLo     The type of check against the low limit.
    **
    **  @param  limLo       The low limit value.
    **
    **  @param  deadbandLo  The low limit deadband value.
    **
    **  @param  alarmLo     The name of the alarm for the low limit.
    **
    **  @param  checkHi     The type of check against the high limit.
    **
    **  @param  limHi       The high limit value.
    **
    **  @param  deadbandHi  The high limit deadband value.
    **
    **  @param  alarmHi     The name of the alarm for the high limit.
    **
    ***************************************************************************
    */
    public Channel(String desc, String unit, String devName, int chan,
                   String type, String subtype, double offs, double scal,
                   String checkLo, double limLo, double deadbandLo,
                   String alarmLo, 
                   String checkHi, double limHi, double deadbandHi,
                   String alarmHi)
    {
        this(desc, null, unit, devName, chan, type, subtype, offs, scal,
             checkLo, limLo, deadbandLo, alarmLo, checkHi, limHi, deadbandHi,
             alarmHi);
    }


   /**
    ***************************************************************************
    **
    **  Configures channel description.
    **
    ***************************************************************************
    */
    void configure(Monitor mon, int id)
    {
        name = getName();
        this.mon = mon;
        this.id = id;
        log = mon.log;

        checkLo = LIMIT_CHECK_FLAG;
        checkHi = LIMIT_CHECK_FLAG;
        type    = TYPE_UNKNOWN;
        subtype = -1;

        try {
            devc = mon.getDevice(devcName);
            if (devc == null) {
                mon.reportError(name, "device name", devcName);
            }
        }
        catch (Exception e) {
        }

        try {
            String origFmt = format;
            try {
                String.format(format, 0.0);
            }
            catch (IllegalFormatException e) {
                format = DEFAULT_FORMAT;
                mon.reportError(name, "display format", origFmt);
            }

            Integer iCheck = checkMap.get(checkLoS.toUpperCase());
            if (iCheck == null) {
                mon.reportError(name, "low check option", checkLoS);
            }
            checkLo = iCheck;
            iCheck = checkMap.get(checkHiS.toUpperCase());
            if (iCheck == null) {
                mon.reportError(name, "high check option", checkHiS);
            }
            checkHi = iCheck;

            if (alarmLoS != null) {
                alarmLo = mon.getAlarm(alarmLoS);
            }
            if (alarmLo == null
                  && (checkLo == LIMIT_CHECK_ALARM || alarmLoS != null)) {
                mon.reportError(name, "low alarm name", alarmLoS);
            }
            if (alarmHiS != null) {
                alarmHi = mon.getAlarm(alarmHiS);
            }
            if (alarmHi == null
                  && (checkHi == LIMIT_CHECK_ALARM || alarmHiS != null)) {
                    mon.reportError(name, "high alarm name", alarmHiS);
            }

            int[] types = devc.checkChannel(name, hwChan, typeS, subtypeS);
            type = types[0];
            subtype = types[1];

            valid = true;
            devc.addChannel(id);
        }
        catch (Exception e) {
        }
    }


   /**
    ***************************************************************************
    **
    **  Completes configuration of calculation channel description.
    **
    ***************************************************************************
    */
    void configCalc()
    {
        String[] names = subtypeS.split(":", -1);
        int id0 = 0, id1 = 0;
        try {
            if (names.length != 2
                  || (id0 = mon.getChannelId(names[0])) < 0
                  || (id1 = mon.getChannelId(names[1])) < 0) {
                mon.reportError(name, "subtype", subtypeS);
            }
            subtype = (id0 << 16) | id1;
        }
        catch (Exception e) {
            valid = false;
            devc.dropChannel(id);
        }
    }


   /**
    ***************************************************************************
    **
    **  Initializes the hardware sensor.
    **
    ***************************************************************************
    */
    void initSensor()
    {
        if (!valid) return;
        devc.initChannel(name, id, hwChan, type, subtype);
    }


   /**
    ***************************************************************************
    **
    **  Reads the hardware sensor.
    **
    ***************************************************************************
    */
    void readSensor()
    {
        double val;
        if (!valid || !devc.isOnline()) {
            val = Double.NaN;
        }
        else if (devc != mon.calcDevc) {
            val = offset + scale * devc.readChannel(hwChan, type);
        }
        else {
            Channel chan1 = mon.getChannel(subtype >> 16);
            Channel chan2 = mon.getChannel(subtype & 0xffff);
            if (chan1.state != STATE_OFFLINE
                  && chan2.state != STATE_OFFLINE) {
                val = chan1.value - chan2.value;
            }
            else {
                val = Double.NaN;
            }
        }
        if (!Double.isNaN(val)) {
            value = val;
            timestamp = System.currentTimeMillis();
            state = STATE_GOOD;
        }
        else {
            state = STATE_OFFLINE;
        }
    }


   /**
    ***************************************************************************
    **
    **  Checks whether current value is within limits.
    **
    **  @param  cState  The channel state to be updated
    **
    ***************************************************************************
    */
    void checkLimits(BitSet cState)
    {
        if (!valid || state == STATE_OFFLINE) return;
        boolean ok = true;
        if (value < limitLo) {
            if (checkLo != LIMIT_CHECK_NONE) {
                ok = false;
                if (checkLo == LIMIT_CHECK_ALARM && !ignrAlarmLo) {
                    alarmLo.updateState(Alarm.STATE_ERROR);
                }
            }
        }
        else if (value > limitHi) {
            if (checkHi != LIMIT_CHECK_NONE) {
                ok = false;
                if (checkHi == LIMIT_CHECK_ALARM && !ignrAlarmHi) {
                    alarmHi.updateState(Alarm.STATE_ERROR);
                }
            }
        }
        else {
            if (checkLo == LIMIT_CHECK_ALARM && !ignrAlarmLo) {
                alarmLo.updateState(value >= resetLo ? Alarm.STATE_GOOD
                                                     : Alarm.STATE_DEADBAND);
            }
            if (checkHi == LIMIT_CHECK_ALARM && !ignrAlarmHi) {
                alarmHi.updateState(value <= resetHi ? Alarm.STATE_GOOD
                                                     : Alarm.STATE_DEADBAND);
            }
        }
        if (ok) {
            cState.set(id);
            state = STATE_GOOD;
        }
        else {
            state = STATE_ERROR;
        }
    }


   /**
    ***************************************************************************
    **
    **  Checks whether channel is online.
    **
    **  @param  cState  The channel state to be updated
    **
    ***************************************************************************
    */
    void checkOnline(BitSet cState)
    {
        cState.set(id, valid && state != STATE_OFFLINE);
    }


   /**
    ***************************************************************************
    **
    **  Gets the channel description.
    **
    **  @return  The channel description
    **
    ***************************************************************************
    */
    String getDescription()
    {
        return description;
    }


   /**
    ***************************************************************************
    **
    **  Gets the display format.
    **
    **  @return  The channel description
    **
    ***************************************************************************
    */
    String getFormat()
    {
        return format;
    }


   /**
    ***************************************************************************
    **
    **  Gets the channel units.
    **
    **  @return  The channel units
    **
    ***************************************************************************
    */
    String getUnits()
    {
        return units;
    }


   /**
    ***************************************************************************
    **
    **  Gets the channel low limit.
    **
    **  @return  The channel low limit
    **
    ***************************************************************************
    */
    double getLimitLo()
    {
        return limitLo;
    }


   /**
    ***************************************************************************
    **
    **  Gets the channel low limit alarm name.
    **
    **  @return  The channel low limit alarm name
    **
    ***************************************************************************
    */
    String getAlarmNameLo()
    {
        return checkLo == LIMIT_CHECK_ALARM ? alarmLoS
                 : checkLo == LIMIT_CHECK_FLAG ? "" : null;
    }


   /**
    ***************************************************************************
    **
    **  Gets the channel high limit.
    **
    **  @return  The channel high limit
    **
    ***************************************************************************
    */
    double getLimitHi()
    {
        return limitHi;
    }


   /**
    ***************************************************************************
    **
    **  Gets the channel high limit alarm name.
    **
    **  @return  The channel high limit alarm name
    **
    ***************************************************************************
    */
    String getAlarmNameHi()
    {
        return checkHi == LIMIT_CHECK_ALARM ? alarmHiS
                 : checkHi == LIMIT_CHECK_FLAG ? "" : null;
    }


   /**
    ***************************************************************************
    **
    **  Gets the channel ID.
    **
    **  @return  The channel ID
    **
    ***************************************************************************
    */
    int getId()
    {
        return id;
    }


   /**
    ***************************************************************************
    **
    **  Sets the low limit.
    **
    **  @param  limit  The low limit value to set.
    **
    ***************************************************************************
    */
    @ConfigChanger
    public void setLimitLo(double limit)
    {
        if (!valid || limit == limitLo) return;
        limitLo = limit;
        resetLo = dbandLo <= 0 ? limitLo : limitLo + dbandLo;
        mon.loLimChange.set(id);
        KeyValueDataList data = new KeyValueDataList(name);
        data.addData("alarmLow", String.valueOf(limit),
                     KeyValueData.KeyValueDataType.KeyValueMetaData);
        mon.subsys.publishSubsystemDataOnStatusBus(data);
        mon.publishState();
    }


   /**
    ***************************************************************************
    **
    **  Sets the high limit.
    **
    **  @param  limit  The high limit value to set.
    **
    ***************************************************************************
    */
    @ConfigChanger
    public void setLimitHi(double limit)
    {
        if (!valid || limit == limitHi) return;
        limitHi = limit;
        resetHi = dbandHi <= 0 ? limitHi : limitHi - dbandHi;
        mon.hiLimChange.set(id);
        KeyValueDataList data = new KeyValueDataList(name);
        data.addData("alarmHigh", String.valueOf(limit),
                     KeyValueData.KeyValueDataType.KeyValueMetaData);
        mon.subsys.publishSubsystemDataOnStatusBus(data);
        mon.publishState();
    }


   /**
    ***************************************************************************
    **
    **  Enables or disables an alarm.
    **
    **  @param  isLow   True for low limit alarm, false for high.
    **
    **  @param  enable  True to enable the alarm, false to disable it.
    **
    ***************************************************************************
    */
    public void enableAlarm(boolean isLow, boolean enable)
    {
        if (isLow) {
            ignrAlarmLo = !enable;
        }
        else {
            ignrAlarmHi = !enable;
        }
    }


   /**
    ***************************************************************************
    **
    **  Gets the channel value.
    **
    **  @return  The last value successfully read from the channel.
    **
    ***************************************************************************
    */
    public double getValue()
    {
        return value;
    }


   /**
    ***************************************************************************
    **
    **  Gets the channel timestamp.
    **
    **  @return  The timestamp of the last successful read.
    **
    ***************************************************************************
    */
    public long getTimestamp()
    {
        return timestamp;
    }


   /**
    ***************************************************************************
    **
    **  Gets the channel state.
    **
    **  @return  The channel state.
    **
    ***************************************************************************
    */
    public int getState()
    {
        return state;
    }

}
