package org.lsst.ccs.subsystem.monitor;

import org.lsst.ccs.framework.ConfigurableComponent;

/**
 ***************************************************************************
 **
 **  Implements hardware output lines.
 **
 **  @author Owen Saxton
 **
 ***************************************************************************
 */
public class Line extends ConfigurableComponent {

   /**
    ***********************************************************************
    **
    **  Data fields.
    **
    ***********************************************************************
    */
    // Supplied immutable fields
    private final String  devcName;    // Device name
    private final int     chan;        // HW channel number
    private final String  devcNameC;   // Complementary device name
    private final int     chanC;       // Complementary HW channel number

    // Derived immutable fields
    private Device        device;      // Device for this line
    private Device        deviceC;     // Device for the complementary line


   /**
    ***********************************************************************
    **
    **  Constructor.
    **
    **  @param  devcName   The name of the associated bit-set device, or
    **                     null if none.
    **
    **  @param  chan       The hardware channel number on the bit-set
    **                     device.
    **
    **  @param  devcNameC  The name of the associated bit-clear device, or
    **                     null if none.
    **
    **  @param  chanC      The hardware channel number on the bit-clear
    **                     device.
    **
    ***********************************************************************
    */
    public Line(String devcName, int chan, String devcNameC, int chanC)
    {
        this.devcName  = devcName;
        this.chan      = chan;
        this.devcNameC = devcNameC;
        this.chanC     = chanC;
    }


   /**
    ***********************************************************************
    **
    **  Configures the line.
    **
    **  @param  mon  The associated monitor
    **
    ***********************************************************************
    */
    void configure(Monitor mon)
    {
        try {
            if (devcName != null) {
                device = mon.getDevice(devcName);
                if (device == null) {
                    mon.reportError(getName(), "device", devcName);
                }
                device.checkLine(getName(), chan);
            }
            if (devcNameC != null) {
                deviceC = mon.getDevice(devcNameC);
                if (deviceC == null) {
                    mon.reportError(getName(), "device", devcNameC);
                }
                deviceC.checkLine(getName(), chanC);
            }
        }
        catch(Exception e) {
        }
    }


   /**
    ***********************************************************************
    **
    **  Sets the warm-start state of the line.
    **
    **  @param  on  True to set the warm-start state, false to clear it.
    **
    ***********************************************************************
    */
    public void setWarm(boolean on)
    {
        if (device != null) {
            device.setLineWarm(chan, on);
        }
        if (deviceC != null) {
            deviceC.setLineWarm(chanC, !on);
        }
    }


   /**
    ***********************************************************************
    **
    **  Sets the line value.
    **
    **  @param  on  True to set the line, false to clear it.
    **
    ***********************************************************************
    */
    public void set(boolean on)
    {
        if (device != null && device.online) {
            device.setLine(chan, on);
        }
        if (deviceC != null && deviceC.online) {
            deviceC.setLine(chanC, !on);
        }
    }


   /**
    ***********************************************************************
    **
    **  Gets the line value.
    **
    **  @return  The value of the line: true or false.
    **
    ***********************************************************************
    */
    public boolean isSet()
    {
        boolean state = false;
        if (device != null && device.online) {
            state = device.isLineSet(chan);
        }
        else if (deviceC != null && deviceC.online) {
            state = !deviceC.isLineSet(chanC);
        }

        return state;
    }

}
