package org.lsst.ccs.subsystem.common.ui.focalplane.filter;

import java.util.*;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.DataProviderInfo;
import org.lsst.ccs.gconsole.annotations.services.persist.Create;
import org.lsst.ccs.gconsole.annotations.services.persist.Par;
import org.lsst.ccs.gconsole.base.Console;
import org.lsst.ccs.gconsole.services.aggregator.AgentChannel;
import org.lsst.ccs.subsystem.common.focalplane.data.HasFocalPlaneData;
import org.lsst.ccs.subsystem.common.ui.focalplane.Segment;

/**
 * Filter that puts channels into groups based on the value of the {@link DataProviderInfo.Attribute.DATA_GROUP DATA_GROUP} attribute.
 *
 * @author onoprien
 */
public class GroupAttributeFilter extends FocalPlaneFilter {

// -- Fields : -----------------------------------------------------------------
    
    static private final String KEY = DataProviderInfo.Attribute.DATA_GROUP.getName();
    private final Descriptor descriptor;
    
    private HashSet<String> groups;

// -- Life cycle : -------------------------------------------------------------
    
    @Create(category = CATEGORY,
            name = "Standard data groups",
            path = "Built-In/Standard Groups/One Subsystem",
            description = "Standard pre-defined focal plane data groups determined by the DATA_GROUP attribute.")
    public GroupAttributeFilter(
            @Par(desc = "Subsystem name.") String subsystem)
    {
        descriptor = new Descriptor();
        descriptor.setAgent(subsystem);
    }
    
    public GroupAttributeFilter(Descriptor desc) {
        descriptor = desc.clone();
        if (descriptor.groups != null) {
            groups = new HashSet<>(Arrays.asList(descriptor.groups));
        }
    }
    
    
// -- Filtering : --------------------------------------------------------------

    @Override
    public String getName() {
        return descriptor.getAgent() == null ? super.getName() : descriptor.getAgent();
    }

    @Override
    public List<String> getAgents() {
        return descriptor.agent == null ? null : Collections.singletonList(descriptor.agent);
    }

    @Override
    public String getDisplayPath(AgentChannel channel) {
        if (descriptor.agent != null && !channel.getAgentName().equals(descriptor.agent)) {
            return null;
        }
        Object gr = channel.get(KEY);
        if (!(gr instanceof String)) {
            return null;
        }
        String group = (String) gr;
        if (groups != null && !groups.contains(group)) {
            return null;
        }
        String prefix = Segment.getPathPrefix(channel);
        if (prefix == null || prefix.isEmpty()) {
            return null;
        }
        return prefix + group;
    }

    @Override
    public List<String> getGroups() {
        if (descriptor.groups != null) {
            return Arrays.asList(descriptor.groups);
        } else if (descriptor.agent == null) {
            return null;
        } else {
            AgentInfo info = Console.getConsole().getStatusAggregator().getAgent(descriptor.agent);
            if (info == null) return null;
            String prop = info.getAgentProperty(HasFocalPlaneData.AGENT_PROPERTY);
            if (prop == null) return null;
            String[] ss = HasFocalPlaneData.getFocalPlaneDataGroups(prop);
            return Arrays.asList(ss);
        }
    }
    
    
// -- Saving/restoring : -------------------------------------------------------
    
    @Override
    public Descriptor getDescriptor() {
        return (Descriptor) descriptor;
    }
    
    static public class Descriptor extends FocalPlaneFilter.Descriptor {

        private String agent;
        private String[] groups;

        public String getAgent() {
            return agent;
        }

        public void setAgent(String agent) {
            this.agent = agent;
        }

        public String[] getGroups() {
            return groups;
        }

        public void setGroups(String[] groups) {
            this.groups = groups;
        }

        @Override
        public Descriptor clone() {
            return (Descriptor) super.clone();
        }
        
    }

}
