package org.lsst.ccs.subsystem.common.ui;

import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import javax.swing.JLabel;
import javax.swing.JPanel;
import org.lsst.ccs.subsystem.common.ui.jas.CommandSender;

/**
 *  Implements the system status line sub-panel.
 *
 *  This needs to be incorporated into a "top-level" panel
 *
 *  @author Owen Saxton
 */
public class SystemStatusPanel extends JPanel implements UiUtilities.ActionHandler {

    private static final int STATUS_WIDTH = UiUtilities.maxLabelWidth(new String[]{"STOPPED", "RUNNING"}, "");

    private final CommandSender sender;
    private final String node, command;
    private final boolean isFast;
    private final UiUtilities uiUtils;
    private JLabel stateStatus;
    private TextFieldX periodTextField;
    private int periodMillis = 0;

    public SystemStatusPanel(CommandSender sender) {
        this(sender, null, "setUpdatePeriod", false);
    }

    public SystemStatusPanel(CommandSender sender, String node) {
        this(sender, node, "setPublishPeriod", false);
    }

    public SystemStatusPanel(CommandSender sender, String node, boolean isFast) {
        this(sender, node, "setFastPeriod", isFast);
    }

    private SystemStatusPanel(CommandSender sender, String node, String command, boolean isFast) {
        this.sender = sender;
        this.node = node;
        this.command = command;
        this.isFast = isFast;
        uiUtils = new UiUtilities(this);
        initComponents();
        disablePanel();
    }

    public void updatePanel(int periodMillis) {
        stateStatus.setText("RUNNING");
        stateStatus.setForeground(UiConstants.GREEN);
        this.periodMillis = periodMillis;
        updatePeriod();
    }

    public void disablePanel() {
        stateStatus.setText("STOPPED");
        stateStatus.setForeground(UiConstants.RED);
        periodTextField.setDisabled();
    }

    private void initComponents() {
        stateStatus = UiUtilities.newLabel("XXX", STATUS_WIDTH);
        if (isFast) {
            periodTextField = uiUtils.newTextFieldX("999:59:59", "", TextFieldX.TYPE_TEXT, true);
        }
        else {
            periodTextField = uiUtils.newTextFieldX("99999.9", "", TextFieldX.TYPE_DOUBLE, true);
        }
        setLayout(new GridBagLayout());
        GridBagConstraints gbh = new GridBagConstraints();
        gbh.insets = new Insets(5, 5, 5, 0);
        gbh.gridx = 0;
        gbh.gridy = 0;
        gbh.anchor = GridBagConstraints.WEST;
        add(UiUtilities.newLabel("System State: ", 0), gbh);
        gbh.gridx++;
        add(stateStatus, gbh);
        gbh.gridx++;
        gbh.insets.left = isFast ? 36 : 72;
        add(UiUtilities.newLabel(isFast ? "Fast Update Time: " : "Update Period: ", 0), gbh);
        gbh.gridx++;
        gbh.insets.left = 5;
        gbh.insets.top -= 2;
        add(periodTextField, gbh);
        gbh.gridx++;
        gbh.insets.top += 2;
        add(UiUtilities.newLabel(isFast ? "[[hr:]min:]sec" : "sec", 0), gbh);
    }

    @Override
    public void handleTextFieldX(String name, Object value) {
        if (isFast) {
            String[] fields = ((String)value).split(":");
            int fieldS = fields.length - 1;
            if (fieldS <= 2) {
                try {
                    periodMillis = 1000 * (Integer.decode(fields[fieldS]) 
                                             + 60 * (fieldS == 0 ? 0 : Integer.decode(fields[fieldS - 1])
                                                                         + 60 * (fieldS == 1 ? 0 : Integer.decode(fields[fieldS - 2]))));
                    sender.sendCommand(node, command, periodMillis);
                }
                catch (NumberFormatException e) {
                    //  No additional action needed
                }
            }
        }
        else {
            periodMillis = (int)(1000 * (Double)value);
            sender.sendCommand(node, command, periodMillis);
        }
        updatePeriod();
    }

    private void updatePeriod() {
        if (isFast) {
            int periodSecs = (periodMillis + 500) / 1000, periodMins = periodSecs / 60;
            periodTextField.update(String.format("%d:%02d:%02d", periodMins / 60, periodMins %60, periodSecs % 60), true);
        }
        else {
            periodTextField.update(periodMillis / 1000.0, true);
        }
    }

    private static final long serialVersionUID = 1L;
}
