package org.lsst.ccs.subsystem.common.ui.focalplane.view;

import java.awt.BorderLayout;
import java.awt.Color;
import java.util.*;
import java.util.concurrent.CancellationException;
import java.util.stream.Collectors;
import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JPanel;
import org.lsst.ccs.bus.data.ConfigurationParameterInfo;
import org.lsst.ccs.bus.states.DataProviderState;
import org.lsst.ccs.gconsole.annotations.services.persist.Create;
import org.lsst.ccs.gconsole.base.Console;
import org.lsst.ccs.gconsole.base.Const;
import org.lsst.ccs.gconsole.base.filter.AgentChannelsFilter;
import org.lsst.ccs.gconsole.plugins.monitor.AbstractMonitorView;
import org.lsst.ccs.gconsole.plugins.monitor.DefaultMonitorCell;
import org.lsst.ccs.gconsole.plugins.monitor.DisplayChannel;
import org.lsst.ccs.gconsole.plugins.monitor.MonitorField;
import org.lsst.ccs.gconsole.plugins.monitor.MonitorFormat;
import org.lsst.ccs.gconsole.plugins.monitor.MonitorView;
import org.lsst.ccs.gconsole.plugins.monitor.Updatable;
import org.lsst.ccs.gconsole.services.aggregator.AgentChannel;
import org.lsst.ccs.gconsole.services.persist.PersistenceService;
import org.lsst.ccs.subsystem.common.ui.focalplane.Segment;
import org.lsst.ccs.subsystem.common.ui.focalplane.filter.FocalPlaneFilter;
import org.lsst.ccs.subsystem.common.ui.focalplane.fpmap.AbstractFocalPlaneMapModel;
import org.lsst.ccs.subsystem.common.ui.focalplane.fpmap.FocalPlaneMap;
import org.lsst.ccs.subsystem.common.ui.focalplane.fpmap.FocalPlaneMapModelEvent;
import org.lsst.ccs.subsystem.common.ui.focalplane.fpmap.FocalPlaneMapValue;

/**
 * {@link MonitorView} that displays {@link FocalPlaneMap}.
 * All access to this class should happen on EDT (except status message listening handled internally by {@code AbstractMonitorView3}).
 * <p>
 * Implementation notes:<ul>
 * <li>On construction, the view sets its descriptor, creates an empty model and and an empty panel.
 * <li>On installation, the {@code installed} flag is set and {@code super.install()} is called.
 * <li>On un-installation, the {@code installed} flag is reset, the model and the panel are cleared, and {@code super.uninstall()} is called.
 * <li>On setFilter(...), saves the filter and, if the {@code installed} flag is {@code true}, un-installs and re-installs this view.
 * </ul>
 *
 * @author onoprien
 */
public class MapView extends AbstractMonitorView implements FocalPlaneView {

// -- Fields : -----------------------------------------------------------------
    
    static private final String FILTER = "Filter...";
    
    static private final Color COLOR_ALARM = Color.RED;
    static private final Color COLOR_WARNING = Color.ORANGE;
    static private final Color COLOR_NOMINAL = new Color(0, 200, 0);
    static private final int RGB_WARNING = COLOR_WARNING.getRGB() & 0xFFFFFF;
    static private final int RGB_NOMINAL = COLOR_NOMINAL.getRGB() & 0xFFFFFF;
    
    private final Descriptor descriptor;
    
//    private FocalPlaneFilter filter;
    private boolean installed = false;
    private final Model model;
    private final Region panel;

// -- Life cycle : -------------------------------------------------------------
    
    @Create(category = FocalPlaneView.CATEGORY,
            name = "Focal Plane Map",
            path = "Built-In/Map/Single",
            description = "Focal plane color coded map, one per page.")
    public MapView() {
        descriptor = new Descriptor();
        model = new Model();
        panel = new Region();
    }
    
    public MapView(Descriptor desc) {
        descriptor = desc.clone();
        model = new Model();
        panel = new Region();
        FocalPlaneFilter.Descriptor filterDescriptor = desc.getFilter();
        if (filterDescriptor != null) {
            PersistenceService service = Console.getConsole().getSingleton(PersistenceService.class);
            filter = (FocalPlaneFilter) service.make(filterDescriptor);
        }
    }

    @Override
    public JComponent getPanel() {
        return panel;
    }

    @Override
    public FocalPlaneFilter getFilter() {
        return (FocalPlaneFilter) super.getFilter();
    }

    @Override
    public void setFilter(AgentChannelsFilter filter) {
        if (!(filter instanceof FocalPlaneFilter)) throw new IllegalArgumentException(getClass().getName() +" only accepts FocalPlaneFilter filters");
        boolean isInstalled = installed;
        if (isInstalled) uninstall();
        super.setFilter(filter);
        descriptor.setFilter(getFilter().save());
        if (isInstalled) install();
    }

    @Override
    public void install() {
        if (!installed) {
            super.install();
            installed = true;
        }
    }

    @Override
    public void uninstall() {
        if (installed) {
            super.uninstall();
            model.clear();
            data.clear();
            panel.setGroups(null, null);
            installed = false;
        }
    }

    @Override
    public List<String> getGroups() {
        List<String> out = FocalPlaneView.super.getGroups();
        if (out == null) out = super.getGroups();
        return out;
    }
    

// -- Graphic component : ------------------------------------------------------
    
    private final class Region extends JPanel {

        private final FocalPlaneMap map;
        private final JLabel titleLabel;
        private final JComboBox<String> comboBox;
        
        private boolean armed = true;
        
        Region() {
            super(new BorderLayout());
            
            setBorder(BorderFactory.createCompoundBorder(
                    BorderFactory.createEmptyBorder(1, 1, 1, 1),
                    BorderFactory.createLineBorder(Color.BLACK, 1, true)));

            map = new FocalPlaneMap();
            map.setModel(model);
            add(map, BorderLayout.CENTER);

            Box header = Box.createHorizontalBox();
            add(header, BorderLayout.NORTH);

            comboBox = new JComboBox<>();
            header.add(comboBox);
            comboBox.addActionListener(e -> {
                if (armed) {
                    String s = (String) comboBox.getSelectedItem();
                    onGroupSelection(s);
                }
            });

            header.add(Box.createRigidArea(Const.HDIM));
            titleLabel = new JLabel(filter == null ? "" : filter.getName());
            header.add(titleLabel);

            header.add(Box.createHorizontalGlue());
        }
        
        /**
         * Sets the list of channel groups.
         * 
         * @param groups List of groups. If {@code null}, this region becomes empty and disabled.
         * @param selection Group to be selected.
         * @return Selected group.
         */
        String setGroups(List<String> groups, String selection) {
            armed = false;
            comboBox.removeAllItems();
            if (groups == null) {
                setEnabled(false);
            } else {
                setEnabled(true);
                comboBox.addItem(FILTER);
                groups.forEach(g -> comboBox.addItem(g));
                comboBox.setSelectedItem(selection);
            }
            armed = true;
            setTitle(filter.getName());
            return (String) comboBox.getSelectedItem();
        }
        
        String setSelectedGroup(String selection) {
            armed = false;
            comboBox.setSelectedItem(selection);
            armed = true;
            return (String) comboBox.getSelectedItem();
        }
        
        void setTitle(String title) {
            titleLabel.setText(title == null ? "" : title);
        }
    }
    
    
// -- Responding to external events : ------------------------------------------
    
    /**
     * Called in response to user selection in the groups combo box.
     */
    private void onGroupSelection(String group) {
        FocalPlaneFilter f = null;
        if (group == null || group.isEmpty()) {
            descriptor.setSelection(null);
            model.clear();
        } else if (FILTER.equals(group)) {
            PersistenceService service = Console.getConsole().getSingleton(PersistenceService.class);
            try {
                if (filter == null) {
                    f = (FocalPlaneFilter) service.make(descriptor.getFilter(), null, panel, FocalPlaneFilter.CATEGORY);
                } else {
                    f = (FocalPlaneFilter) service.edit(filter, null, panel);
//                    f = (FocalPlaneFilter) filter.edit(null, panel);
                }
            } catch (CancellationException x) {
            } catch (RuntimeException x) {
                Console.getConsole().error("Failed to create the filter", x);
            }
            if (f == null) {
                panel.setSelectedGroup(descriptor.getSelection());
            } else {
                setFilter(f);
            }
        } else {
            descriptor.setSelection(group);
            model.rebuild();
        }
    }

    /**
     * Called in response to status changes, when the set of channels changes.
     */
    @Override
    protected void resetChannels() {
        List<String> groups = getGroups();
        if (groups == null) groups = Collections.emptyList();
        String selection = descriptor.getSelection();
        if (!groups.contains(selection)) {
            if (groups.isEmpty()) {
                selection = null;
            } else {
                selection = groups.get(0);
            }
        }
        panel.setGroups(groups, selection);
        onGroupSelection(selection);
    }

    /**
     * Called in response to status changes, when the set of channels does not change.
     */
    @Override
    protected void update() {
        model.fireEvent();
    }
    
    
// -- Map model : --------------------------------------------------------------
    
    private class Model extends AbstractFocalPlaneMapModel {
        
        private Cell root;

        @Override
        public FocalPlaneMapValue getValue() {
            return FocalPlaneMapValue.EMPTY;
        }

        @Override
        public FocalPlaneMapValue getValue(int raftX, int raftY) {
            return get(raftX, raftY);
        }

        @Override
        public FocalPlaneMapValue getValue(int raftX, int raftY, int reb) {
           return get(raftX, raftY, reb);
        }

        @Override
        public FocalPlaneMapValue getValue(int raftX, int raftY, int reb, int ccdY) {
            return get(raftX, raftY, reb, ccdY);
        }

        @Override
        public FocalPlaneMapValue getValue(int raftX, int raftY, int reb, int ccdY, int ampX, int ampY) {
            return get(raftX, raftY, reb, ccdY, ampX, ampY);
        }
        
        private FocalPlaneMapValue get(int... indices) {
            if (root == null) return null;
            try {
                Cell cell = root.getChild(indices);
                if (cell == null) return null;
                FocalPlaneMapValue fv = cell.getFormattedValue();
                return fv == null ? format(cell) : fv;
            } catch (ArrayIndexOutOfBoundsException x) {
                return null;
            }
        }
        
        void fireEvent() {
            notifyListeners(new FocalPlaneMapModelEvent(this));
        }
        
        void clear() {
            root = new Cell();
            root.setFormattedValue(FocalPlaneMapValue.NONE);
            data.values().forEach(channel -> channel.setTarget(null));
            fireEvent();
        }
        
        void rebuild() {

            root = new Cell();
            root.setFormattedValue(FocalPlaneMapValue.NONE);

            Iterator<Map.Entry<String, DisplayChannel>> it = data.entrySet().iterator();
            while (it.hasNext()) {
                DisplayChannel ch = it.next().getValue();
                String path = ch.getPath();
                int[] indices = Segment.getIndices(path);
                if (indices == null) {
                    it.remove();
                    continue;
                }
                String group = getGroup(path);
                if (group.equals(descriptor.selection)) {
//                    if (indices[0] % 4 == 0 && indices[1] % 4 == 0 && indices[2] == 0) indices[3] = -1; // both SW0 and SW1 go to RebW
                    Cell cell = addOrGetChild(indices);
                    cell.add(ch);
                    ch.setTarget(cell);
                }
            }

            root.trim();
            fireEvent();
        }
        
        private Cell addOrGetChild(int... indices) {
            Cell cell = root;
            for (int i = 0; i < indices.length; i++) {
                int index = indices[i];
                if (index == -1) {
                    return cell;
                } else {
                    if (cell.children == null) cell.children = new Cell[Segment.N[i]];
                    Cell child = cell.children[index];
                    if (child == null) {
                        child = new Cell();
                        child.parent = cell;
                        cell.children[index] = child;
                    }
                    cell = child;
                }
            }
            return cell;
        }
        
    }
    
    static private class Cell extends DefaultMonitorCell implements Updatable {
                
        Cell() {
            super(new ArrayList<>(1), MonitorField.NULL);
        }
        
        Cell parent;
        Cell[] children;
        
        void add(DisplayChannel channelHandle) {
            channels.add(channelHandle);
        }
        
        Cell getChild(int... indices) {
            Cell cell = this;
            for (int i = 0; i < indices.length; i++) {
                int index = indices[i];
                if (index == -1) {
                    return cell;
                } else {
                    if (cell.children == null) return null;
                    cell = cell.children[index];
                    if (cell == null) return null;
                }
            }
            return cell;
        }
        
        void trim() {
            if (channels.isEmpty()) {
                channels = Collections.emptyList();
            } else {
                ((ArrayList)channels).trimToSize();
            }
            if (children != null) {
                for (Cell c : children) {
                    if (c != null) c.trim();
                }
            }
        }

        @Override
        public void update(DisplayChannel channelHandle, List<MonitorField> fields) {
            update(channelHandle);
        }

        @Override
        public void update(DisplayChannel channelHandle) {
            if (! channels.isEmpty()) setFormattedValue(null);
            if (children != null) {
                for (Cell c : children) {
                    if (c != null) {
                        c.update(channelHandle);
                    }
                }
            }
        }

        @Override
        public FocalPlaneMapValue getFormattedValue() {
            return (FocalPlaneMapValue) super.getFormattedValue();
        }
        
    }

    
// -- Formatting : ------------------------------------------------------------- 
    
    private FocalPlaneMapValue format(Cell cell) {
        
        if (cell.getChannels().isEmpty()) {
            cell.setFormattedValue(FocalPlaneMapValue.NONE);
            return FocalPlaneMapValue.NONE;
        }
        
        FocalPlaneMapValue parentValue = FocalPlaneMapValue.NONE;
        Cell parent = cell.parent;
        while (parentValue == FocalPlaneMapValue.NONE && parent != null) {
            parentValue = parent.getFormattedValue();
            if (parentValue == null) {
                format(parent);
                parentValue = parent.getFormattedValue();
            }
            parent = parent.parent;
        }
        
        FocalPlaneMapValue fv = new FocalPlaneMapValue();
        fv.setSplit(cell.children != null);
        fv.bgColor = parentValue == FocalPlaneMapValue.NONE ? null : parentValue.getBgColor();
        
        List<AgentChannel> originalChannels = cell.getChannels().stream().flatMap(dc -> dc.getChannels().stream()).collect(Collectors.toList());
        if (fv.bgColor == null && originalChannels.size() == 1) {
            AgentChannel channel = originalChannels.get(0);
            if (channel == null) {
                fv.bgColor = MonitorFormat.COLOR_OFF;
            } else {
                fv.toolTip = "<html>"+ channel.getPath() +"<br>"+ channel.get();
                fv.bgColor = computeColor(channel);
            }
        } else {
            for (AgentChannel ch : originalChannels) {
                Color c = computeColor(ch);
                fv.bgColor = mergeColor(c, fv.bgColor);
            }
        }
        
        cell.setFormattedValue(fv);
        return fv;
    }
    
    private Color computeColor(AgentChannel channel) {
        DataProviderState state = channel.get(AgentChannel.Key.STATE);
        Color color = Color.LIGHT_GRAY;
        
        if (state != null) {
            switch (state) {
                case ALARM:
                    return Color.RED;
                case WARNING:
                    color = COLOR_WARNING;
                    break;
                case NOMINAL:
                    int alpha = 255;
                    try {
                        double v = getDouble(channel, AgentChannel.Key.VALUE);
                        double low, high;
                        try {
                            low = getDouble(channel, AgentChannel.Key.LOW_WARN);
                            low = getDouble(channel, AgentChannel.Key.LOW_ALARM) + low;
                        } catch (RuntimeException x) {
                            low = getDouble(channel, AgentChannel.Key.LOW_ALARM);
                        }
                        try {
                            high = getDouble(channel, AgentChannel.Key.HIGH_WARN);
                            high = getDouble(channel, AgentChannel.Key.HIGH_ALARM) - high;
                        } catch (RuntimeException x) {
                            high = getDouble(channel, AgentChannel.Key.HIGH_ALARM);
                        }
                        double d = (high-low)/4.;
                        double a = Math.min((v-low)/d, (high-v)/d);
                        alpha = (int) (a*255);
                        if (alpha >= 255) {
                            return COLOR_NOMINAL;
                        } else if (alpha < 70) {
                            alpha = 70;
                        }
                        return new Color(RGB_NOMINAL | (alpha << 24), true);
                    } catch (RuntimeException x) {
                        return COLOR_NOMINAL;
                    }
                default:
                    color = MonitorFormat.COLOR_STATE.get(state);
            }
        }
        return color;
    }
    
    private Color mergeColor(Color c1, Color c2) {
        if (c2 == null) return c1; // no default
        if (COLOR_ALARM.equals(c1) || COLOR_ALARM.equals(c2)) return COLOR_ALARM; // one is RED
        int rgb1 = c1.getRGB() & 0xffffff;
        int rgb2 = c2.getRGB() & 0xffffff;
        if (rgb1 != RGB_WARNING && rgb1 != RGB_NOMINAL) return c2; // c1 neither GREEN nor YELLOW
        if (rgb2 != RGB_WARNING && rgb2 != RGB_NOMINAL) return c1; // c2 neither GREEN nor YELLOW
        if (rgb1 == RGB_NOMINAL) {
            if (rgb2 == RGB_NOMINAL) {
                return c1.getAlpha() < c2.getAlpha() ? c1 : c2;
            } else {
                return c2;
            }
        } else {
            if (rgb2 == RGB_NOMINAL) {
                return c1;
            } else {
                return c1.getAlpha() > c2.getAlpha() ? c1 : c2;
            }
            
        }
    }
    
    private double getDouble(AgentChannel channel, String key) {
        Object o = channel.get(key);
        if (o == null) throw new RuntimeException();
        if (o instanceof ConfigurationParameterInfo) {
            o = ((ConfigurationParameterInfo)o).getCurrentValue();
        }
        double value;
        if (o instanceof Double) {
            value = (Double)o;
        } else {
            value = Double.parseDouble(o.toString());
        }
        return value;
    }
    
    
// -- Saving/restoring : -------------------------------------------------------
    
    static public class Descriptor extends FocalPlaneView.Descriptor {

        private FocalPlaneFilter.Descriptor filter;
        private String selection;

        public FocalPlaneFilter.Descriptor getFilter() {
            return filter;
        }

        public void setFilter(FocalPlaneFilter.Descriptor filter) {
            this.filter = filter;
        }

        public String getSelection() {
            return selection;
        }

        public void setSelection(String selection) {
            this.selection = selection;
        }

        @Override
        public Descriptor clone() {
            Descriptor desc = (Descriptor) super.clone();
            if (desc.filter != null) {
                 desc.filter =  desc.filter.clone();
            }
            return desc;
        }
        
    }

    @Override
    public Descriptor getDescriptor() {
        return descriptor;
    }

}
