package org.lsst.ccs.subsystem.common.ui.jas;

import java.time.Duration;
import java.util.Collections;
import java.util.HashSet;
import java.util.Set;
import org.lsst.ccs.gconsole.agent.command.CommandTask;
import org.lsst.ccs.gconsole.services.command.CommandService;
import org.lsst.ccs.gconsole.services.command.DefaultCommandHandle;

/**
 * Class for sending a command and receiving the response.
 * 
 * @author saxton
 */
public class CommandSender {

    public interface ReplyHandler {

        public void onCommandReply(Object reply, String path, String command, Object[] args);

    }

    public class CommandHandler extends DefaultCommandHandle {
        
        @Override
        public void onSuccess(Object reply, CommandTask task) {
            String command = task.getCommand();
            int first = command.indexOf('/');
            int last = command.lastIndexOf('/');
            String path = first == last ? null : command.substring(first + 1, last);
            String cmnd = command.substring(last + 1);
            for (ReplyHandler handler : handlers) {
                handler.onCommandReply(reply, path, cmnd, task.getArguments());
            }
        }

    }

    private final String dest;
    private final CommandService service = CommandService.getService();
    private final CommandHandler cmndHandler = new CommandHandler();
    private final Set<ReplyHandler> handlers = Collections.synchronizedSet(new HashSet<>());
    private int defTimeout = 5000;

    /**
     * Constructor
     *
     * @param dest The name of the destination subsystem for the commands
     */
    public CommandSender(String dest) {
        this(dest, null);
    }

    /**
     * Constructor
     *
     * @param dest The name of the destination subsystem for the commands
     * @param handler The handler for command replies
     */
    public CommandSender(String dest, ReplyHandler handler) {
        this.dest = dest;
        handlers.add(handler);
    }

    /**
     * Sends a command without expecting a reply.
     * 
     * The default timeout is used
     * 
     * @param path The path within the destination to receive the command,
     *             or null if the top level
     * @param cmnd The command to execute
     * @param args The command arguments
     */
    public void sendCommand(String path, String cmnd, Object... args) {
        sendCommand(false, defTimeout, path, cmnd, args);
    }

    /**
     * Sends a command without expecting a reply.
     * 
     * @param timeout The command timeout (ms)
     * @param path    The path within the destination to receive the command,
     *                or null if the top level
     * @param cmnd    The command to execute
     * @param args    The command arguments
     */
    public void sendCommand(int timeout, String path, String cmnd, Object... args) {
        sendCommand(false, timeout, path, cmnd, args);
    }

    /**
     * Sends a command.
     * 
     * The default timeout is used
     * 
     * @param reply Whether to handle the command reply
     * @param path  The path within the destination to receive the command,
     *              or null if the top level
     * @param cmnd  The command to execute
     * @param args  The command arguments
     */
    public void sendCommand(boolean reply, String path, String cmnd, Object... args) {
        sendCommand(reply, defTimeout, path, cmnd, args);
    }

    /**
     * Sends a command.
     * 
     * @param reply   Whether to handle the command reply
     * @param timeout The command timeout (ms)
     * @param path    The path within the destination to receive the command,
     *                or null if the top level
     * @param cmnd    The command to execute
     * @param args    The command arguments
     */
    public void sendCommand(boolean reply, int timeout, String path, String cmnd, Object... args) {
        String cmd = dest + (path == null ? "" : "/" + path) + "/" + cmnd;
        service.execute(reply ? cmndHandler : null, Duration.ofMillis(timeout), cmd, args);
    }

    /**
     * Sets the default timeout.
     * 
     * @param timeout The default timeout (ms).  If zero, the command waits forever.
     *                If positive, the maximum of this value and the ACK custom timeout is used.
     *                If negative, its absolute value is used.  The initial value is 5000.
     */
    public void setTimeout(int timeout) {
        defTimeout = timeout;
    }

    /**
     * Adds a reply handler
     *
     * @param handler The handler to add
     */
    public void addHandler(ReplyHandler handler) {
        handlers.add(handler);
    }

    /**
     * Removes a reply handler
     *
     * @param handler The handler to remove
     */
    public void removeHandler(ReplyHandler handler) {
        handlers.remove(handler);
    }

}
