package org.lsst.ccs.subsystem.common.ui.focalplane;

import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.lsst.ccs.bus.data.DataProviderInfo;
import org.lsst.ccs.gconsole.services.aggregator.AgentChannel;

/**
 * Enumeration of the Focal Plane standard hardware segments.
 *
 * @author onoprien
 */
public enum Segment {
    
// -- Enum constants : ---------------------------------------------------------
    
    RAFT(3,3, DataProviderInfo.Attribute.RAFT_ID.getName(), "R"),
    REB(3,1, DataProviderInfo.Attribute.REB_ID.getName(), "Reb"),
    CCD(1,3, DataProviderInfo.Attribute.SENSOR_ID.getName(), "S"),
    AMP(2,8, DataProviderInfo.Attribute.AMPLIFIER_ID.getName(), "Seg");
    
// -- Fields : -----------------------------------------------------------------
    
    final int nX, nY;
    final String key;
    final String path;
    
    static public final String AGENT_GROUP = "agent";
    static public final String GROUP_GROUP = "group";
    // groups that capture RAFT, REB,CCD, and AMP numbers are referenced by enum's name
    // expected format: [subsystem/]RXX[/RebX][/SXX[/SegmentXX]]/group
    static public final Pattern FP_PATH_PATTERN = Pattern.compile("(?:(?<"+ AGENT_GROUP +">[^/]+)/+)?"+ RAFT.path +"(?<"+ RAFT.name() +">[0-4]{2})(?:/+"+ REB.path +"(?<"+ REB.name() +">[0-2WG]))?(?:/+"+ CCD.path +"(?<"+ CCD.name() +">W|(?:[0-2WG][0-2]))(?:/+"+ AMP.path +"(?<"+ AMP.name() +">[0-1][0-7]))?)?(?:/+(?<"+ GROUP_GROUP +">.+))?");
    
// -- Life cycle : -------------------------------------------------------------
    
    private Segment(int nX, int nY, String key, String path) {
        this.nX = nX;
        this.nY = nY;
        this.key = key;
        this.path = path;
    }

// -- Getters : ----------------------------------------------------------------
    
    public int getNX() {
        return nX;
    }
    
    public int getNY() {
        return nY;
    }

    /**
     * Returns the attribute key that can be used to retrieve the corresponding segment index value from {@link DataProviderInfo}.
     * @return Attribute key.
     */
    public String getKey() {
        return key;
    }

    /**
     * Returns string representation of this segment type.
     * This representation should be used for constructing or parsing channel paths,
     * and anywhere human-readable text needs to be produced.
     * @return String representation of this segment type.
     */
    @Override
    public String toString() {
        return path;
    }
    
    
// -- Utility methods : --------------------------------------------------------
    
    /**
     * Computes focal plane segment indices for the given channel, based on attributes.
     * 
     * @param channel Data channel.
     * @return Array of length 6: [raftX, raftY, reb = ccdX, ccdY, ampX, ampY], or {@code null} if the channel does not belong to a valid segment.
     *         Trailing indices will be -1 if there is no corresponding segment at this level, but at least raft indices will have valid non-negative values.
     */
    static public int[] getIndices(AgentChannel channel) {
        return getIndices(channel.getPath()); // Not using attributes anymore
//        int[] indices = {-1, -1, -1, -1, -1, -1};
//        int raftX, raftY;
//        String ID;
//        try {
//            ID = channel.get(DataProviderInfo.Attribute.RAFT_ID.getName());
//            raftX = Integer.parseInt(ID.substring(1, 2));
//            raftY = Integer.parseInt(ID.substring(2, 3));
//            if (raftX < 0 || raftX > 4 || raftY < 0 || raftY > 4) return null;
//            indices[0] = raftX;
//            indices[1] = raftY;
//            ID = channel.get(DataProviderInfo.Attribute.REB_ID.getName());
//            int reb = -1;
//            if (ID != null) {
//                String sreb = ID.substring(3, 4);
//                if ("W".equals(sreb)) {
//                    reb = 0;
//                } else if ("G".equals(sreb)) {
//                    reb = 1;
//                } else {
//                    reb = Integer.parseInt(sreb);
//                    if (reb < 0 || reb > 2) return null;
//                }
//                indices[2] = reb;
//            }
//            ID = channel.get(DataProviderInfo.Attribute.SENSOR_ID.getName());
//            if (ID != null) {
//                int ccdX = Integer.parseInt(ID.substring(1, 2));
//                int ccdY = Integer.parseInt(ID.substring(2, 3));
//                if (ccdX < 0 || ccdX > 2 || ccdY < 0 || ccdY > 2 || (reb != -1 && reb != ccdX)) return null;
//                indices[2] = ccdX;
//                indices[3] = ccdY;
//                ID = channel.get(DataProviderInfo.Attribute.AMPLIFIER_ID.getName());
//                if (ID != null) {
//                    int ampX = Integer.parseInt(ID.substring(0, 1));
//                    int ampY = Integer.parseInt(ID.substring(1, 2));
//                    if (ampX < 0 || ampX > 1 || ampY < 0 || ampY > 7) return null;
//                    indices[4] = ampX;
//                    indices[5] = ampY;
//                }
//            }
//            return indices;
//        } catch (IndexOutOfBoundsException | NullPointerException | NumberFormatException x) {
//            return null;
//        }
    }
    
    /**
     * Computes focal plane segment indices for the given path.
     * 
     * @param path Original or display path in {@code [subsystem/]RXX[/RebX][/SXX[/SegmentXX]]/group} format.
     * @return Array of length 6: [raftX, raftY, reb = ccdX, ccdY, ampX, ampY], or {@code null} if the channel does not belong to a valid segment.
     *         Trailing indices will be -1 if there is no corresponding segment at this level, but at least raft indices will have valid non-negative values.
     */
    static public int[] getIndices(String path) {
        Matcher m = FP_PATH_PATTERN.matcher(path);
        if(m.matches()) {
            int[] indices = {-1, -1, -1, -1, -1, -1};
            String s = m.group(RAFT.name());
            indices[0] = Integer.parseInt(s.substring(0, 1));
            indices[1] = Integer.parseInt(s.substring(1));        
            s = m.group(REB.name());
            if (s != null) {
                boolean isW;
                if ("W".equals(s)) {
                    indices[2] = 0;
                    isW = true;
                } else if ("G".equals(s)) {
                    indices[2] = 1;
                    isW = false;
                } else {
                    indices[2] = Integer.parseInt(s);
                    isW = false;
                }
                s = m.group(CCD.name());
                if (s != null) {
                    indices[3] = isW ? 1 : Integer.parseInt(s.substring(1)); // W or SW1 --> 0 1
                    s = m.group(AMP.name());
                    if (s != null) {
                        indices[4] = Integer.parseInt(s.substring(0, 1));
                        indices[5] = Integer.parseInt(s.substring(1));
                    }
                }
            }
            return indices;
        } else {
            return null;
        }
    }
    
    /**
     * Returns standardized path prefix for a channel with given segment indices.
     * The prefix format is [RXX/[RebX/[SXX/[SegmentXX/]]]] (the REB segment is always present if the CCD segment is)
     * The returned prefix is empty if raft indices are "-1" or the argument is {@code null}.
     * 
     * @param indices Array of length 6: [raftX, raftY, reb = ccdX, ccdY, ampX, ampY].
     * @return Path prefix.
     */
    static public String getPathPrefix(int[] indices) {
        if (indices == null) return "";
        StringBuilder sb = new StringBuilder();
        if (indices[0] != -1) {
            sb.append(RAFT).append(indices[0]).append(indices[1]).append("/");
            if (indices[2] != -1) {
                char reb;
                if (indices[0]%4 == 0 && indices[1]%4 == 0) { // corner
                    reb = indices[2] == 0 ? 'W' : 'G';
                } else {
                    reb = Integer.toString(indices[2]).charAt(0);
                }
                sb.append(REB).append(reb).append("/");
                if (indices[3] != -1) {
                    sb.append(CCD).append(reb);
                    if (reb != 'W') {
                        sb.append(indices[3]);
                    }
                    sb.append("/");
                    if (indices[4] != -1) {
                        sb.append(AMP).append(indices[4]).append(indices[5]).append("/");
                    }
                }
            }
        }
        return sb.toString();
    }
    
    /**
     * Returns standardized path prefix for the specified channel.
     * The prefix format is [RXX/[RebX/[SXX/[SegmentXX/]]]] (the REB segment is always present if the CCD segment is)
     * This method constructs the prefix based on channel attributes, not on channel path.
     * The returned prefix is empty if the channel does not belong to a valid segment; otherwise, the prefix ends with "/";
     * 
     * @param channel Data channel.
     * @return Path prefix.
     */
    static public String getPathPrefix(AgentChannel channel) {
        return getPathPrefix(getIndices(channel));
    }
    
    /**
     * Returns standardized path prefix for the specified channel.
     * The returned prefix is empty if the provided path does not allow determination of the focal plane segment is belongs to.
     * 
     * @param path Data channel path.
     * @return Path prefix.
     */
    static public String getPathPrefix(String path) {
        return getPathPrefix(getIndices(path));
    }
    
    /**
     * Strips path prefix indicating the segment location on the focal plane from the given path.
     * The path should be in {@code [subsystem/]RXX[/RebX][/SXX[/SegmentXX]]/group} format.
     * The returned string is {@code [subsystem/]group}.
     * If the path does not contain a prefix in the expected format, the entire path is returned.
     * 
     * @param path Original or display path
     * @return Path with the prefix stripped.
     */
    static public String stripPathPrefix(String path) {
        Matcher m = FP_PATH_PATTERN.matcher(path);
        if(m.matches()) {
            String group = m.group(GROUP_GROUP);
            String agent = m.group(AGENT_GROUP);
            if (agent == null) {
                if (group == null) {
                    return "";
                } else {
                    return group;
                }
            } else {
                if (group == null) {
                    return agent;
                } else {
                    return agent +"/"+ group;
                }
            }
        } else {
            return path;
        }
    }
    
// -- Static constants : -------------------------------------------------------
    
    static public final int[] N = {5,5,3,3,2,8};
    
}
