package org.lsst.ccs.subsystem.common.ui;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.FocusAdapter;
import java.awt.event.FocusEvent;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.JTextField;
import javax.swing.SwingConstants;
import javax.swing.border.LineBorder;
import javax.swing.border.TitledBorder;

/**
 * Some useful UI utility routines
 *
 * @author saxton
 */
public class UiUtilities {

    public interface ActionHandler {

        public default void handleButton(String name) {}

        public default void handleRadioButton(String name) {}

        public default void handleTextField(String name, String text) {}

        public default void handleTextFieldX(String name, Object value) {}

        public default void handleCheckBox(String name, boolean isSelected) {}

    }

    private final ActionHandler handler;

    public UiUtilities(ActionHandler handler) {
        this.handler = handler;
    }

    public static JLabel newLabel(String text, int width) {
        JLabel label = new JLabel(text);
        label.setFont(UiConstants.FONT);
        Dimension d = label.getPreferredSize();
        if (width > 0) {
            d.width = width;
        }
        label.setPreferredSize(d);
        return label;
    }

    public JRadioButton newRadioButton(String title, String name) {
        JRadioButton rb = new JRadioButton(title);
        rb.setFont(UiConstants.FONT);
        rb.setFocusable(false);
        rb.setName(name);
        rb.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent evt) {
                handler.handleRadioButton(((JRadioButton)evt.getSource()).getName());
            }
        });
        return rb;
    }

    public JButton newButton(String title, String name) {
        return newButton(title, name, false);
    }

    public JButton newButton(String title, String name, boolean fullSize) {
        JButton btn = new JButton(title);
        if (!fullSize) {
            Dimension charSize = getSize("H");
            Insets i = btn.getMargin();
            i.left -= charSize.width / 2;
            i.right -= charSize.width / 2;
            btn.setMargin(i);
            Dimension d = btn.getPreferredSize();
            d.height -= charSize.height / 2;
            btn.setPreferredSize(d);
        }
        btn.setFont(UiConstants.FONT);
        btn.setFocusable(false);
        btn.setName(name);
        btn.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent evt) {
                handler.handleButton(((JButton)evt.getSource()).getName());
            }
        });
        return btn;
    }

    public JTextField newTextField(String text, String name) {
        JTextField tf = new JTextField(text);
        Dimension d = tf.getPreferredSize();
        tf.setPreferredSize(d);
        tf.setHorizontalAlignment(SwingConstants.CENTER);
        tf.setName(name);
        tf.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent evt) {
                JTextField tf = (JTextField)evt.getSource();
                handler.handleTextField(tf.getName(), tf.getText());
            }
        });
        return tf;
    }

    public TextFieldX newTextFieldX(String text, String name, int type) {
        TextFieldX tf = new TextFieldX(text, type);
        Dimension d = tf.getPreferredSize();
        tf.setPreferredSize(d);
        tf.setHorizontalAlignment(SwingConstants.CENTER);
        tf.setName(name);
        tf.addFocusListener(new FocusAdapter() {
            @Override
            public void focusLost(FocusEvent evt) {
                ((TextFieldX)evt.getSource()).focusLost();
            }
        });
        tf.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent evt) {
                TextFieldX tf = (TextFieldX)evt.getSource();
                Object value = tf.action();
                if (value != null) {
                    handler.handleTextFieldX(tf.getName(), value);
                }
            }
        });
        tf.addKeyListener(new KeyAdapter() {
            @Override
            public void keyTyped(KeyEvent evt) {
                ((TextFieldX)evt.getSource()).keyTyped(evt.getKeyChar());
            }
        });
        return tf;
    }

    public JCheckBox newCheckBox(String title, String name) {
        JCheckBox cbx = new JCheckBox(title);
        cbx.setFont(UiConstants.FONT);
        cbx.setFocusable(false);
        cbx.setName(name);
        cbx.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent evt) {
                JCheckBox cbx = (JCheckBox)evt.getSource();
                handler.handleCheckBox(cbx.getName(), cbx.isSelected());
            }
        });
        return cbx;
    }

    public static JPanel newBorderedPanel(String title) {
        JPanel panel = new JPanel();
        TitledBorder border = new TitledBorder(new LineBorder(Color.BLACK), title);
        border.setTitleJustification(TitledBorder.CENTER);
        border.setTitleFont(UiConstants.FONT);
        border.setTitleColor(UiConstants.BLUE);
        panel.setBorder(border);
        panel.setLayout(new GridBagLayout());
        return panel;
    }

    public static int maxLabelWidth(String[] text, String term) {
        JLabel label = new JLabel();
        label.setFont(UiConstants.FONT);
        int width = 0;
        for (String item : text) {
            label.setText(item + term);
            width = Math.max(width, label.getPreferredSize().width);
        }
        return width;
    }

    public static int maxEnumLabelWidth(Class enumClass) {
        JLabel label = new JLabel();
        label.setFont(UiConstants.FONT);
        int width = 0;
        for (Object en : enumClass.getEnumConstants()) {
            label.setText(en.toString());
            width = Math.max(width, label.getPreferredSize().width);
        }
        return width;
    }

    public static Dimension getSize(String text) {
        JLabel label = new JLabel(text);
        label.setFont(UiConstants.FONT);
        return label.getPreferredSize();
    }

}
