package org.lsst.ccs.subsystem.common.ui.focalplane.filter;

import org.lsst.ccs.subsystem.common.ui.focalplane.Segment;
import java.io.Serializable;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.lsst.ccs.gconsole.services.aggregator.AgentChannel;

/**
 * A template that defines a class of data channels - one channel per focal plane segment.
 *
 * @author onoprien
 */
public class Template implements Serializable, Comparable<Template> {

// -- Fields : -----------------------------------------------------------------
    
    public final String code; // "[agent]/non-geometry-path"
    public final Segment segment;
    
    static private final Pattern fromString = Pattern.compile("(.+) \\(per (\\w+)\\)");

// -- Life cycle : -------------------------------------------------------------
    
    public Template(String code, Segment segment) {
        this.code = code;
        this.segment = segment;
    }
    
    
// -- Getters : ----------------------------------------------------------------
    
    public String getAgent() {
        return code.substring(0, code.indexOf("/"));
    }
    
    public boolean hasAgent() {
        return !code.startsWith("/");
    }
    
    
// -- Operations : -------------------------------------------------------------
    
    public Template stripAgent() {
        int i = code.indexOf("/");
        if (i == 0) {
            return this;
        } else {
            return new Template(code.substring(i), segment);
        }
    }
    
    /**
     * Constructs a template corresponding to the specified channel.
     * 
     * @param channel Data channel.
     * @return Template, or {@code null} if the supplied channel does not belong to a location on the focal plane.
     */
    static public Template valueOf(AgentChannel channel) {
        return valueOf(channel.getPath());
    }
    
    /**
     * Constructs a template corresponding to the specified channel path.
     * 
     * @param path Data channel path.
     * @param indices Array of length 6: [raftX, raftY, reb = ccdX, ccdY, ampX, ampY].
     * @return Template, or {@code null} if the {@code indices} is {@code null}.
     */
    static public Template valueOf(String path, int[] indices) {
        if (indices == null) return null;
        Segment segment = Segment.RAFT;
        String p = "/"+ Segment.RAFT.toString() + indices[0] + indices[1];
        path = path.replaceFirst(p, "");
        if (indices[2] != -1) {
            segment = Segment.REB;
            p = "/"+ Segment.REB.toString() + indices[2];
            path = path.replaceFirst(p, "");
            if (indices[3] != -1) {
                segment = Segment.CCD;
                p = "/"+ Segment.CCD.toString() + indices[2] + indices[3];
                path = path.replaceFirst(p, "");
                if (indices[4] != -1) {
                    segment = Segment.AMP;
                    p = String.valueOf(indices[4]) + indices[5];
                    path = path.replaceFirst(p, "");
                }
            }
        }
        return new Template(path, segment);
    }
    
    /**
     * Constructs a template corresponding to the specified channel path.
     * 
     * @param path Data channel path.
     * @return Template, or {@code null} if the supplied path does not correspond to a location on the focal plane.
     */
    static public Template valueOf(String path) {
        Matcher m = Segment.FP_PATH_PATTERN.matcher(path);
        if(m.matches()) {
            String group = m.group(Segment.GROUP_GROUP);
            if (group == null) return null;
            String agent = m.group(Segment.AGENT_GROUP);
            if (agent == null) agent = "";
            String code = agent +"/"+ group;
            Segment segment = Segment.RAFT;
            if (m.group(Segment.AMP.name()) != null) {
                segment = Segment.AMP;
            } else if (m.group(Segment.CCD.name()) != null) {
                segment = Segment.CCD;
            } else if (m.group(Segment.REB.name()) != null) {
                segment = Segment.REB;
            }
            return new Template(code, segment);
        } else {
            return null;
        }
    }
    
    /**
     * Constructs a template corresponding to the specified String representation.
     * 
     * @param string String representation created by {@code toString()} method.
     * @return Template.
     * @throws IllegalArgumentException if the specified String cannot be converted to a template.
     */
    static public Template fromString(String string) {
        Matcher m = fromString.matcher(string);
        if (m.matches()) {
            String code = m.group(1);
            Segment segment = Segment.valueOf(m.group(2));
            return new Template(code, segment);
        } else {
            throw new IllegalArgumentException("Cannot convert to template: "+ string);
        }
    }
    
    static public String getAgent(String encodedTemplate) {
        return encodedTemplate.substring(0, encodedTemplate.indexOf("/"));
    }
    
// -- Overriding Object : ------------------------------------------------------

    /**
     * Produces a String representation of this template.
     * The string is human-readable, and can be converted back to a {@code Template}
     * by calling {@code valueOf(String string)} method.
     * 
     * @return String representation of this template.
     */
    @Override
    public String toString() {
        return code +" (per "+ segment.name() +")";
    }

    @Override
    public boolean equals(Object obj) {
        if (obj instanceof Template) {
            Template other = (Template) obj;
            return (code + segment).equals(other.code + other.segment);
        } else {
            return false;
        }
    }

    @Override
    public int hashCode() {
        return (code + segment).hashCode();
    }

    @Override
    public int compareTo(Template other) {
        return (code + segment).compareTo(other.code + other.segment);
    }

}
