package org.lsst.ccs.subsystem.common.ui.focalplane.filter;

import java.util.*;
import java.util.stream.Collectors;
import org.lsst.ccs.bus.data.DataProviderInfo;
import org.lsst.ccs.gconsole.base.filter.AbstractChannelsFilter;
import org.lsst.ccs.gconsole.base.filter.AgentChannelsFilter;
import org.lsst.ccs.gconsole.base.filter.PersistableAgentChannelsFilter;
import org.lsst.ccs.gconsole.services.aggregator.AgentChannel;
import org.lsst.ccs.gconsole.services.aggregator.AgentStatusAggregator;
import org.lsst.ccs.gconsole.services.aggregator.AgentStatusListener;
import org.lsst.ccs.subsystem.common.ui.focalplane.Segment;

/**
 * Base class for channels filters used with focal plane views.
 * <p>
 * {@code FocalPlaneFilter} selects some of the data channels associated with particular focal plane segments, 
 * and splits them into one or more groups. The number of groups should be small enough for monitoring views to
 * conveniently display (typically 10 or less). Some views treat groups as columns in a table, others provide
 * combo boxes for the user to select the group to be displayed.
 * <p>
 * Filters extending this class specify (possibly many-to-many) mappings between original published data channels
 * and display paths in the <br>
 * {@code RXX[[/RebX][/SXX[/SegmentXX]]]/GroupName}<br>
 * format.
 * <p>
 * This class conforms to the general {@link AgentChannelsFilter} contract, so its subclasses can be used as 
 * general purpose filters by any tool. It also provides two additional public methods - {@link #getGroups()} and 
 * {@link #getGroupFields()}, focal plane specific default implementations for several methods, and utility
 * methods intended to simplify implementing concrete subclasses.
 * <p>
 * To be useful, subclasses should implement at least one method - 
 * {@link #getDisplayPaths(AgentChannel) getDisplayPaths(channel)} - either directly or by overriding one of the
 * methods its default implementation uses: {@link #getDisplayPath(AgentChannel) getDisplayPath(channel)} or
 * {@link #getDisplayPath(String) getDisplayPath(originalPath)}.
 * <p>
 * {@link Segment} class provides static utility methods that can be useful for implementing focal plane filters.
 *
 * @author onoprien
 */
abstract public class FocalPlaneFilter extends AbstractChannelsFilter {
    
    /**
     * Persistable category for focal plane filters.
     */
    static public final String CATEGORY = "FocalPlaneFilter";
    
// -- Implementing AgentChannelsFilter : ---------------------------------------

    /**
     * Returns a list of subsystem names accepted by this filter, or {@code null} if any subsystem might be accepted.
     * <p>
     * The implementation provided by this class returns {@code null}.
     * 
     * @return Names of accepted agents.
     */
    @Override
    public List<String> getAgents() {
        return null;
    }

    /**
     * Returns a list of original channel paths accepted by this filter.
     * The list may contain templates and selectors. See 
     * {@link AgentStatusAggregator#addListener(AgentStatusListener, Collection, Collection) AgentStatusAggregator.addListener(listener, agents, channels)}
     * for details.
     * <p>
     * The implementation provided by this class returns a singleton list that contains a selector for channels that have RAFT ID attribute.
     * 
     * @return Original channel paths accepted by this filter, or {@code null} if any original path for which
     *         {@link #getDisplayPaths(AgentChannel) getDisplayPaths(channel)} returns a non-empty list is accepted.
     */
    @Override
    public List<String> getOriginChannels() {
        return Collections.singletonList(DataProviderInfo.Attribute.RAFT_ID.getName());
    }

    /**
     * Returns the original path corresponding to the specified display channel.
     * If multiple original paths are mapped to the specified display path, this
     * method may return one of them, or {@code null}.
     * <p>
     * None of the currently known focal plane views call this method directly. However, it is called by
     * {@link #getOriginPaths(String) getOriginPaths(displayPath)} implementation unless the latter is overridden.
     * <p>
     * The implementation provided by this class returns {@code null}.
     * 
     * @param displayPath Display channel path.
     * @return Original path, or {@code null} if the specified path does not correspond to any original channel.
     */
    @Override
    public String getOriginPath(String displayPath) {
        return null;
    }

    /**
     * Returns the list of original paths mapped to the specified display channel.
     * <p>
     * The implementation provided by this class returns a list that contains one element - the
     * path returned by {@link #getOriginPath getOriginPath(...)} -  or an
     * empty list if {@link #getOriginPath getOriginPath(...)} returns {@code null}.
     * 
     * @param displayPath Display channel path.
     * @return Original path, or an empty list if the specified path does not correspond to any original channel.
     */
    @Override
    public List<String> getOriginPaths(String displayPath) {
        return super.getOriginPaths(displayPath);
    }

    /**
     * Returns the display channel corresponding to the specified original channel.
     * If the original channel is mapped to multiple display paths, this method may
     * return one of them, or {@code null}.
     * <p>
     * None of the currently known focal plane views call this method directly. However, it is called by
     * {@link #getDisplayPaths(AgentChannel) getDisplayPaths(channel)} implementation unless the latter is overridden.
     * <p>
     * The implementation provided by this class forwards the call to {@link #getDisplayPath(String)} method.
     * 
     * @param channel Original data channel.
     * @return List of display channels.
     */
    @Override
    public String getDisplayPath(AgentChannel channel) {
        return getDisplayPath(channel.getPath());
    }

    /**
     * Returns the display channel corresponding to the specified original path.
     * If the original channel is mapped to multiple display paths, this method may
     * return one of them, or {@code null}.
     * <p>
     * None of the currently known focal plane views call this method directly. However, it is called by
     * {@link #getDisplayPath(AgentChannel) getDisplayPath(channel)} implementation unless the latter is overridden.
     * <p>
     * The implementation provided by this class returns {@code null}.
     * 
     * @param originPath Original path.
     * @return List of display channels.
     */
    @Override
    public String getDisplayPath(String originPath) {
        return null;
    }

    /**
     * Returns a list of display channels for the specified original channel.
     * <p>
     * Note that if {@link #getOriginChannels()} method returns a non-null list and that list does not contain
     * the path of the specified original channel, most views will not display it.
     * <p>
     * The implementation provided by this class returns a list that contains one element - the
     * path returned by {@link #getDisplayPath(AgentChannel) getDisplayPath(channel)} -  or an
     * empty list if {@link #getDisplayPath(AgentChannel) getDisplayPath(channel)} returns {@code null}.
     * 
     * @param channel Original data channel.
     * @return List of display channels.
     */
    @Override
    public List<String> getDisplayPaths(AgentChannel channel) {
        return super.getDisplayPaths(channel);
    }

//    /**
//     * Returns the {@link Persistable} category this filter belongs to.
//     * <p>
//     * The implementation provided by this class returns {@link #CATEGORY}.
//     * 
//     * @return Category of this {@code Persistable}.
//     */
//    @Override
//    public String getCategory() {
//        return CATEGORY;
//    }
    
    
// -- Additional public methods : ----------------------------------------------
    
    /**
     * Returns a list of groups generated by this filter.
     * <p>
     * The implementation provided by this class returns {@code null}.
     * 
     * @return Group names, or {@code null} if this filter does not provide an explicit list of groups.
     */
    public List<String> getGroups() {
        return null;
    }

//    /**
//     * Suggests a list of monitor fields to be used for formatting output for each channel group.
//     * To work well with all focal plane monitoring views, the chosen fields should support formatting multiple channels.
//     * Some monitoring views use either preset or configurable monitor fields, so they will ignore the list returned by this
//     * method, or use it as a starting point only. The length of the list should be either equal to the length of the list
//     * returned by the {@link #getGroups()}, or equal to 1, in which case the field it contains will be used for all groups.
//     * <p>
//     * The implementation provided by this class returns {@code null}.
//     * 
//     * @return Display fields, or {@code null} if this filter does not provide any hints on what fields should be used.
//     */
//    public List<MonitorField> getGroupFields() {
//        return null;
//    }
    
    
// -- Utility methods : --------------------------------------------------------
    
    protected List<String> getDisplayPaths(AgentChannel channel, List<String> groups) {
        String prefix = Segment.getPathPrefix(channel);
        return prefix.isEmpty() ? Collections.emptyList() : groups.stream().map(g -> prefix +"/"+ g).collect(Collectors.toList());
    }
    
    protected String getDisplayPath(AgentChannel channel, String group) {
        String prefix = Segment.getPathPrefix(channel);
        return prefix.isEmpty() ? null : prefix +"/"+ group;
    }
    
// -- Saving/restoring : -------------------------------------------------------
    
    static public class Descriptor extends PersistableAgentChannelsFilter.Descriptor {
        public Descriptor() {
            setCategory(CATEGORY);
        }
        @Override
        public Descriptor clone() {
            return (Descriptor) super.clone();
        }
    }

    @Override
    public Descriptor getDescriptor() {
        return (Descriptor) super.getDescriptor();
    }

    @Override
    public Descriptor save() {
        return (Descriptor) super.save();
    }

}
