package org.lsst.ccs.subsystem.common.ui.jas;

import java.time.Duration;
import org.lsst.ccs.gconsole.base.Console;
import org.lsst.ccs.gconsole.services.command.CommandService;

/**
 * Class for sending a command and receiving the response.
 * 
 * @author saxton
 */
public class CommandSender {

    private final String dest;
    private final CommandService sender = new CommandService();
    private int defTimeout = 5000;

    /**
     * Constructor
     *
     * @param dest The name of the destination subsystem for the commands
     */
    public CommandSender(String dest) {
        this.dest = dest;
    }

    /**
     * Sends a command and receives the response.
     * 
     * The default timeout is used
     * 
     * @param path The path within the destination to receive the command,
     *             or null if the top level
     * @param cmnd The command to execute
     * @param args The command arguments
     * @return The command response
     */
    public Object sendCommand(String path, String cmnd, Object... args) {
        return sendCommand(defTimeout, path, cmnd, args);
    }

    /**
     * Sends a command and receives the response.
     * 
     * @param timeout The command timeout (ms)
     * @param path    The path within the destination to receive the command,
     *                or null if the top level
     * @param cmnd    The command to execute
     * @param args    The command arguments
     * @return The command response
     */
    public Object sendCommand(int timeout, String path, String cmnd, Object... args) {
        try {
            String cmd = dest + (path == null ? "" : "/" + path) + "/" + cmnd;
            return sender.execute(Duration.ofMillis(timeout), cmd, args).get();
        }
        catch (Exception e) {
            Console.getConsole().getLoggerUI().error("Command error: ", e);
            return null;
        }
    }

    /**
     * Sets the default timeout.
     * 
     * @param timeout The default timeout (ms).  If zero, the command waits forever.
     *                If positive, the maximum of this value and the ACK custom timeout is used.
     *                If negative, its absolute value is used.  The initial value is 5000.
     */
    public void setTimeout(int timeout) {
        defTimeout = timeout;
    }

}
