package org.lsst.ccs.subsystem.monitor.ui;

import java.awt.Color;
import java.awt.Component;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.Point;
import java.awt.Window;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.util.ArrayList;
import java.util.BitSet;
import java.util.HashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import javax.swing.BoxLayout;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTabbedPane;
import javax.swing.JTable;
import javax.swing.Popup;
import javax.swing.PopupFactory;
import javax.swing.SwingConstants;
import javax.swing.SwingUtilities;
import javax.swing.border.LineBorder;
import javax.swing.event.TableModelEvent;
import javax.swing.event.TableModelListener;
import javax.swing.table.DefaultTableCellRenderer;
import javax.swing.table.DefaultTableModel;
import javax.swing.table.JTableHeader;
import javax.swing.table.TableCellRenderer;
import javax.swing.table.TableColumn;
import javax.swing.table.TableColumnModel;
import org.freehep.application.studio.Studio;
import org.lsst.ccs.gconsole.plugins.trending.TrendingService;
import org.lsst.ccs.subsystem.monitor.data.MonitorChan;
import org.lsst.ccs.subsystem.monitor.data.MonitorFullState;
import org.lsst.ccs.subsystem.monitor.data.MonitorState;

/*
 * MonitorTrendingTable.java
 *
 * @author turri
 */
public class MonitorTrendingTable extends JPanel {

    static final String[] colNames = {"Description", "Value", "Units",
                                      "Low Limit", "Al.", "High Limit", "Al.",
                                      "Name"};
    static final Class[] colTypes = {String.class, TrendingValue.class,
                                     String.class, Double.class,
                                     AlarmMarker.class, Double.class,
                                     AlarmMarker.class, String.class};
    static final Color colGood = new Color(160, 255, 160),
                       colWarn = new Color(255, 255, 100),
                       colError = new Color(255, 160, 160),
                       colOffln = new Color(160, 200, 255),
                       colPopup = new Color(255, 255, 160);
    static final Font myFont = new Font("Helvetica", Font.PLAIN, 12);
    static final Font changeFont = new Font("Helvetica", Font.BOLD, 12);

    static final int DESCRIPTION_IND = 0;
    static final int VALUE_IND = 1;
    static final int UNITS_IND = 2;
    static final int LOW_LIMIT_IND = 3;
    static final int LOW_ALARM_IND = 4;
    static final int HIGH_LIMIT_IND = 5;
    static final int HIGH_ALARM_IND = 6;
    static final int NAME_IND = 7;

    private final JTabbedPane tabs = new JTabbedPane();

    private final Map<String, SubsysDesc> subsysMap = new HashMap<>();
    private final CommandSender sender;
    private Popup popup;


    /**
     *  Constructor.
     *
     *  @param  sender  Command sender object
     */
    public MonitorTrendingTable(CommandSender sender) {
        this.sender = sender;
        setLayout(new BoxLayout(this, BoxLayout.LINE_AXIS));
    }


    private void tableMouseClicked(MouseEvent evt) {
        if (popup != null) {
            popup.hide();
            popup = null;
        }
        TrendingTable table = (TrendingTable)evt.getSource();
        Point point = evt.getPoint();
        int row = table.rowAtPoint(point);
        if (table.sDesc == null) return;
        if (evt.getClickCount() == 1) {
            int column = table.columnAtPoint(point);
            if (column != LOW_ALARM_IND && column != HIGH_ALARM_IND) return;
            String name = (String)table.getValueAt(row, column);
            if (name == null || name.length() == 0) return;
            PopupFactory fact = PopupFactory.getSharedInstance();
            popup = fact.getPopup(null, table.sDesc.alarmMap.get(name),
                                  evt.getXOnScreen(), evt.getYOnScreen());
            popup.show();
        }
        else if (evt.getClickCount() == 2) {
            String name = (String)table.getValueAt(row, NAME_IND);
            if (name.length() == 0) return;
            String[] path = {table.sDesc.name, name};
            Studio studio = (Studio)Studio.getApplication();
            if (studio == null) return;
            TrendingService trending = (TrendingService)studio.getLookup()
                                         .lookup(TrendingService.class);
            if (trending == null) return;
            trending.show(path);
        }
    }


    public void setSubsystems(String... sNames) {
        for (String sName : sNames) {
            SubsysDesc sDesc = new SubsysDesc();
            sDesc.name = sName;
            subsysMap.put(sName, sDesc);
        }
    }


    public void updateTableModel(String sName, MonitorFullState status) {
        SubsysDesc sDesc = subsysMap.get(sName);
        if (sDesc == null) return;
        SwingUtilities.invokeLater(new UpdateTrendingTableModel(sDesc, status));
    }


    public void updateTableValue(String sName, String cName, double value) {
        SubsysDesc sDesc = subsysMap.get(sName);
        if (sDesc == null) return;
        SwingUtilities.invokeLater(new UpdateTableValue(sDesc, cName, value));
    }


    public void updateTableLimit(String sName, String cName, String limName, String value) {
        SubsysDesc sDesc = subsysMap.get(sName);
        if (sDesc == null) return;
        SwingUtilities.invokeLater(new UpdateLimitValue(sDesc, cName, limName, value));
    }


    public void updateTableState(String sName, MonitorState status) {
        SubsysDesc sDesc = subsysMap.get(sName);
        if (sDesc == null) return;
        SwingUtilities.invokeLater(new UpdateState(sDesc, status));
    }


    public void disableSystem(String sName) {
        SubsysDesc sDesc = subsysMap.get(sName);
        if (sDesc == null) return;
        SwingUtilities.invokeLater(new DisableSystem(sDesc));
    }


    void updateState(SubsysDesc sDesc, MonitorState s) {
        BitSet updateValue = (BitSet)s.getGoodChans().clone();
        updateValue.xor(sDesc.goodChans);
        sDesc.goodChans.xor(updateValue);
        BitSet updateOnline = (BitSet)s.getOnlineChans().clone();
        updateOnline.xor(sDesc.onlineChans);
        sDesc.onlineChans.xor(updateOnline);
        if (!sDesc.channels.isEmpty()) {
            updateValue.or(updateOnline);
            for (int sRow = updateValue.nextSetBit(0); sRow >= 0;
                 sRow = updateValue.nextSetBit(sRow + 1)) {
                ChanDesc cDesc = sDesc.channels.get(sRow);
                if (cDesc != null) {
                    TrendingTable table = cDesc.table;
                    table.setValueAt(table.getValueAt(cDesc.tRow, VALUE_IND), cDesc.tRow, VALUE_IND);
                }
            }
        }
        BitSet updateHigh = (BitSet)s.getHighLimitChange().clone();
        updateHigh.xor(sDesc.highLimitChange);
        sDesc.highLimitChange.xor(updateHigh);
        for (int sRow = updateHigh.nextSetBit(0); sRow >= 0;
             sRow = updateHigh.nextSetBit(sRow + 1)) {
            ChanDesc cDesc = sDesc.channels.get(sRow);
            if (cDesc != null) {
                cDesc.table.setValueAt(cDesc.highLimit, cDesc.tRow, HIGH_LIMIT_IND);
            }
        }
        BitSet updateLow = (BitSet)s.getLowLimitChange().clone();
        updateLow.xor(sDesc.lowLimitChange);
        sDesc.lowLimitChange.xor(updateLow);
        for (int sRow = updateLow.nextSetBit(0); sRow >= 0;
             sRow = updateLow.nextSetBit(sRow + 1)) {
            ChanDesc cDesc = sDesc.channels.get(sRow);
            if (cDesc != null) {
                cDesc.table.setValueAt(cDesc.lowLimit, cDesc.tRow, LOW_LIMIT_IND);
            }
        }
    }


    void addTrendingRow(SubsysDesc sDesc, MonitorChan chan) {
        TrendingTable table = sDesc.pageMap.get(chan.getPage());
        TrendingTableModel model = table.model();
        String[] descrip = chan.getDescription().split("\\\\", 2);
        if (descrip.length == 2) {
            if (descrip[0].length() == 0) {
                table.indent = false;
            }
            else {
                model.addRow(new Object[]{descrip[0], null, "", null, "", null, "", ""});
                table.tsRowMap.add(-1);
                table.indent = true;
            }
            descrip[0] = descrip[1];
        }
        ChanDesc cDesc = new ChanDesc();
        cDesc.sRow = sDesc.channels.size();
        cDesc.table = table;
        cDesc.tRow = table.tsRowMap.size();
        cDesc.lowLimit = chan.getLowLimit();
        cDesc.highLimit = chan.getHighLimit();
        cDesc.format = chan.getFormat();
        sDesc.chanMap.put(chan.getName(), cDesc);
        table.tsRowMap.add(sDesc.channels.size());
        sDesc.channels.add(cDesc);
        model.addRow(new Object[]{(table.indent ? "   " : "") + descrip[0],
                                  chan.getValue(), chan.getUnits(),
                                  chan.getLowLimit(), chan.getLowAlarm(),
                                  chan.getHighLimit(), chan.getHighAlarm(),
                                  chan.getName()});
    }


    private static String fmt(String format, double value) {
        return String.format(format, value);
    }


   /*
    *  Trending table class
    */
    class TrendingTable extends JTable {

        SubsysDesc     sDesc;      // Subsystem descriptor
        boolean        indent;     // Whether description is indented
        List<Integer>  tsRowMap = new ArrayList<>(); // Mapping of table to system rows
        JScrollPane    pane;       // Containing scroll pane

        void initialize(SubsysDesc sDesc, String name) {
            this.sDesc = sDesc;
            TrendingTableModel model = new TrendingTableModel(this);
            setModel(model);
            setName(name);
            addMouseListener(new MouseAdapter() {
                @Override
                public void mouseClicked(MouseEvent evt) {
                    tableMouseClicked(evt);
                }
            });
            pane = new JScrollPane();
            pane.setViewportView(this);
            pane.setName(name);
            tabs.add(pane);
            JTableHeader hdr = getTableHeader();
            hdr.setReorderingAllowed(false);
            hdr.setSize(hdr.getWidth(), hdr.getHeight() + 2);
            setDefaultRenderer(TrendingValue.class, new TrendingTableCellRenderer());
            setDefaultRenderer(Double.class, new LimitsCellRenderer());
            setDefaultRenderer(String.class, new TextCellRenderer());
            setDefaultRenderer(AlarmMarker.class, new AlarmCellRenderer());
            setAutoResizeMode(JTable.AUTO_RESIZE_LAST_COLUMN);
            setRowSelectionAllowed(false);
            setColumnSelectionAllowed(false);
            setFont(myFont);
            setRowHeight(getRowHeight() + 2);
        }

        public TrendingTableModel model() {
            return (TrendingTableModel)super.getModel();
        }

        String getFormat(int tRow) {
            int sRow = tsRowMap.get(tRow);
            return sRow < 0 ? "" : sDesc.channels.get(sRow).format;
        }

        Color getChannelColor(int tRow) {
            int sRow = tsRowMap.get(tRow);
            return sRow < 0 ? Color.WHITE : !sDesc.enabled ? Color.LIGHT_GRAY
                            : !sDesc.onlineChans.get(sRow) ? colOffln 
                            : sDesc.goodChans.get(sRow) ? colGood : colError;
        }

        boolean hasLowLimitChanged(int tRow) {
            int sRow = tsRowMap.get(tRow);
            return sRow >= 0 ? sDesc.lowLimitChange.get(sRow) : false;
        }

        boolean hasHighLimitChanged(int tRow) {
            int sRow = tsRowMap.get(tRow);
            return sRow >= 0 ? sDesc.highLimitChange.get(sRow) : false;
        }

        boolean isNewSection(int tRow) {
            return tsRowMap.get(tRow) < 0;
        }

        private static final long serialVersionUID = 1L;
    }


   /*
    *  Trending table model class
    */
    class TrendingTableModel extends DefaultTableModel implements TableModelListener {

        TrendingTable table;   // Associated table

        public TrendingTableModel(TrendingTable table) {
            super(colNames, 0);
            addTableModelListener(this);
            this.table = table;
        }

        @Override
        public Class getColumnClass(int column) {
            return colTypes[column];
        }

        @Override
        public boolean isCellEditable(int row, int column) {
            return (column == LOW_LIMIT_IND || column == HIGH_LIMIT_IND)
                     && getValueAt(row, column) != null;
        }

        @Override
        public void tableChanged(TableModelEvent e) {
            int column = e.getColumn();
            if (column != LOW_LIMIT_IND && column != HIGH_LIMIT_IND) return;
            boolean isLow = column == LOW_LIMIT_IND;
            int tRow = e.getFirstRow();
            int sRow = table.tsRowMap.get(tRow);
            double newValue = (Double)getValueAt(tRow, column);
            double oldValue = isLow ? table.sDesc.channels.get(sRow).lowLimit
                                    : table.sDesc.channels.get(sRow).highLimit;
            if (newValue != oldValue) {
                setValueAt(oldValue, tRow, column);    // Put it back for now
                sender.sendCommand(table.sDesc.name,
                                   (String)getValueAt(tRow, NAME_IND), "change",
                                   isLow ? "limitLo" : "limitHi", newValue);
            }
        }

        private static final long serialVersionUID = 1L;
    }


    static class SubsysDesc {
        String name;                                            // Subsystem name
        boolean enabled;                                        // Whether subsystem is enabled
        Map<String, JLabel> alarmMap = new HashMap<>();         // Map of alarm names to pop-ups
        Map<String, ChanDesc> chanMap = new HashMap<>();        // Map of channel names to descriptors
        Map<Integer, TrendingTable> pageMap = new HashMap<>();  // Map of page IDs to tables
        List<TrendingTable> tables = new ArrayList<>();         // List of tables present
        List<ChanDesc> channels = new ArrayList<>();            // Ordered list of channel descriptors
        BitSet goodChans = new BitSet();                        // Ordered set of good channels
        BitSet onlineChans = new BitSet();                      // Ordered set of online channels
        BitSet lowLimitChange = new BitSet();                   // Ordered set of channels with changed low limit
        BitSet highLimitChange = new BitSet();                  // Ordered set of channels with changed high limit
    }

    static class ChanDesc {
        int            sRow;       // Row number within subsystem
        TrendingTable  table;      // Associated table
        int            tRow;       // Row number within table
        double         lowLimit;   // Low limit
        double         highLimit;  // High limit
        String         format;     // Format string
    }

    static class TrendingValue {
    }

    static class AlarmMarker {
    }

    class UpdateTableValue implements Runnable {

        SubsysDesc sDesc;
        String cName;
        double value;

        UpdateTableValue(SubsysDesc sDesc, String cName, double value) {
            this.sDesc = sDesc;
            this.cName = cName;
            this.value = value;
        }

        @Override
        public void run() {
            ChanDesc cDesc = sDesc.chanMap.get(cName);
            if (cDesc != null) {
                cDesc.table.setValueAt(value, cDesc.tRow, VALUE_IND);
            }
        }
    }

    class UpdateLimitValue implements Runnable {

        SubsysDesc sDesc;
        String cName, limName, value;

        UpdateLimitValue(SubsysDesc sDesc, String cName, String limName, String value) {
            this.sDesc = sDesc;
            this.cName = cName;
            this.limName = limName;
            this.value = value;
        }

        @Override
        public void run() {
            ChanDesc cDesc = sDesc.chanMap.get(cName);
            if (cDesc == null) return;
            double limit = Double.valueOf(value);
            if (limName.equals("alarmHigh")) {
                cDesc.highLimit = limit;
                cDesc.table.setValueAt(limit, cDesc.tRow, HIGH_LIMIT_IND);
            }
            else {
                cDesc.lowLimit = limit;
                cDesc.table.setValueAt(limit, cDesc.tRow, LOW_LIMIT_IND);
            }
        }

    }

    class UpdateState implements Runnable {

        SubsysDesc sDesc;
        MonitorState s;

        UpdateState(SubsysDesc sDesc, MonitorState s) {
            this.sDesc = sDesc;
            this.s = s;
        }

        @Override
        public void run() {
            updateState(sDesc, s);
        }

    }

    class UpdateTrendingTableModel implements Runnable {

        SubsysDesc sDesc;
        MonitorFullState s;

        UpdateTrendingTableModel(SubsysDesc sDesc, MonitorFullState s) {
            this.sDesc = sDesc;
            this.s = s;
        }

        @Override
        public void run() {

            for (TrendingTable table : sDesc.tables) {
                TrendingTableModel model = table.model();
                while (model.getRowCount() > 0) {
                    model.removeRow(0);
                }
                tabs.remove(table.pane);
            }
            sDesc.enabled = true;
            sDesc.tables.clear();
            sDesc.alarmMap.clear();
            sDesc.pageMap.clear();
            sDesc.chanMap.clear();
            sDesc.channels.clear();
            sDesc.goodChans.clear();
            sDesc.onlineChans.clear();
            sDesc.lowLimitChange.clear();
            sDesc.highLimitChange.clear();

            Set<Integer> pages = new LinkedHashSet<>();
            pages.addAll(s.getPages().keySet());
            for (MonitorChan chan : s.getChannels()) {
                pages.add(chan.getPage());
            }
            for (int page : pages) {
                String pageName = s.getPages().get(page);
                pageName = pageName != null ? pageName : "Page " + page;
                TrendingTable table = new TrendingTable();
                table.initialize(sDesc, pageName);
                sDesc.tables.add(table);
                sDesc.pageMap.put(page, table);
            }

            Map<String, String> alarms = s.getAlarms();
            for (String name : alarms.keySet()) {
                JLabel label = new JLabel(alarms.get(name));
                label.setBorder(LineBorder.createBlackLineBorder());
                label.setBackground(colPopup);
                label.setFont(myFont);
                sDesc.alarmMap.put(name, label);
            }

            for (MonitorChan chan : s.getChannels()) {
                addTrendingRow(sDesc, chan);
            }

            for (TrendingTable table : sDesc.tables) {
                if (table.getRowCount() == 0) {
                    tabs.remove(table.pane);
                    continue;
                }
                for (int c = 0; c < table.getColumnCount(); c++) {
                    TableColumnModel colModel = table.getColumnModel();
                    TableColumn col = colModel.getColumn(c);
                    TableCellRenderer rndr;
                    Component comp;

                    rndr = table.getTableHeader().getDefaultRenderer();
                    comp = rndr.getTableCellRendererComponent(table, col.getHeaderValue(),
                                                              false, false, 0, 0);
                    int width = comp.getPreferredSize().width;

                    rndr = table.getCellRenderer(0, c);
                    Class colClass = table.getColumnClass(c);
                    if (colClass.equals(String.class)
                        || colClass.equals(AlarmMarker.class)) {
                        for (int r = 0; r < table.model().getRowCount(); r++) {
                            Object value = table.getValueAt(r, c);
                            comp = rndr.getTableCellRendererComponent(table, value, false, false, r, c);
                            width = Math.max(width, comp.getPreferredSize().width);
                        }
                    }
                    else {
                        for (int r = 0; r < table.model().getRowCount(); r++) {
                            comp = rndr.getTableCellRendererComponent(table, -999.99, false, false, r, c);
                            width = Math.max(width, comp.getPreferredSize().width);
                        }
                    }
                    col.setPreferredWidth(width + 4);
                    col.setMinWidth(width + 4);
                }

                updateState(sDesc, s.getMonitorState());

                Container anc = getTopLevelAncestor();
                if (anc instanceof Window) {
                    Dimension td = table.pane.getPreferredSize(), wd = anc.getSize();
                    wd.width = Math.max(wd.width, td.width);
                    wd.height = Math.max(wd.height, td.height);
                    anc.setSize(wd);
                }
            }

            removeAll();
            if (tabs.getTabCount() == 1) {
                add(tabs.getComponentAt(0));
            }
            else {
                add(tabs);
            }
        }

    }

    class DisableSystem implements Runnable {
        
        SubsysDesc sDesc;

        DisableSystem(SubsysDesc sDesc) {
            this.sDesc = sDesc;
        }

        @Override
        public void run() {
            sDesc.enabled = false;
            for (TrendingTable table : sDesc.tables) {
                for (int row = 0; row < table.model().getRowCount(); row++) {
                    table.setValueAt(table.getValueAt(row, VALUE_IND), row, VALUE_IND);
                }
            }
        }
    }

    class LimitsCellRenderer extends DefaultTableCellRenderer {

        @Override
        public Component getTableCellRendererComponent(JTable table, Object value, boolean isSelected,
                                                       boolean hasFocus, int row, int column) {
            if (value == null) {
                return super.getTableCellRendererComponent(table, "", false, false, row, column);
            }
            TrendingTable tTable = (TrendingTable)table;
            String text = fmt(tTable.getFormat(row), (Double)value);
            Component c = super.getTableCellRendererComponent(table, text, false, hasFocus, row, column);
            if (column == LOW_LIMIT_IND && tTable.hasLowLimitChanged(row)
                  || column == HIGH_LIMIT_IND && tTable.hasHighLimitChanged(row)) {
                c.setFont(changeFont);
                c.setForeground(Color.blue);
            }
            else {
                c.setForeground(Color.black);
            }
            ((JLabel)c).setHorizontalAlignment(SwingConstants.RIGHT);

            return c;
        }

        private static final long serialVersionUID = 1L;
    }

    class TrendingTableCellRenderer extends DefaultTableCellRenderer {

        @Override
        public Component getTableCellRendererComponent(JTable table, Object value, boolean isSelected,
                                                       boolean hasFocus, int row, int column) {
            Component c;
            if (value == null) {
                c = super.getTableCellRendererComponent(table, "", false, false, row, column);
                c.setBackground(Color.WHITE);
            }
            else {
                TrendingTable tTable = (TrendingTable)table;
                String text = fmt(tTable.getFormat(row), (Double)value);
                c = super.getTableCellRendererComponent(table, text, false, false, row, column);
                c.setBackground(tTable.getChannelColor(row));
                ((JLabel)c).setHorizontalAlignment(SwingConstants.RIGHT);
            }

            return c;
        }

        private static final long serialVersionUID = 1L;
}

    class AlarmCellRenderer extends DefaultTableCellRenderer {

        @Override
        public Component getTableCellRendererComponent(JTable table, Object value, boolean isSelected,
                                                       boolean hasFocus, int row, int column) {
            String text = (String)value;
            text = text == null || text.length() == 0 ? "" : "  \u2713";
            return super.getTableCellRendererComponent(table, text, false, false, row, column);
        }

        private static final long serialVersionUID = 1L;
    }

    class TextCellRenderer extends DefaultTableCellRenderer {

        @Override
        public Component getTableCellRendererComponent(JTable table, Object value, boolean isSelected,
                                                       boolean hasFocus, int row, int column) {
            Component c = super.getTableCellRendererComponent(table, "  " + (String)value,
                                                              false, false, row, column);
            TrendingTable tTable = (TrendingTable)table;
            if (column == DESCRIPTION_IND && tTable.isNewSection(row)) {
                c.setFont(changeFont);
            }

            return c;
        }

        private static final long serialVersionUID = 1L;
    }

    private static final long serialVersionUID = 1L;
}
