package org.lsst.ccs.subsystem.monitor.ui;

import java.io.PrintWriter;
import java.util.HashMap;
import java.util.Map;
import java.util.Timer;
import java.util.TimerTask;
import javax.swing.JComponent;
import javax.swing.JFrame;
import javax.swing.JPanel;
import org.lsst.ccs.Agent;
import org.lsst.ccs.ConcurrentMessagingUtils;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.messages.CommandRequest;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.bus.messages.StatusSubsystemData;
import org.lsst.ccs.bus.states.PhaseState;
import org.lsst.ccs.bus.states.StateBundle;
import org.lsst.ccs.messaging.AgentMessagingLayer;
import org.lsst.ccs.messaging.AgentPresenceListener;
import org.lsst.ccs.messaging.AgentPresenceManager;
import org.lsst.ccs.messaging.StatusMessageListener;
import org.lsst.ccs.utilities.logging.Logger;

/**
 ******************************************************************************
 **
 **  General monitoring (and control) GUI.
 **
 **  @author Owen Saxton
 **
 ******************************************************************************
 */
public class MonitorGui implements CommandSender, StatusMessageListener,
                                   AgentPresenceListener {

    static class Subsys {

        final String name, initCmd;
        final int id;
        long time;
        boolean enabled;

        Subsys(String name, int id, String initCmd) {
            this.name = name;
            this.id = id;
            this.initCmd = initCmd;
            time = 0;
            enabled = false;
        }
    }

    public static class Panel extends JPanel {

        protected void enableSystem(String name, int id) {
        }

        protected void disableSystem(String name, int id) {
        }

        protected void updatePanel(String name, int id, Object d) {
        }

        protected void handleStatusData(String name, int id, long time,
                                        String key, Object d) {
        }

        protected void handleStatusMessage(String name, int id,
                                           StatusMessage s) {
        }

        private static final long serialVersionUID = 1L;

    }

    protected Panel panel;
    private final static long CHECK_TIME = 5000;
    private PrintWriter out = new PrintWriter(System.out, true);
    private final Map<String, Subsys> subsysMap = new HashMap<>();
    private final Logger log = Logger.getLogger("org.lsst.ccs.monitor");
    private ConcurrentMessagingUtils cmu;
    private AgentPresenceManager apm;

    public void addSubsys(String name, int id, String initCmd) {
        subsysMap.put(name, new Subsys(name, id, initCmd));
    }

    public void initGui() {
        AgentMessagingLayer aml = Agent.getEnvironmentMessagingAccess();
        cmu = new ConcurrentMessagingUtils(aml);
        aml.addStatusMessageListener(this);
        apm = aml.getAgentPresenceManager();
        apm.addAgentPresenceListener(this);

        Timer t = new Timer();
        t.scheduleAtFixedRate(new TimerTask() {
            @Override
            public void run() {
                for (Subsys ss : subsysMap.values()) {
                    if ((System.currentTimeMillis() - ss.time) / 1000 > 10) {
                        disable(ss);
                    }
                }
            }
        }, CHECK_TIME, CHECK_TIME);
    }

    public JComponent getGuiLayout() {
        return panel;
    }

    public Logger getLogger() {
        return log;
    }

    public void resetGui() {
    }

    public void setPrintWriter(PrintWriter out) {
        this.out = out;
    }

    public void startGui(String name, String title) {
        (new Agent(name, AgentInfo.AgentType.CONSOLE)).start();
        initGui();
        final JFrame f = new JFrame(title);
        f.setContentPane(getGuiLayout());
        f.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
        f.pack();
        f.setVisible(true);
    }

    @Override
    public void onStatusMessage(StatusMessage s) {
        Subsys ss = subsysMap.get(s.getOriginAgentInfo().getName());
        if (ss == null) return;
        ss.time = s.getTimeStamp();
        if (!ss.enabled) {
            StateBundle sb = s.getState();
            if (sb != null && sb.isInState(PhaseState.OPERATIONAL)
                  && apm.agentExists(ss.name)) {
                if (ss.initCmd == null || ss.initCmd.isEmpty()) {
                    ss.enabled = true;
                    panel.enableSystem(ss.name, ss.id);
                }
                else {
                    Object resp = sendCommand(ss.name, null, ss.initCmd);
                    if (resp != null) {
                        ss.enabled = true;
                        panel.updatePanel(ss.name, ss.id, resp);
                    }
                }
            }
        }
        if (s instanceof StatusSubsystemData) {
            StatusSubsystemData sd = (StatusSubsystemData)s;
            panel.handleStatusData(ss.name, ss.id, sd.getTimeStamp(),
                                   sd.getDataKey(), sd.getSubsystemData());
        }
    }

    @Override
    public void connecting(AgentInfo agent) {
    }

    @Override
    public void disconnecting(AgentInfo agent) {
        disable(subsysMap.get(agent.getName()));
    }

    @Override
    public Object sendCommand(String dest, String target, String cmnd,
                              Object... args) {
        try {
            return sendCommandRaw(dest, target, cmnd, args);
        }
        catch (Exception e) {
            log.error("Command error: ", e);
            return null;
        }
    }

    @Override
    public Object sendCommandRaw(String dest, String target, String cmnd,
                                 Object... args) throws Exception {
        String dst = dest + (target == null ? "" : "/" + target);
        CommandRequest cmd = new CommandRequest(dst, cmnd, args);
        return cmu.sendSynchronousCommand(cmd, 5000);
    }

    private void disable(Subsys ss) {
        if (ss == null || !ss.enabled) return;
        ss.enabled = false;
        panel.disableSystem(ss.name, ss.id);
    }

}
