package org.lsst.ccs.subsystem.monitor.ui;

import java.awt.Color;
import java.awt.Component;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.Point;
import java.awt.Window;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.util.ArrayList;
import java.util.BitSet;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import javax.swing.BoxLayout;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTable;
import javax.swing.Popup;
import javax.swing.PopupFactory;
import javax.swing.JScrollPane;
import javax.swing.SwingConstants;
import javax.swing.SwingUtilities;
import javax.swing.border.LineBorder;
import javax.swing.event.TableModelEvent;
import javax.swing.event.TableModelListener;
import javax.swing.table.DefaultTableCellRenderer;
import javax.swing.table.DefaultTableModel;
import javax.swing.table.JTableHeader;
import javax.swing.table.TableCellRenderer;
import javax.swing.table.TableColumn;
import javax.swing.table.TableColumnModel;
import org.freehep.application.studio.Studio;
import org.lsst.ccs.bus.MetadataStatus;
import org.lsst.ccs.plugin.jas3.trending.TrendingService;
import org.lsst.ccs.subsystem.monitor.data.MonitorChan;
import org.lsst.ccs.subsystem.monitor.data.MonitorFullState;
import org.lsst.ccs.subsystem.monitor.data.MonitorState;

/*
 * MonitorTrendingTable.java
 *
 * @author turri
 */
public class MonitorTrendingTable extends JPanel {

    final static String[] colNames = {"Description", "Value", "Units",
                                      "Low Limit", "Al.", "High Limit", "Al.",
                                      "Name"};
    final static Class[] colTypes = {String.class, TrendingValue.class,
                                     String.class, Double.class,
                                     AlarmMarker.class, Double.class,
                                     AlarmMarker.class, String.class};
    final static Color colGood = new Color(160, 255, 160),
                       colError = new Color(255, 160, 160),
                       colOffln = new Color(160, 200, 255),
                       colPopup = new Color(255, 255, 160);
    final static Font myFont = new Font("Helvetica", Font.PLAIN, 12);
    final static Font changeFont = new Font("Helvetica", Font.BOLD, 12);

    final static int DESCRIPTION_IND = 0;
    final static int VALUE_IND = 1;
    final static int UNITS_IND = 2;
    final static int LOW_LIMIT_IND = 3;
    final static int LOW_ALARM_IND = 4;
    final static int HIGH_LIMIT_IND = 5;
    final static int HIGH_ALARM_IND = 6;
    final static int NAME_IND = 7;

    private JScrollPane pane;
    private JTable table;

    Map<String, SubsysDesc> subsysMap = new HashMap<>();
    CommandSender sender;
    int lastFocusRow;
    Popup popup;


    /* Creates new form MonitorTrendingTable */
    public MonitorTrendingTable(CommandSender sender) {
        initComponents();
        JTableHeader hdr = table.getTableHeader();
        hdr.setReorderingAllowed(false);
        hdr.setSize(hdr.getWidth(), hdr.getHeight() + 2);
        table.setDefaultRenderer(TrendingValue.class,
                                 new TrendingTableCellRenderer());
        table.setDefaultRenderer(Double.class, new LimitsCellRenderer());
        table.setDefaultRenderer(String.class, new TextCellRenderer());
        table.setDefaultRenderer(AlarmMarker.class, new AlarmCellRenderer());
        table.setAutoResizeMode(JTable.AUTO_RESIZE_LAST_COLUMN);
        table.setRowSelectionAllowed(false);
        table.setColumnSelectionAllowed(false);
        table.setFont(myFont);
        table.setRowHeight(table.getRowHeight() + 2);

        this.sender = sender;
    }

    private void initComponents() {

        pane = new JScrollPane();
        table = new JTable();

        setLayout(new BoxLayout(this, BoxLayout.LINE_AXIS));

        table.setModel(new TrendingTableModel());
        table.setName("monTable");
        table.addMouseListener(new MouseAdapter() {
            @Override
            public void mouseClicked(MouseEvent evt) {
                tableMouseClicked(evt);
            }
        });
        pane.setViewportView(table);

        add(pane);
    }

    private void tableMouseClicked(MouseEvent evt) {
        if (popup != null) {
            popup.hide();
            popup = null;
        }
        JTable source = (JTable)evt.getSource();
        Point point = evt.getPoint();
        int row = source.rowAtPoint(point);
        SubsysDesc desc = findSubsys(row);
        if (desc == null) return;
        if (evt.getClickCount() == 1) {
            int column = source.columnAtPoint(point);
            if (column != LOW_ALARM_IND && column != HIGH_ALARM_IND) return;
            String name = (String)source.getModel().getValueAt(row, column);
            if (name == null || name.length() == 0) return;
            PopupFactory fact = PopupFactory.getSharedInstance();
            popup = fact.getPopup(null, desc.alarmMap.get(name),
                                  evt.getXOnScreen(), evt.getYOnScreen());
            popup.show();
        }
        else if (evt.getClickCount() == 2) {
            String name = (String)source.getModel().getValueAt(row, NAME_IND);
            if (name.length() == 0) return;
            String[] path = {desc.name, name};
            Studio studio = (Studio)Studio.getApplication();
            if (studio == null) return;
            TrendingService trending = (TrendingService)studio.getLookup()
                                         .lookup(TrendingService.class);
            if (trending == null) return;
            trending.show(path);
        }
    }

    public void setSubsystems(String... sNames) {
        for (String sName : sNames) {
            SubsysDesc desc = new SubsysDesc();
            desc.name = sName;
            subsysMap.put(sName, desc);
        }
    }

    public void updateTableModel(String sName, MonitorFullState status) {
        SubsysDesc desc = subsysMap.get(sName);
        if (desc == null) return;
        SwingUtilities.invokeLater(new UpdateTrendingTableModel(desc, status));
    }

    public void updateTableValue(String sName, String iName, double value) {
        SubsysDesc desc = subsysMap.get(sName);
        if (desc == null) return;
        SwingUtilities.invokeLater(new UpdateTableValue(desc, iName, value));
    }

    public void updateTableLimit(MetadataStatus status) {
        SubsysDesc desc = subsysMap.get(status.getOrigin());
        if (desc == null) return;
        SwingUtilities.invokeLater(new UpdateLimitValue(desc, status));
    }

    public void updateTableState(String sName, MonitorState status) {
        SubsysDesc desc = subsysMap.get(sName);
        if (desc == null) return;
        SwingUtilities.invokeLater(new UpdateState(desc, status));
    }

    public void disableSystem(String sName) {
        SubsysDesc desc = subsysMap.get(sName);
        if (desc == null) return;
        SwingUtilities.invokeLater(new DisableSystem(desc));
    }

    SubsysDesc findSubsys(int row) {
        for (SubsysDesc desc : subsysMap.values()) {
            if (row >= desc.start && row < desc.end) return desc;
        }
        return null;
    }

    Color getChannelColor(int row) {
        SubsysDesc desc = findSubsys(row);
        int sRow = desc.tsRowMap.get(row - desc.start);
        return sRow < 0 ? Color.WHITE : !desc.enabled ? Color.LIGHT_GRAY
                 : !desc.onlineChans.get(sRow) ? colOffln 
                 : desc.goodChans.get(sRow) ? colGood : colError;
    }

    boolean hasLowLimitChanged(int row) {
        SubsysDesc desc = findSubsys(row);
        int sRow = desc.tsRowMap.get(row - desc.start);
        return sRow >= 0 ? desc.lowLimitChange.get(sRow) : false;
    }
        
    boolean hasHighLimitChanged(int row) {
        SubsysDesc desc = findSubsys(row);
        int sRow = desc.tsRowMap.get(row - desc.start);
        return sRow >= 0 ? desc.highLimitChange.get(sRow) : false;
    }

    boolean isNewSection(int row) {
        SubsysDesc desc = findSubsys(row);
        return desc.tsRowMap.get(row - desc.start) < 0;
    }

    private static String fmt(double val) {
        return String.format("%.2f  ", val);
    }

    class TrendingTableModel extends DefaultTableModel
                             implements TableModelListener {

        public TrendingTableModel() {
            super(colNames, 0);
            addTableModelListener(this);
        }

        @Override
        public Class getColumnClass(int column) {
            return colTypes[column];
        }

        @Override
        public boolean isCellEditable(int row, int column) {
            if (column == LOW_LIMIT_IND || column == HIGH_LIMIT_IND) {
                return !(getValueAt(row, column) == null);
            }
            return false;
        }

        @Override
        public void tableChanged(TableModelEvent e) {
            int column = e.getColumn();
            if (column != LOW_LIMIT_IND && column != HIGH_LIMIT_IND) return;
            boolean isLow = column == LOW_LIMIT_IND;
            int row = e.getFirstRow();
            SubsysDesc desc = findSubsys(row);
            int sRow = desc.tsRowMap.get(row - desc.start);
            double newValue = (Double)getValueAt(row, column);
            double oldValue = isLow ? desc.lowLimits.get(sRow)
                : desc.highLimits.get(sRow);

            if (newValue != oldValue) {
                setValueAt(oldValue, row, column);    // Put it back for now
                sender.sendCommand(desc.name,
                                   (String)getValueAt(row, NAME_IND), "change",
                                   isLow ? "limitLo" : "limitHi", newValue);
            }
        }

        void addTrendingRow(SubsysDesc desc, MonitorChan chan) {
            String[] descrip = chan.getDescription().split("\\\\", 2);
            if (descrip.length == 2) {
                if (descrip[0].length() == 0) {
                    desc.indent = false;
                }
                else {
                    insertRow(desc.end++,
                              new Object[]{descrip[0], null, "", null, "",
                                           null, "", ""});
                    desc.tsRowMap.add(-1);
                    desc.indent = true;
                }
                descrip[0] = descrip[1];
            }
            desc.chanMap.put(chan.getName(), desc.tsRowMap.size());
            desc.stRowMap.add(desc.tsRowMap.size());
            desc.tsRowMap.add(desc.lowLimits.size());
            desc.lowLimits.add(chan.getLowLimit());
            desc.highLimits.add(chan.getHighLimit());
            insertRow(desc.end++,
                      new Object[]{(desc.indent ? "   " : "") + descrip[0],
                                   chan.getValue(), chan.getUnits(),
                                   chan.getLowLimit(), chan.getLowAlarm(),
                                   chan.getHighLimit(), chan.getHighAlarm(),
                                   chan.getName()});
        }

        void updateValue(SubsysDesc desc, String name, double value) {
            Integer tRow = desc.chanMap.get(name);
            if (tRow == null) return;
            setValueAt(value, desc.start + tRow, VALUE_IND);
        }

        void updateLimitValue(SubsysDesc desc, MetadataStatus s) {
            boolean high = s.getMetadataName().equals("alarmHigh");
            double limit = Double.valueOf(s.getMetadataValue());
            Integer column, tRow = desc.chanMap.get(s.getDataName());
            if (tRow == null) return;
            if (high) {
                desc.highLimits.set(desc.tsRowMap.get(tRow), limit);
                column = HIGH_LIMIT_IND;
            }
            else {
                desc.lowLimits.set(desc.tsRowMap.get(tRow), limit);
                column = LOW_LIMIT_IND;
            }
            setValueAt(limit, desc.start + tRow, column);
        }

        void updateState(SubsysDesc desc, MonitorState s) {
            desc.goodChans = s.getGoodChans();
            desc.onlineChans = s.getOnlineChans();
            BitSet updateHigh = (BitSet)s.getHighLimitChange().clone();
            updateHigh.xor(desc.highLimitChange);
            desc.highLimitChange.xor(updateHigh);
            for (int sRow = updateHigh.nextSetBit(0); sRow >= 0;
                 sRow = updateHigh.nextSetBit(sRow + 1)) {
                setValueAt(desc.highLimits.get(sRow),
                           desc.start + desc.stRowMap.get(sRow),
                           HIGH_LIMIT_IND);
            }
            BitSet updateLow = (BitSet)s.getLowLimitChange().clone();
            updateLow.xor(desc.lowLimitChange);
            desc.lowLimitChange.xor(updateLow);
            for (int sRow = updateLow.nextSetBit(0); sRow >= 0;
                 sRow = updateLow.nextSetBit(sRow + 1)) {
                setValueAt(desc.lowLimits.get(sRow),
                           desc.start + desc.stRowMap.get(sRow),
                           LOW_LIMIT_IND);
            }
        }

        private static final long serialVersionUID = 1L;
    }

    class SubsysDesc {

        String name;
        int start, size, end;
        boolean indent, enabled;
        Map<String, JLabel> alarmMap = new HashMap<>();
        Map<String, Integer> chanMap = new HashMap<>();
        List<Integer> tsRowMap = new ArrayList<>(), stRowMap = new ArrayList<>();
        BitSet goodChans = new BitSet(), onlineChans = new BitSet(),
               lowLimitChange = new BitSet(), highLimitChange = new BitSet();
        List<Double> lowLimits = new ArrayList<>(),
                     highLimits = new ArrayList<>();

    }

    class TrendingValue {
    }

    class AlarmMarker {
    }

    class UpdateTableValue implements Runnable {

        SubsysDesc desc;
        String n;
        double v;

        UpdateTableValue(SubsysDesc desc, String n, double v) {
            this.desc = desc;
            this.n = n;
            this.v = v;
        }

        @Override
        public void run() {
            ((TrendingTableModel)table.getModel()).updateValue(desc, n, v);
        }
    }

    class UpdateLimitValue implements Runnable {

        SubsysDesc desc;
        MetadataStatus s;

        UpdateLimitValue(SubsysDesc desc, MetadataStatus s) {
            this.desc = desc;
            this.s = s;
        }

        @Override
        public void run() {
            ((TrendingTableModel)table.getModel()).updateLimitValue(desc, s);
        }

    }

    class UpdateState implements Runnable {

        SubsysDesc desc;
        MonitorState s;

        UpdateState(SubsysDesc desc, MonitorState s) {
            this.desc = desc;
            this.s = s;
        }

        @Override
        public void run() {
            ((TrendingTableModel)table.getModel()).updateState(desc, s);
        }

    }

    class UpdateTrendingTableModel implements Runnable {

        SubsysDesc desc;
        MonitorFullState s;

        UpdateTrendingTableModel(SubsysDesc desc, MonitorFullState s) {
            this.desc = desc;
            this.s = s;
        }

        @Override
        public void run() {

            desc.indent = false;
            desc.enabled = true;
            desc.alarmMap.clear();
            desc.chanMap.clear();
            desc.tsRowMap.clear();
            desc.stRowMap.clear();
            desc.goodChans.clear();
            desc.onlineChans.clear();
            desc.lowLimitChange.clear();
            desc.highLimitChange.clear();
            desc.lowLimits.clear();
            desc.highLimits.clear();

            Map<String, String> alarms = s.getAlarms();
            for (String name : alarms.keySet()) {
                JLabel label = new JLabel(alarms.get(name));
                label.setBorder(LineBorder.createBlackLineBorder());
                label.setBackground(colPopup);
                label.setFont(myFont);
                desc.alarmMap.put(name, label);
            }

            TrendingTableModel model = (TrendingTableModel)table.getModel();
            for (int j = 0; j < desc.size; j++) {
                model.removeRow(desc.start);
            }

            desc.end = desc.start;
            for (MonitorChan chan : s.getChannels()) {
                model.addTrendingRow(desc, chan);
            }
            desc.size = desc.end - desc.start;

            int row = 0;
            for (SubsysDesc d : subsysMap.values()) {
                d.start = row;
                d.end = (row += d.size);
            }

            for (int c = 0; c < table.getColumnCount(); c++) {
                TableColumnModel colModel = table.getColumnModel();
                TableColumn col = colModel.getColumn(c);
                TableCellRenderer rndr;
                Component comp;

                rndr = table.getTableHeader().getDefaultRenderer();
                comp = rndr.getTableCellRendererComponent(table,
                                                          col.getHeaderValue(),
                                                          false, false, 0, 0);
                int width = comp.getPreferredSize().width;

                rndr = table.getCellRenderer(0, c);
                Class colClass = table.getColumnClass(c);
                if (colClass.equals(String.class)
                    || colClass.equals(AlarmMarker.class)) {
                    for (int r = desc.start; r < desc.end; r++) {
                        Object value = table.getValueAt(r, c);
                        comp = rndr.getTableCellRendererComponent(table, value,
                                                                  false, false,
                                                                  r, c);
                        width = Math.max(width, comp.getPreferredSize().width);
                    }
                }
                else {
                    comp = rndr.getTableCellRendererComponent(table, -999.99,
                                                              false, false, 0, c);
                    width = Math.max(width, comp.getPreferredSize().width);
                }
                col.setPreferredWidth(width + 4);
                col.setMinWidth(width + 4);
            }

            model.updateState(desc, s.getMonitorState());
            Container anc = getTopLevelAncestor();
            if (anc instanceof Window) {
                Dimension td = table.getPreferredSize(), wd = anc.getSize();
                wd.width = Math.max(wd.width, td.width);
                anc.setSize(wd);
            }
        }
    }

    class DisableSystem implements Runnable {
        
        SubsysDesc desc;

        DisableSystem(SubsysDesc desc) {
            this.desc = desc;
        }

        @Override
        public void run() {
            desc.enabled = false;
            for (int row = desc.start; row < desc.end; row++) {
                table.setValueAt(table.getValueAt(row, VALUE_IND),
                                 row, VALUE_IND);
            }
        }
    }

    class LimitsCellRenderer extends DefaultTableCellRenderer {

        @Override
        public Component getTableCellRendererComponent(JTable table, Object value,
                                                       boolean isSelected,
                                                       boolean hasFocus,
                                                       int row, int column) {
            Component c;
            if (value == null) {
                c = super.getTableCellRendererComponent(table, "", false, false,
                                                        row, column);
                return c;
            }

            c = super.getTableCellRendererComponent(table, fmt((Double)value),
                                                    false, hasFocus, row, column);
            if (column == LOW_LIMIT_IND && hasLowLimitChanged(row)
                || column == HIGH_LIMIT_IND && hasHighLimitChanged(row)) {
                c.setFont(changeFont);
                c.setForeground(Color.blue);
            }
            else {
                c.setForeground(Color.black);
            }
            ((JLabel)c).setHorizontalAlignment(SwingConstants.RIGHT);
            if (hasFocus) {
                lastFocusRow = row;
            }

            return c;
        }

        private static final long serialVersionUID = 1L;
    }

    class TrendingTableCellRenderer extends DefaultTableCellRenderer {

        @Override
        public Component getTableCellRendererComponent(JTable table, Object value,
                                                       boolean isSelected,
                                                       boolean hasFocus,
                                                       int row, int column) {
            Component c;
            if (value == null) {
                c = super.getTableCellRendererComponent(table, "", false, false,
                                                        row, column);
                c.setBackground(Color.WHITE);
            }
            else {
                c = super.getTableCellRendererComponent(table, fmt((Double)value),
                                                        false, false, row, column);
                c.setBackground(getChannelColor(row));
                ((JLabel)c).setHorizontalAlignment(SwingConstants.RIGHT);
            }

            return c;
        }

        private static final long serialVersionUID = 1L;
}

    class AlarmCellRenderer extends DefaultTableCellRenderer {

        @Override
        public Component getTableCellRendererComponent(JTable table, Object value,
                                                       boolean isSelected,
                                                       boolean hasFocus,
                                                       int row, int column) {
            String text = (String)value;
            text = text == null || text.length() == 0 ? "" : "  \u2713";
            return super.getTableCellRendererComponent(table, text, false,
                                                       false, row, column);
        }

        private static final long serialVersionUID = 1L;
    }

    class TextCellRenderer extends DefaultTableCellRenderer {

        @Override
        public Component getTableCellRendererComponent(JTable table, Object value,
                                                       boolean isSelected,
                                                       boolean hasFocus,
                                                       int row, int column) {
            Component c = super.getTableCellRendererComponent(table,
                                                              "  " + (String)value,
                                                              false, false,
                                                              row, column);
            if (column == DESCRIPTION_IND && isNewSection(row)) {
                c.setFont(changeFont);
            }

            return c;
        }

        private static final long serialVersionUID = 1L;
    }

    private static final long serialVersionUID = 1L;
}
